//
//  NMAAuthAccountManager.h
//
//  NMAAuthLib0
//
//  Created by iam on 6/26/15.
//  Copyright (c) 2015 NetIQ. All rights reserved.
//
//  inspired by nxtbgthng Tobias Kräntzer on 12.07.11.
//
//  Copyright 2011 nxtbgthng. All rights reserved.
//
//  Licenced under the new BSD-licence.
//  See README.md in this repository for
//  the full licence.
//

#import <Foundation/Foundation.h>

#import "NMAAuthTrustDelegate.h"

@class NMAAuthAccount;
@class NMAAuthConnection;

#pragma mark Notifications

extern NSString * const NMAAuthAccountManagerDidFailToRequestAccessNotification;
extern NSString * const NMAAuthAccountManagerAccountsDidChangeNotification;

extern NSString * const NMAAuthAccountManagerNewAccountUserInfoKey;

#pragma mark Configuration

extern NSString * const kNMAAuthAccountManagerConfigurationClientID;
extern NSString * const kNMAAuthAccountManagerConfigurationSecret;
extern NSString * const kNMAAuthAccountManagerConfigurationAuthorizeURL;
extern NSString * const kNMAAuthAccountManagerConfigurationTokenURL;
extern NSString * const kNMAAuthAccountManagerConfigurationRedirectURL;
extern NSString * const kNMAAuthAccountManagerConfigurationScope;
extern NSString * const kNMAAuthAccountManagerConfigurationTokenType;
extern NSString * const kNMAAuthAccountManagerConfigurationTokenRequestHTTPMethod;
extern NSString * const kNMAAuthAccountManagerConfigurationKeyChainGroup;
extern NSString * const kNMAAuthAccountManagerConfigurationKeyChainAccessGroup;

/*
 * Requires a NSDictionary as a value.
 * They are passed onto the authentication request as additional query parameters.
 * The dictionary may not contain the keys "grant_type", "client_id", "client_secret",
 * "username", "password", "redirect_uri", "code", "assertion_type" and "assertion" are not allowed.
 */
extern NSString * const kNMAAuthAccountManagerConfigurationAdditionalAuthenticationParameters;

/*
 * Requires a NSDictionary as a value.
 * They are passed onto the HTTP Header Fields request as additional parameters.
 * Example of a valid setup: @{ @"Content-type" : @"application/x-www-form-urlencoded" }
 */
extern NSString * const kNMAAuthAccountManagerConfigurationCustomHeaderFields;


#pragma mark Account Type

extern NSString * const kNMAAuthAccountManagerAccountName;


#pragma mark Handler

typedef NMAAuthTrustMode(^NMAAuthTrustModeHandler)(NMAAuthConnection *connection, NSString *hostname);
typedef NSArray *(^NMAAuthTrustedCertificatesHandler)(NSString *hostname);
typedef void(^NMAAuthPreparedAuthorizationURLHandler)(NSURL *preparedURL);


@interface NMAAuthAccountManager : NSObject {
@private
    NSMutableDictionary *pendingOAuthClients;
    NSMutableDictionary *accountsDict;
    NSMutableDictionary *configurations;
    NSMutableDictionary *trustModeHandler;
    NSMutableDictionary *trustedCertificatesHandler;    
}

#pragma mark -

+ (instancetype)sharedInstance;

#pragma mark Accessors

@property(nonatomic, strong) NSString *keychainAccessGroup;
@property(nonatomic, strong) NSString *keychainServiceName;
@property(nonatomic, strong, readonly) NSArray *accounts;
- (NSArray *)accountsWithAccountName:(NSString *)AccountName;
- (NMAAuthAccount *)accountWithIdentifier:(NSString *)identifier;


#pragma mark Configuration

- (void)setClientID:(NSString *)aClientID
             secret:(NSString *)aSecret
   authorizationURL:(NSURL *)anAuthorizationURL
           tokenURL:(NSURL *)aTokenURL
        redirectURL:(NSURL *)aRedirectURL
     forAccountName:(NSString *)anAccountName;

- (void)setClientID:(NSString *)aClientID
             secret:(NSString *)aSecret
              scope:(NSSet *)scope
   authorizationURL:(NSURL *)anAuthorizationURL
           tokenURL:(NSURL *)aTokenURL
        redirectURL:(NSURL *)aRedirectURL
     forAccountName:(NSString *)anAccountName;

- (void)setClientID:(NSString *)aClientID
             secret:(NSString *)aSecret
              scope:(NSSet *)theScope
   authorizationURL:(NSURL *)anAuthorizationURL
           tokenURL:(NSURL *)aTokenURL
        redirectURL:(NSURL *)aRedirectURL
      keyChainGroup:(NSString *)aKeyChainGroup
     forAccountName:(NSString *)anAccountName;

- (void)setClientID:(NSString *)aClientID
             secret:(NSString *)aSecret
              scope:(NSSet *)theScope
   authorizationURL:(NSURL *)anAuthorizationURL
           tokenURL:(NSURL *)aTokenURL
        redirectURL:(NSURL *)aRedirectURL
      keyChainGroup:(NSString *)aKeyChainGroup
          tokenType:(NSString *)aTokenType
     forAccountName:(NSString *)anAccountName;

- (void)setConfiguration:(NSDictionary *)configuration forAccountName:(NSString *)accountName;

- (NSDictionary *)configurationForAccountName:(NSString *)accountName;


#pragma Trust Mode Handler

- (void)setTrustModeHandlerForAccountName:(NSString *)AccountName block:(NMAAuthTrustModeHandler)handler;
- (NMAAuthTrustModeHandler)trustModeHandlerForAccountName:(NSString *)accountName;

- (void)setTrustedCertificatesHandlerForAccountName:(NSString *)accountName block:(NMAAuthTrustedCertificatesHandler)handler;
- (NMAAuthTrustedCertificatesHandler)trustedCertificatesHandlerForAccountName:(NSString *)accountName;


#pragma mark Manage Accounts

- (void)signInToAccountWithName:(NSString *)accountName;
- (void)signInToAccountWithName:(NSString *)accountName withPreparedAuthorizationURLHandler:(NMAAuthPreparedAuthorizationURLHandler)aPreparedAuthorizationURLHandler;
- (void)signInToAccountWithName:(NSString *)accountName username:(NSString *)username password:(NSString *)password;
- (void)signInToAccountWithName:(NSString *)accountName assertionType:(NSURL *)assertionType assertion:(NSString *)assertion;
- (void)requestClientCredentialsAccessWithName:(NSString *)accountName;
- (void)addAccount:(NMAAuthAccount *)account;
- (void)removeAccount:(NMAAuthAccount *)account;


#pragma mark Handle OAuth2 Redirects

- (BOOL)handleRedirectURL:(NSURL *)URL;
- (BOOL)handleRedirectURL:(NSURL *)aURL error: (NSError**) error;

@end
