// common.js contains some common variables and functions used by the other JavaScript examples
// Written and tested using Node.js v6.10.2 LTS and Access Manager 4.4


//////////////////////////////////////////////////////////
//  Please make modifications to the following section  //
//////////////////////////////////////////////////////////


// ACTION REQUIRED: Change adminUser, adminPass, host, and port to match your Access Manager admin console
var adminUser = 'cn=admin,o=novell';
var adminPass = 'password';
var host = 'example.com';
var port = 8443;

// ACTION REQUIRED: If your Access Manager system has more than one IDP cluster, sepecify EITHER the 
//   idpClusterDisplayName OR the idpClusterId variable below. If both are left blank, the first cluster
//   returned by the GET idpclusters API will be used. See getIdpClusters.js for an example of retrieving
//   a cluster ID programatically.
var idpClusterDisplayName = '';
var idpClusterId = '';

// ACTION OPTIONAL: The following line allows https connections to servers using self-signed certificates. 
//   Remove it (or comment it out) if your Access Manager admin console uses a certificate signed by a 
//   well-known and trusted certificate authority or if you use a different way of letting Node.js trust 
//   the admin console's SSL certificate.
process.env.NODE_TLS_REJECT_UNAUTHORIZED = '0';


////////////////////////////////////////////////////////////
//  Code below this comment doesn't require modification  //
////////////////////////////////////////////////////////////


// Prepares the https object for calling https.request()
var https = require('https');

// Calls the async https.request() and returns a promise to the result, which can be quite useful when 
// dealing with APIs and you need the result before continuing. Suitable for requests which return 
// utf8-encoded bodies. The object returned on resolve includes keys for statusCode, statusMessage, and
// headers. If the response body is not empty, there will be either a 'json' key or a 'body' key
// depending on whether or not the body is JSON parseable.
function httpsRequest(options, postData) {
    return new Promise(function(resolve, reject) {
        var req = https.request(options, function(response) {
            response.setEncoding('utf8');
            var body = '';
            response.on('data', function(chunk) {
                body += chunk;
            });
            response.on('end', function() {
                var promiseResponse = {
                    statusCode: response.statusCode,
                    statusMessage: response.statusMessage,
                    headers: response.headers
                };
                if (body.length > 0) {
                    try {
                        promiseResponse.json = JSON.parse(body);
                    } catch (err) {
                        promiseResponse.body = body;
                    }
                }
                resolve(promiseResponse);
            });
        });
        req.on('error', function(err) {
            reject(err);
        });
        if (postData) {
            req.write(postData);
        }
        req.end();
    });
}

function prettyJson(obj) {
	if (typeof obj === 'string') {
		try {
			obj = JSON.parse(obj);
		} catch(err) {}
	}
	return JSON.stringify(obj, null, 4);
}

// Prepares the basic authorization header used when calling AMService APIs
var authHeader = 'Basic ' + Buffer.from(adminUser + ':' + adminPass).toString('base64');

// The exportData object contains data we want to make available to other example scripts. It will be made
// available from this module via a promise to give us a chance to asynchronously call the GET idpclusters
// API if necessary and include it in the exportData.
var exportData = {
    host: host,
    port: port,
    authHeader: authHeader,
    httpsRequest: httpsRequest,
    prettyJson: prettyJson
};

module.exports = new Promise(function(resolve, reject) {

    // Finds the clusterId needed by most of the examples and adds it to the exportData
    if (idpClusterId) {

        // If the user provided a value for idpClusterId directly, then we don't have to call the GET 
        // idpclusters API and we can immediately resolve this module's promise.
        exportData.clusterId = idpClusterId;
        resolve(exportData);

    } else {

        // Sets up options for the API call
        var options = {
            host: host,
            port: port,
            path: '/amsvc/v1/idpclusters',
            method: 'GET',
            headers: {
                'Authorization': authHeader,
                'Accept': 'application/json'
            }
        };

        // Calls the GET idpclusters API to read the correct clusterId, then resolves this module's promise
        httpsRequest(options).then(function(response) {
            if (response.statusCode >= 200 && response.statusCode < 300 && response.json) {
                if (idpClusterDisplayName) {
                    for (var i = 0; i < response.json.idpClusterList.idpCluster.length; i++) {
                        if (idpClusterDisplayName === response.json.idpClusterList.idpCluster[i].displayName) {
                            idpClusterId = response.json.idpClusterList.idpCluster[i].instanceID;
                            break;
                        }
                    }
                } else {
                    idpClusterId = response.json.idpClusterList.idpCluster[0].instanceID;
                }
                exportData.clusterId = idpClusterId;
                resolve(exportData);
            } else {
                // If the API call returns a non-success status code, or a body that can't be parsed as JSON,
                // reject this module's promise and return the error response.
                delete response.headers;
                reject(JSON.stringify(response));
            }

            // If an error while making the API call, reject this module's promise and return the error
        }).catch(function(err) {
            reject(err);
        });
    }
});