/*
 * Copyright (c) 2017 Microfocus Corporation. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.microfocus.android.sampleapp.shyanimals;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.support.v4.graphics.drawable.RoundedBitmapDrawable;
import android.support.v4.graphics.drawable.RoundedBitmapDrawableFactory;
import android.util.Pair;

import java.util.Random;

/**
 * Provides pre-loaded and cached animal bitmaps to the sample application
 * Shy Animals.
 * <p>
 * This class is not essential to understanding how an implementor might
 * integrate with the Microfocus Android Mobile SDK.
 */
public class AnimalBitmapProvider {

    private AnimalBitmapProvider(){}

    private static Random random = new Random();

    private static int[] animalResourceIds = new int[]{
            R.drawable.ic_animal_ape,
            R.drawable.ic_animal_racoon,
            R.drawable.ic_animal_piglet,
            R.drawable.ic_animal_puppy,
            R.drawable.ic_animal_bandicoon,
            R.drawable.ic_animal_panda,
            R.drawable.ic_animal_rabbit,
            R.drawable.ic_animal_kangaroo,
            R.drawable.ic_animal_elephant,
            // For sure public domain
            R.drawable.ic_animal_lion,
            R.drawable.ic_animal_robin,
            R.drawable.ic_animal_eyes,
            R.drawable.ic_animal_cat,
            R.drawable.ic_animal_dogone,
            R.drawable.ic_animal_dogtwo,
            R.drawable.ic_animal_dogthree,
            R.drawable.ic_animal_giraffe,
            R.drawable.ic_animal_horse,
            R.drawable.ic_animal_vulture
    };

    private static RoundedBitmapDrawable[] animalBitmaps;

    private static boolean[] animalsRecentlyUsed;

    public static RoundedBitmapDrawable getRandomAnimal(Context context)
    {
        if (null == animalBitmaps)
        {
            animalsRecentlyUsed = new boolean[animalResourceIds.length];
            animalBitmaps = new RoundedBitmapDrawable[animalResourceIds.length];
            for (int i=0; i<animalResourceIds.length; i++)
            {
                Bitmap animalBitmap = BitmapFactory.decodeResource(context.getResources(), animalResourceIds[i]);
                animalBitmaps[i] = RoundedBitmapDrawableFactory.create(context.getResources(), animalBitmap);
                animalBitmaps[i].setCornerRadius(Math.max(animalBitmap.getWidth(), animalBitmap.getHeight()) / 2.0f);
            }
        }
        return getNonRecentAnimal();
    }

    public static Pair<RoundedBitmapDrawable, RoundedBitmapDrawable> getRandomAnimals(Context context)
    {
        RoundedBitmapDrawable firstAnimal = getRandomAnimal(context);
        RoundedBitmapDrawable secondAnimal = getRandomAnimal(context);
        while (firstAnimal == secondAnimal)
        {
            secondAnimal = getRandomAnimal(context);
        }
        return new Pair(firstAnimal, secondAnimal);
    }

    /**
     * Get an animal bitmap. Consider which animal bitmaps have been recently used
     * so as to not show duplicates in close time proximity.
     *
     * @return The cached bitmap to the animal to be used in display. Should not
     * be a recent duplicate.
     */
    private static RoundedBitmapDrawable getNonRecentAnimal()
    {
        int iNotRecentlyUsedAnimals = getNotRecentCount();
        if (1 == iNotRecentlyUsedAnimals)
        {
            return animalBitmaps[markRecent(0)];
        }
        if (0 == iNotRecentlyUsedAnimals)
        {
            markAllNotRecent();
            iNotRecentlyUsedAnimals = getNotRecentCount();
        }
        return animalBitmaps[markRecent(random.nextInt(iNotRecentlyUsedAnimals - 1))];
    }

    /**
     * Get the number of entries in {@code animalsRecentlyUsed} that have a {@code false}
     * value.
     * <p>
     * These represent animals that have not been recently used in display.
     */
    private static int getNotRecentCount()
    {
        int iCount = 0;
        for (boolean b : animalsRecentlyUsed)
        {
            if (!b)
            {
                iCount++;
            }
        }
        return iCount;
    }

    /**
     * All entries in {@code animalsRecentlyUsed} are now {@code true}.
     * Reset them all to {@code false} to start over with no previously
     * used animals.
     */
    private static void markAllNotRecent()
    {
        for (int i=0; i<animalsRecentlyUsed.length; i++)
        {
            animalsRecentlyUsed[i] = false;
        }
    }

    /**
     * Mark the iNth {@code false} entry in the {@code animalsRecentlyUsed}
     * array as {@code true} and return the index in {@code animalsRecentlyUsed}
     * where the {@code false} was located.
     *
     * @param iNth The zero based index to entries with a {@code false} value.
     * @return The index where the iNth {@code false} entry was found.
     */
    private static int markRecent(int iNth)
    {
        int iNthCount = 0;
        for (int i=0; i<animalsRecentlyUsed.length; i++)
        {
            if (false == animalsRecentlyUsed[i])
            {
                if (iNthCount == iNth)
                {
                    animalsRecentlyUsed[i] = true;
                    return i;
                }
                iNthCount++;
            }
        }
        // Execution will never get to here assuming calling code limits the {@code iNth} value
        // to less than the actual number of {@code false} values in {@code animalsRecentlyUsed}.
        return -1;
    }
}
