/*
 * Copyright (c) 2017 Microfocus Corporation. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.microfocus.android.sampleapp.shyanimals;

import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.support.v7.app.AppCompatActivity;
import android.os.Bundle;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.CheckBox;
import android.widget.RadioButton;
import android.widget.RadioGroup;
import android.widget.TextView;
import android.widget.Toast;

import com.microfocus.android.mobile.sdk.library.Authenticator;
import com.microfocus.android.mobile.sdk.library.AuthenticatorBuilder;
import com.microfocus.android.mobile.sdk.library.AuthenticatorClientCallback;
import com.microfocus.android.mobile.sdk.library.AuthenticatorReport;
import com.microfocus.android.mobile.sdk.library.IdentityProvider;
import com.microfocus.android.mobile.sdk.library.ServiceContextFactory;

import net.openid.appauth.browser.BrowserDescriptor;
import net.openid.appauth.browser.BrowserSelector;

import java.util.List;

/**
 * The activity invoked to initiate an Authentication. Presents the end user
 * with per-authentication options to customize the authentication process
 * and the 'Go' button to start the authentication.
 */
public class AuthenticateActivity extends AppCompatActivity {
    final static String AUTHENTICATE_IDENTITY_PROVIDER_JSON = "identityProviderJson";
    final static String TAG = "AuthenticateActivity";
    private IdentityProvider identityProvider;
    private List<BrowserDescriptor> browserDescriptors;
    private RadioGroup radioGroup = null;

    private boolean bForceIdentityProviderRefresh = false;
    private boolean bForceAccessTokenRefresh = false;
    private boolean bForceRefreshTokenRefresh = false;
    private boolean bDisallowIdentityProviderRefresh = false;
    private boolean bDisallowAccessTokenRefresh = false;
    private boolean bDisallowRefreshTokenRefresh = false;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_athenticate);
        Intent intent = getIntent();
        String json = intent.getStringExtra(AUTHENTICATE_IDENTITY_PROVIDER_JSON);
        try {
            if (null == json) {
                throw new Exception("Identity Provider JSON Missing");
            }
            identityProvider = ServiceContextFactory.getInstance().getIdentityProviderBuilder(json).build();
            ((TextView) findViewById(R.id.authenticate_domain_name)).setText(identityProvider.getHost());
            ((TextView) findViewById(R.id.authenticate_client_id)).setText(identityProvider.getClientId());
            final CheckBox cbFCD = (CheckBox) findViewById(R.id.checkFCD);
            final CheckBox cbFAR = (CheckBox) findViewById(R.id.checkFAR);
            final CheckBox cbFRR = (CheckBox) findViewById(R.id.checkFRR);
            final CheckBox cbDCD = (CheckBox) findViewById(R.id.checkDCD);
            final CheckBox cbDAR = (CheckBox) findViewById(R.id.checkDAR);
            final CheckBox cbDRR = (CheckBox) findViewById(R.id.checkDRR);
            cbFCD.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    bForceIdentityProviderRefresh = cbFCD.isChecked();
                    if (cbDCD.isChecked()) {
                        cbDCD.setChecked(false);
                        bDisallowIdentityProviderRefresh = false;
                    }
                }
            });
            cbFRR.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    bForceRefreshTokenRefresh = cbFRR.isChecked();
                    if (cbDRR.isChecked()) {
                        cbDRR.setChecked(false);
                        bDisallowRefreshTokenRefresh = false;
                    }
                }
            });
            cbFAR.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    bForceAccessTokenRefresh = cbFAR.isChecked();
                    if (cbDAR.isChecked()) {
                        cbDAR.setChecked(false);
                        bDisallowAccessTokenRefresh = false;
                    }
                }
            });
            cbDCD.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    bDisallowIdentityProviderRefresh = cbDCD.isChecked();
                    if (cbFCD.isChecked()) {
                        cbFCD.setChecked(false);
                        bForceIdentityProviderRefresh = false;
                    }
                }
            });
            cbDRR.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    bDisallowRefreshTokenRefresh = cbDRR.isChecked();
                    if (cbFRR.isChecked()) {
                        cbFRR.setChecked(false);
                        bForceRefreshTokenRefresh = false;
                    }
                }
            });
            cbDAR.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View view) {
                    bDisallowAccessTokenRefresh = cbDAR.isChecked();
                    if (cbFAR.isChecked()) {
                        cbFAR.setChecked(false);
                        bForceAccessTokenRefresh = false;
                    }
                }
            });

            browserDescriptors = BrowserSelector.getAllBrowsers(AuthenticateActivity.this);
            PackageManager pm = AuthenticateActivity.this.getPackageManager();
            radioGroup = (RadioGroup) findViewById(R.id.radioGroup);

            for (int i = 0; i < browserDescriptors.size(); i++) {
                String label;
                RadioButton rb = new RadioButton(this);
                try {
                    ApplicationInfo info = pm.getApplicationInfo(browserDescriptors.get(i).packageName, 0);
                    label = pm.getApplicationLabel(info).toString();

                    if (browserDescriptors.get(i).useCustomTab) {
                        label += " Custom Tab";
                    }
                } catch (PackageManager.NameNotFoundException e) {
                    e.printStackTrace();
                    label = browserDescriptors.get(i).packageName;
                }
                rb.setText(label);
                rb.setId(i);
                if (i == 0)
                    rb.setChecked(true);
                radioGroup.addView(rb);
            }
        }
        catch (Exception e)
        {
            ((TextView) findViewById(R.id.authenticate_domain_name)).setText("Identity Provider Not Available");
            ((TextView) findViewById(R.id.authenticate_client_id)).setText("Error loading Identity Provider");
        }
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu)
    {
        MenuItem go = menu.add("Go");
        go.setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS);
        go.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem menuItem) {

                try
                {
                    Settings settings = new Settings(AuthenticateActivity.this);
                    AuthenticatorBuilder builder = ServiceContextFactory.getInstance().getAuthenticatorBuilder(AuthenticateActivity.this, new AuthenticateActivity.AuthenticatorCallbackImpl());
                    builder.setBrowserDescriptor(browserDescriptors.get(radioGroup.getCheckedRadioButtonId()));
                    builder.setAccessTokenRefreshBeforeExpirationBy(settings.getAccessTokenRefreshBeforeExpirationBy());
                    builder.setForceIdentityProviderRefresh(bForceIdentityProviderRefresh);
                    builder.setForceAccessTokenRefresh(bForceAccessTokenRefresh);
                    builder.setForceRefreshTokenRefresh(bForceRefreshTokenRefresh);
                    builder.setDisallowIdentityProviderRefresh(bDisallowIdentityProviderRefresh);
                    builder.setDisallowAccessTokenRefresh(bDisallowAccessTokenRefresh);
                    builder.setDisallowRefreshTokenRefresh(bDisallowRefreshTokenRefresh);
                    Authenticator aa = builder.build();
                    ServiceContextFactory.getInstance().authenticate(aa);
                } catch (Exception e) {
                    Toast.makeText(AuthenticateActivity.this, e.getMessage(), Toast.LENGTH_LONG).show();
                }
                return false;
            }
        });
        return true;
    }

    private class AuthenticatorCallbackImpl
            implements AuthenticatorClientCallback
    {
        public void onAuthenticationComplete (AuthenticatorReport report)
        {
            Intent intent = new Intent(AuthenticateActivity.this, AuthenticateReportActivity.class);
            String json = ServiceContextFactory.getInstance().toJson(report);
            intent.putExtra(AuthenticateReportActivity.REPORT_ACTIVITY_JSON, json);
            startActivity(intent);
        }
    }

}
