/*
 * Copyright (c) 2017 Microfocus Corporation. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.microfocus.android.sampleapp.shyanimals;

import android.os.Bundle;
import android.support.v7.app.AppCompatActivity;
import android.view.View;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.microfocus.android.mobile.sdk.library.AuthenticatorReport;
import com.microfocus.android.mobile.sdk.library.ServiceContextFactory;
import com.microfocus.android.mobile.sdk.library.util.DateTimeUtil;

import net.openid.appauth.AuthorizationException;

import java.text.MessageFormat;
import java.util.Collection;

/**
 * The activity invoked after an Authentication is complete. Presents the end user
 * with a static custom report detailing the results of the Authentication process.
 */
public class AuthenticateReportActivity extends AppCompatActivity {
    final private String TAG = "AuthenticateReportActivity";
    final public static String REPORT_ACTIVITY_JSON = "reportActivityJson";

    private static final String INITIATED_FALSE = "Initiated: false";
    private static final String INITIATED_TRUE = "Initiated: true";
    private static final String INITIATED_BEFORE_EXPIRATION_FALSE = "Initiated Before Expiration: false";
    private static final String INITIATED_BEFORE_EXPIRATION_TRUE = "Initiated Before Expiration: true";
    private static final String EXISTS_NO = "Exists: no";
    private static final String EXISTS_CHARACTERS = "Exists: {0} characters";
    private static final String EXPIRED_AGO = "Expired {0} ago";
    private static final String EXPIRES_IN = "Expires in {0}";

    private static final String AUTHENTICATION_STARTED = "Is Authentication Started: {0}";
    private static final String AUTHENTICATION_STOPPED = "Is Authentication Complete: {0}";
    private static final String DURATION_SINCE_CREATION = "Authenticator Created {0} ago";
    private static final String DURATION_SINCE_STARTED = "Authenticator Started {0} ago";
    private static final String DURATION_AUTHENTICATION = "Authentication Completed in: {0}";


    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_authenticate_report);
        String strReportJson = getIntent().getStringExtra(REPORT_ACTIVITY_JSON);
        if (strReportJson == null) {
            return;
        }
        AuthenticatorReport report = ServiceContextFactory.getInstance().fromJsonAuthenticatorReport(strReportJson);
        LinearLayout layoutParent = (LinearLayout)findViewById(R.id.reportTopLevelLayout);

        TextView labelToRemove = (TextView)findViewById(R.id.successfulAuthenticationLabel);
        if (report.isAuthenticated())
        {
            labelToRemove = (TextView)findViewById(R.id.failedAuthenticationLabel);
        }
        layoutParent.removeView(labelToRemove);

        String strIdentityProviderHost = "Unknown Identity Provider Host";
        String strIdentityProviderClientId = "Unknown Identity Provider Client Id";
        if (null != report.getIdentityProvider())
        {
            strIdentityProviderHost = report.getIdentityProvider().getHost();
            strIdentityProviderClientId = report.getIdentityProvider().getClientId();
        }
        ((TextView)findViewById(R.id.identityProviderHost)).setText(strIdentityProviderHost);
        ((TextView)findViewById(R.id.identityProviderClientId)).setText(strIdentityProviderClientId);

        TextView viewInitiated = (TextView)findViewById(R.id.identityProviderConfigurationInitiated);
        viewInitiated.setText(INITIATED_FALSE);
        if (report.identityProviderConfigurationRefreshInitiated())
        {
            viewInitiated.setText(INITIATED_TRUE);
        }

        viewInitiated = (TextView)findViewById(R.id.accessTokenRefreshInitiated);
        viewInitiated.setText(INITIATED_FALSE);
        if (report.accessTokenRefreshInitiated())
        {
            viewInitiated.setText(INITIATED_TRUE);
        }

        viewInitiated = (TextView)findViewById(R.id.accessTokenRefreshBeforeExpInitiated);
        viewInitiated.setText(INITIATED_BEFORE_EXPIRATION_FALSE);
        if (report.accessTokenRefreshBeforeExpirationInitiated())
        {
            viewInitiated.setText(INITIATED_BEFORE_EXPIRATION_TRUE);
        }

        viewInitiated = (TextView)findViewById(R.id.refreshTokenRefreshInitiated);
        viewInitiated.setText(INITIATED_FALSE);
        if (report.refreshTokenRefreshInitiated())
        {
            viewInitiated.setText(INITIATED_TRUE);
        }

        if (0 == addExceptions(layoutParent,
                              (TextView)findViewById(R.id.identityProviderConfigurationErrors),
                              report.getIdentityProviderConfigurationRefreshExceptions()))
        {
            layoutParent.removeView(findViewById(R.id.identityProviderConfigurationErrors));
        }
        if (0 == addExceptions(layoutParent,
                               (TextView)findViewById(R.id.accessTokenRefreshErrors),
                               report.getAccessTokenRefreshExceptions()))
        {
            layoutParent.removeView(findViewById(R.id.accessTokenRefreshErrors));
        }
        if (0 == addExceptions(layoutParent,
                               (TextView)findViewById(R.id.refreshTokenRefreshErrors),
                               report.getRefreshTokenRefreshExceptions()))
        {
            layoutParent.removeView(findViewById(R.id.refreshTokenRefreshErrors));
        }
        if (0 == addExceptions(layoutParent,
                (TextView)findViewById(R.id.authenticatorErrors),
                report.getAuthenticatorExceptions()))
        {
            layoutParent.removeView(findViewById(R.id.authenticatorErrors));
        }

        TextView viewAccessToken = (TextView)findViewById(R.id.accessTokenExists);
        TextView viewAccessTokenExpiration = (TextView)findViewById(R.id.accessTokenExpiration);
        String strAccessToken = report.getAccessToken();
        if (null == strAccessToken)
        {
            viewAccessToken.setText(EXISTS_NO);
            layoutParent.removeView(viewAccessTokenExpiration);
        }
        else
        {
            viewAccessToken.setText(MessageFormat.format(EXISTS_CHARACTERS, "" + strAccessToken.length()));
            long lExpiresIn = (report.getAccessTokenExpirationTime() - System.currentTimeMillis());
            if (lExpiresIn < 0)
            {
                viewAccessTokenExpiration.setText(MessageFormat.format(EXPIRED_AGO, DateTimeUtil.getDuration(-lExpiresIn)));
            }
            else
            {
                viewAccessTokenExpiration.setText(MessageFormat.format(EXPIRES_IN, DateTimeUtil.getDuration(lExpiresIn)));
            }
        }

        TextView viewAuthenticationStarted = (TextView)findViewById(R.id.isAuthenticationStarted);
        viewAuthenticationStarted.setText(MessageFormat.format(AUTHENTICATION_STARTED, report.isStarted()));

        TextView viewAuthenticationStopped = (TextView)findViewById(R.id.isAuthenticationComplete);
        viewAuthenticationStopped.setText(MessageFormat.format(AUTHENTICATION_STOPPED, report.isStopped()));

        TextView viewDurationSinceCreation = (TextView)findViewById(R.id.durationSinceCreation);
        viewDurationSinceCreation.setText(MessageFormat.format(DURATION_SINCE_CREATION, DateTimeUtil.getDuration(report.getDurationSinceCreation())));

        TextView viewDurationAuthentication = (TextView) findViewById(R.id.durationAuthentication);
        TextView viewDurationSinceStarted = (TextView) findViewById(R.id.durationSinceStarted);
        if (report.isStopped()) {
            viewDurationAuthentication.setText(MessageFormat.format(DURATION_AUTHENTICATION, DateTimeUtil.getDuration(report.getDuration())));
            layoutParent.removeView(viewDurationSinceStarted);
        }
        else if (report.isStarted()) {

            viewDurationSinceStarted.setText(MessageFormat.format(DURATION_SINCE_STARTED, DateTimeUtil.getDuration(report.getDuration())));
            layoutParent.removeView(viewDurationAuthentication);
        }
    }

    private int addExceptions(LinearLayout layoutParent, TextView insertView, Collection<Exception> exceptions) {
        int iCount = 0;
        if ((null != exceptions) && (0 != exceptions.size())) {
            int iAddIndex = layoutParent.indexOfChild(insertView);
            for (Exception current : exceptions) {
                TextView childView = new TextView(this);
                childView.setId(View.generateViewId());
                childView.setLayoutParams(new LinearLayout.LayoutParams(LinearLayout.LayoutParams.WRAP_CONTENT, LinearLayout.LayoutParams.WRAP_CONTENT));
                String exceptionMessage = current.getMessage();
                if (current instanceof AuthorizationException)
                {
                    StringBuilder sb = new StringBuilder(256);
                    AuthorizationException authEx = (AuthorizationException)current;
                    sb.append("(").append(authEx.code).append(")");
                    if ((null != authEx.errorDescription) && (!authEx.errorDescription.isEmpty()))
                    {
                        sb.append(", ").append(authEx.errorDescription);
                    }
                    if ((null != authEx.error) && (!authEx.error.isEmpty()))
                    {
                        sb.append(", ").append(authEx.error);
                    }
                    exceptionMessage = sb.toString();
                }
                childView.setText(" - " + current.getClass().getSimpleName() + ": " + exceptionMessage);
                childView.setPadding(childView.getPaddingLeft() + 15, childView.getPaddingTop(), childView.getPaddingRight(), childView.getPaddingBottom());
                layoutParent.addView(childView, iAddIndex + 1);
                iCount++;
            }
        }
        return iCount;
    }

}
