/*
 * Copyright (c) 2017 Microfocus Corporation. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.microfocus.android.sampleapp.shyanimals;

import android.app.Activity;
import android.content.Intent;
import android.os.AsyncTask;
import android.support.v7.app.AppCompatActivity;
import android.os.Bundle;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.ProgressBar;
import android.widget.TabHost;
import android.widget.Toast;

import com.microfocus.android.mobile.sdk.library.AuthenticatorException;
import com.microfocus.android.mobile.sdk.library.IdentityProvider;
import com.microfocus.android.mobile.sdk.library.IdentityProviderBuilder;
import com.microfocus.android.mobile.sdk.library.ServiceContextFactory;
import com.microfocus.android.mobile.sdk.library.util.JsonUtil;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.Scanner;

/**
 * The activity that allows 'Adding' or 'Editing' an {@link IdentityProvider} object.
 */
public class IdentityProviderEditorActivity extends AppCompatActivity {

    private static final String TAG = "IDPEditorActivity";

    static final int COORELATIONID_IDENTITY_PROVIDER_EDITOR = 15286;

    static final String TAG_IDENTITY_PROVIDER_EDITOR_JSON = "identityProviderJson";

    private boolean bDoingAdd = true;

    private static final String tabManualEntry = "manual";
    private static final String tabFromUrl = "fromUrl";

    private String currentTab = "";

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_identity_provider_editor);

        TabHost host = (TabHost)findViewById(R.id.tabhost);
        host.setup();

        Intent intent = getIntent();
        String json = intent.getStringExtra(TAG_IDENTITY_PROVIDER_EDITOR_JSON);
        bDoingAdd = (null == json);

        //Tab tabManualEntry
        currentTab = tabManualEntry;
        TabHost.TabSpec spec = host.newTabSpec(tabManualEntry);
        spec.setContent(R.id.tabManualEntry);
        spec.setIndicator("Manual entry");
        host.addTab(spec);

        if (bDoingAdd) {   //Tab tabFromUrl
            spec = host.newTabSpec(tabFromUrl);
            spec.setContent(R.id.tabFromUrl);
            spec.setIndicator("From url");
            host.addTab(spec);
            host.setOnTabChangedListener(new TabHost.OnTabChangeListener() {
                @Override
                public void onTabChanged(String s) {
                    currentTab = s;
                }
            });
        }
        else
        {
            LinearLayout tabLayout = (LinearLayout)findViewById(R.id.tabFromUrl);
            tabLayout.removeAllViews();
            setTitle("Update Identity Provider");
        }

        IdentityProvider identityProvider =  null;
        if (!bDoingAdd)
        {
            try {
                identityProvider = ServiceContextFactory.getInstance().getIdentityProviderBuilder(json).build();
            }
            catch (AuthenticatorException ae)
            {   // Switch from "Edit" to "Add" on IDP error
                Log.e(TAG, "Error creating Identity Provider", ae);
                bDoingAdd = true;
            }
        }

        if (!bDoingAdd)
        {
            if (null != identityProvider.getProviderUrl()) {
                ((EditText) findViewById(R.id.editProviderURL)).setText(identityProvider.getProviderUrl());
            }
            if (null != identityProvider.getClientId()) {
                ((EditText) findViewById(R.id.editClientId)).setText(identityProvider.getClientId());
            }
            if (null != identityProvider.getClientSecret()) {
                ((EditText) findViewById(R.id.editClientSecret)).setText(identityProvider.getClientSecret());
            }
            if (null != identityProvider.getScope()) {
                ((EditText) findViewById(R.id.editScope)).setText(identityProvider.getScope());
            }
            if (null != identityProvider.getRedirectUri()) {
                ((EditText) findViewById(R.id.editRedirectUri)).setText(identityProvider.getRedirectUri().toString());
            }
            if (null != identityProvider.getDiscoveryEndPoint()) {
                ((EditText) findViewById(R.id.editDiscoveryURL)).setText(identityProvider.getDiscoveryEndPoint().toString());
            }
            if (null != identityProvider.getLogoutEndPoint()) {
                ((EditText) findViewById(R.id.editLogoutURL)).setText(identityProvider.getLogoutEndPoint().toString());
            }
            if (null != identityProvider.getTokenRevocationEndPoint()) {
                ((EditText) findViewById(R.id.editTokenRevocationURL)).setText(identityProvider.getTokenRevocationEndPoint().toString());
            }
            if (null != identityProvider.getTokenRevocationMethod()) {
                ((EditText) findViewById(R.id.editTokenRevocationMethod)).setText(identityProvider.getTokenRevocationMethod());
            }
            if (null != identityProvider.getTokenRevocationParameterFormatString()) {
                ((EditText) findViewById(R.id.editTokenRevocationParameterFormat)).setText(identityProvider.getTokenRevocationParameterFormatString());
            }
        }
    }


    @Override
    public boolean onCreateOptionsMenu(Menu menu)
    {
        final MenuItem add = menu.add((bDoingAdd ? "Add" : "Update"));
        add.setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS);
        add.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem menuItem) {
                new ProcessData(add).execute();
                return false;
            }
        });
        return true;
    }

    private class ProcessData extends AsyncTask<Void, Void, String> {
        MenuItem add = null;
        ProgressBar progressBar = (ProgressBar)findViewById(R.id.tabProgress);
        TabHost tabHost = (TabHost)findViewById(R.id.tabhost);
        String providerUrl = "",
                clientId = "",
                clientSecret = "",
                scope = "",
                redirectUri = "",
                discoveryUrl = "",
                logoutUrl = "",
                revokeUrl = "",
                revokeMethod = "",
                revokeParameters = "",
                serverUrl = "";

        ProcessData (MenuItem add) {
            this.add = add;
        }

        @Override
        protected void onPreExecute() {
            progressBar.setVisibility(View.VISIBLE);
            add.setEnabled(false);
            tabHost.setVisibility(View.GONE);
            switch (currentTab) {
                case tabManualEntry:
                    providerUrl = ((EditText) findViewById(R.id.editProviderURL)).getText().toString();
                    clientId = ((EditText) findViewById(R.id.editClientId)).getText().toString();
                    clientSecret = ((EditText) findViewById(R.id.editClientSecret)).getText().toString();
                    scope = ((EditText) findViewById(R.id.editScope)).getText().toString();
                    redirectUri = ((EditText) findViewById(R.id.editRedirectUri)).getText().toString();
                    discoveryUrl = ((EditText) findViewById(R.id.editDiscoveryURL)).getText().toString();
                    logoutUrl = ((EditText) findViewById(R.id.editLogoutURL)).getText().toString();
                    revokeUrl = ((EditText) findViewById(R.id.editTokenRevocationURL)).getText().toString();
                    revokeMethod = ((EditText) findViewById(R.id.editTokenRevocationMethod)).getText().toString();
                    revokeParameters = ((EditText) findViewById(R.id.editTokenRevocationParameterFormat)).getText().toString();
                    break;
                case tabFromUrl:
                    serverUrl = ((EditText)findViewById(R.id.editURL)).getText().toString();
            }
        }

        @Override
        protected String doInBackground(Void... voids) {
            if (currentTab.equals(tabFromUrl)) {
                Scanner scanner = null;
                String file = "";
                try {
                    URL url = new URL(serverUrl);
                    scanner = new Scanner(new BufferedReader(new InputStreamReader(url.openStream())));
                    while (scanner.hasNext()){
                        file = file.concat(scanner.nextLine());
                    }
                } catch (Exception e) {
                    Log.d(TAG_IDENTITY_PROVIDER_EDITOR_JSON, e.getLocalizedMessage());
                    return "Error: " + e.getClass().getSimpleName() + ": " + e.getLocalizedMessage();
                } finally {
                    if (scanner != null)
                        scanner.close();
                }
                JSONObject jsonObject;
                try {
                    jsonObject = new JSONObject(file);
                    providerUrl = JsonUtil.getStringIfDefined(jsonObject, "ProviderUrl");
                    clientId = JsonUtil.getStringIfDefined(jsonObject, "ClientId");
                    clientSecret = JsonUtil.getStringIfDefined(jsonObject, "ClientSecret");
                    redirectUri = JsonUtil.getStringIfDefined(jsonObject, "RedirectUri");
                    discoveryUrl = JsonUtil.getStringIfDefined(jsonObject, "DiscoveryUrl");
                    logoutUrl = JsonUtil.getStringIfDefined(jsonObject, "LogoutUrl");
                    revokeUrl = JsonUtil.getStringIfDefined(jsonObject, "RevokeUrl");
                    revokeMethod = JsonUtil.getStringIfDefined(jsonObject, "RevokeMethod");
                    revokeParameters = JsonUtil.getStringIfDefined(jsonObject, "RevokeParameters");
                    scope = JsonUtil.getStringIfDefined(jsonObject, "Scope");
                } catch (JSONException e) {
                    return "Error: " + e.getLocalizedMessage();
                }
            }

            String jsonIdentityProvider = "";
            if (((null != providerUrl) && (!providerUrl.isEmpty())) &&
                ((null != clientId) && (!clientId.isEmpty())) &&
                ((null != clientSecret) && (!clientSecret.isEmpty())) &&
                ((null != redirectUri) && (!redirectUri.isEmpty())))
            {
                try {
                    IdentityProviderBuilder builder = ServiceContextFactory.getInstance().getIdentityProviderBuilder(providerUrl, clientId, clientSecret, redirectUri);
                    builder.setDiscoveryEndPoint(discoveryUrl);
                    builder.setLogoutEndPoint(logoutUrl).setScope(scope);
                    builder.setTokenRevocationEndPoint(revokeUrl).setTokenRevocationMethod(revokeMethod).setTokenRevocationParameterFormatString(revokeParameters);
                    jsonIdentityProvider = builder.build().toJson();
                } catch (Exception e) {
                    return "Error: " + e.getLocalizedMessage();
                }
            } else {
                return "Error: Fill out required fields";
            }
            return jsonIdentityProvider;
        }

        @Override
        protected void onPostExecute(String jsonIdentityProvider) {
            Intent intent = new Intent();
            if (!jsonIdentityProvider.startsWith("Error:") && !jsonIdentityProvider.isEmpty()) {
                intent.putExtra(TAG_IDENTITY_PROVIDER_EDITOR_JSON, jsonIdentityProvider);
                setResult(Activity.RESULT_OK, intent);
                finish();
            } else {
                add.setEnabled(true);
                progressBar.setVisibility(View.GONE);
                tabHost.setVisibility(View.VISIBLE);
                Toast.makeText(IdentityProviderEditorActivity.this, jsonIdentityProvider, Toast.LENGTH_LONG).show();
            }
        }
    }
}
