/*
 * Copyright (c) 2017 Microfocus Corporation. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.microfocus.android.sampleapp.shyanimals;

import android.content.Context;

import com.microfocus.android.mobile.sdk.library.ServiceContext;
import com.microfocus.android.mobile.sdk.library.ServiceContextFactory;
import com.microfocus.android.mobile.sdk.library.util.DateTimeUtil;
import com.microfocus.android.mobile.sdk.library.IdentityProvider;

/**
 * Extracts elements of an {@link IdentityProvider} instance and prepares a human
 * readable status that will be displayed to the end user in the {@link IdentityProviderStatusActivity}.
 * status
 * <p>
 * Demonstrated how to obtain and interpret {@link IdentityProvider} elements.
  */
public class IdentityProviderStatus
{
    private static final float fGoodStateOpacity = 1.0f;
    private static final float fGoodStateMinusOpacity = 0.8f;
    private static final float fBadStateOpacity = 0.1f;

    private float fRefreshTokenAlpha = fBadStateOpacity;
    private float fAccessTokenAlpha = fBadStateOpacity;

    private String refreshTokenStatus;
    private String accessTokenStatus;

    public IdentityProviderStatus(Context context, IdentityProvider identityProvider,
                                  long lAccessTokenRefreshBeforeExpirationBy) {
        StringBuilder sbCompositeStatus = new StringBuilder();
        try {
            refreshTokenStatus = "Refresh token does not exist. A full authentication is required.";
            if (null != ServiceContextFactory.getInstance().getRefreshToken(context))
            {
                fRefreshTokenAlpha = fGoodStateOpacity;
                refreshTokenStatus = "A refresh token exists.";
            }
        } catch (Exception exception) {   // Set opacity to hardly visible to represent error state
            fRefreshTokenAlpha = fBadStateOpacity;
            refreshTokenStatus = "Unable to determine the refresh token health: " + exception.getClass().getSimpleName();
        }
        try {
            long lTimeToAccessTokenExpiration = ServiceContextFactory.getInstance().getTimeToAccessTokenExpiration(context);
            if (lTimeToAccessTokenExpiration == ServiceContext.ACCESS_TOKEN_EXPIRATION_NO_AUTHENTICATION) {   // Authentication information does not exist
                accessTokenStatus = "Authentication information does not exist. A full authentication is required.";
            } else if (lTimeToAccessTokenExpiration == ServiceContext.ACCESS_TOKEN_EXPIRATION_NOT_FOUND) {   // Access Token does not exist
                accessTokenStatus = "Access Token does not exist. A full authentication is required.";
            } else if (lTimeToAccessTokenExpiration < 0) {
                accessTokenStatus = "Access Token expired " + DateTimeUtil.getDuration(-lTimeToAccessTokenExpiration) + " ago. Access Token must be refreshed.";
            } else if (lTimeToAccessTokenExpiration < lAccessTokenRefreshBeforeExpirationBy) {
                fAccessTokenAlpha = fGoodStateMinusOpacity;
                accessTokenStatus = "Access Token will need to be refreshed within " + DateTimeUtil.getDuration(lTimeToAccessTokenExpiration) + ".";
            }
            else
            {
                fAccessTokenAlpha = fGoodStateOpacity;
                accessTokenStatus = "Access Token is valid for another " + DateTimeUtil.getDuration(lTimeToAccessTokenExpiration) + ".";
            }

        } catch (Exception exception) {   // Set opacity to hardly visible to represent error state
            fAccessTokenAlpha = fBadStateOpacity;
            accessTokenStatus = "Unable to determine the access token health: " + exception.getClass().getSimpleName();
        }
    }

    public boolean isAllGood()
    {
        return ((fGoodStateOpacity == getRefreshTokenAlpha()) && (fGoodStateOpacity == getAccessTokenAlpha()));
    }

    public float getRefreshTokenAlpha() {
        return fRefreshTokenAlpha;
    }
    public float getAccessTokenAlpha() {
        return fAccessTokenAlpha;
    }

    public String getRefreshTokenStatus() {
        return refreshTokenStatus;
    }
    public String getAccessTokenStatus() {
        return accessTokenStatus;
    }

}
