/*
 * Copyright (c) 2017 Microfocus Corporation. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.microfocus.android.sampleapp.shyanimals;

import android.annotation.SuppressLint;

import android.app.AlertDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.AsyncTask;
import android.os.Bundle;
import android.support.v4.graphics.drawable.RoundedBitmapDrawable;
import android.support.v7.app.AppCompatActivity;
import android.support.v7.widget.Toolbar;
import android.util.Log;
import android.util.Pair;
import android.view.View;
import android.view.Menu;
import android.view.MenuItem;
import android.widget.ImageView;
import android.widget.TextView;
import android.widget.Toast;

import com.microfocus.android.mobile.sdk.library.IdentityProvider;
import com.microfocus.android.mobile.sdk.library.InvokeClientCallback;
import com.microfocus.android.mobile.sdk.library.InvokeTask;
import com.microfocus.android.mobile.sdk.library.InvokeTaskHttpGet;
import com.microfocus.android.mobile.sdk.library.ServiceContext;
import com.microfocus.android.mobile.sdk.library.ServiceContextFactory;
import com.microfocus.android.mobile.sdk.library.util.DateTimeUtil;

import net.openid.appauth.browser.BrowserDescriptor;
import net.openid.appauth.browser.BrowserSelector;

import java.io.ByteArrayOutputStream;
import java.security.SecureRandom;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.List;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

/**
 * The main activity of the sample application Shy Animals. Contains the list
 * of installed Identity Providers as well as settings menu and a 'Create Identity
 * Provider' button.
 */
public class MainActivity extends AppCompatActivity {
    private static final String TAG = "MainActivity";

    List<String> browserList = new ArrayList<>();
    List<BrowserDescriptor> browserDescriptors = new ArrayList<>();
    private int currentBrowser;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        setContentView(R.layout.activity_main);
        trustAllCertificates();

        Toolbar toolbar = (Toolbar) findViewById(R.id.toolbar);
        setSupportActionBar(toolbar);

        new BrowserFetcher().execute();
    }

    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        // Check which request we're responding to
        if (requestCode == IdentityProviderEditorActivity.COORELATIONID_IDENTITY_PROVIDER_EDITOR) {
            // Make sure the request was successful
            if (resultCode == RESULT_OK) {
                try {
                    String json = data.getStringExtra(IdentityProviderEditorActivity.TAG_IDENTITY_PROVIDER_EDITOR_JSON);
                    if (null != json) {
                        IdentityProvider identityProvider = ServiceContextFactory.getInstance().getIdentityProviderBuilder(json).build();
                        if (null != identityProvider) {
                            ServiceContextFactory.getInstance().putIdentityProvider(this, identityProvider);
                            MainActivity.this.invalidateOptionsMenu();
                            refreshUI();
                        }
                    } else {
                        throw new Exception("Identity Provicer JSON Missing");
                    }
                } catch (Exception e) {
                    Log.e(TAG, "Error creating Identity Provider", e);
                }
            }
        } else if (requestCode == SettingsActivity.COORELATIONID_SETTINGS) {
            if (resultCode == RESULT_OK) {
                long lAccessToken = data.getLongExtra(SettingsActivity.TAG_ACCESS_TOKEN_REFRESH_BEFORE_EXPIRATION_BY, ServiceContext.ACCESS_TOKEN_REFRESH_BEFORE_EXPIRATION_BY_DEFAULT);
                Settings settings = new Settings(MainActivity.this);
                settings.setAccessTokenRefreshBeforeExpirationBy(lAccessToken);
                settings.apply();
            }
        }
    }

    @Override
    protected void onStart() {
        super.onStart();
        ServiceContextFactory.getInstance().onStart(this);
    }

    @Override
    protected void onResume() {
        super.onResume();
        ServiceContextFactory.getInstance().onResume();
        refreshUI();
        HintsTask.doHintIfTime(this);
    }

    @Override
    protected void onPause() {
        super.onPause();
        ServiceContextFactory.getInstance().onPause();
    }

    @Override
    protected void onStop() {
        super.onStop();
        ServiceContextFactory.getInstance().onStop();
    }

    private void refreshUI() {
        IdentityProvider identityProvider = ServiceContextFactory.getInstance().getIdentityProvider();
        TextView domainNameView = (TextView) findViewById(R.id.idpstatus_domainname);
        TextView clientIdView = (TextView) findViewById(R.id.idpstatus_clientid);
        if (null == identityProvider) {
            domainNameView.setText("No Identity Provider Defined");
            clientIdView.setText("Please use the application drop down menu to define an Identity Provider");
            findViewById(R.id.idpstatus_refreshtoken_label).setVisibility(View.GONE);
            findViewById(R.id.idpstatus_accesstoken_label).setVisibility(View.GONE);
            findViewById(R.id.idpstatus_refreshtoken_image).setVisibility(View.GONE);
            findViewById(R.id.idpstatus_refreshtoken_status).setVisibility(View.GONE);
            findViewById(R.id.idpstatus_accesstoken_image).setVisibility(View.GONE);
            findViewById(R.id.idpstatus_accesstoken_status).setVisibility(View.GONE);
        } else {
            findViewById(R.id.idpstatus_refreshtoken_label).setVisibility(View.VISIBLE);
            findViewById(R.id.idpstatus_accesstoken_label).setVisibility(View.VISIBLE);
            findViewById(R.id.idpstatus_refreshtoken_image).setVisibility(View.VISIBLE);
            findViewById(R.id.idpstatus_refreshtoken_status).setVisibility(View.VISIBLE);
            findViewById(R.id.idpstatus_accesstoken_image).setVisibility(View.VISIBLE);
            findViewById(R.id.idpstatus_accesstoken_status).setVisibility(View.VISIBLE);

            domainNameView.setText(identityProvider.getHost());
            clientIdView.setText(identityProvider.getClientId());

            Settings settings = new Settings(this);

            IdentityProviderStatus status = new IdentityProviderStatus(this, identityProvider,
                    settings.getAccessTokenRefreshBeforeExpirationBy());

            ImageView refreshTokenImage = (ImageView) findViewById(R.id.idpstatus_refreshtoken_image);
            ImageView accessTokenImage = (ImageView) findViewById(R.id.idpstatus_accesstoken_image);

            Pair<RoundedBitmapDrawable, RoundedBitmapDrawable> animalBitmaps = AnimalBitmapProvider.getRandomAnimals(this);
            refreshTokenImage.setImageDrawable(animalBitmaps.first);
            accessTokenImage.setImageDrawable(animalBitmaps.second);

            refreshTokenImage.setAlpha(status.getRefreshTokenAlpha());
            accessTokenImage.setAlpha(status.getAccessTokenAlpha());

            TextView refreshTokenStatus = (TextView) findViewById(R.id.idpstatus_refreshtoken_status);
            refreshTokenStatus.setText(((null != status.getRefreshTokenStatus()) ? status.getRefreshTokenStatus() : ""));

            TextView accessTokenStatus = (TextView) findViewById(R.id.idpstatus_accesstoken_status);
            accessTokenStatus.setText(((null != status.getAccessTokenStatus()) ? status.getAccessTokenStatus() : ""));
        }
    }

    public static void trustAllCertificates() {
        try {
            TrustManager[] trustAllCerts = new TrustManager[]{
                    new X509TrustManager() {
                        public X509Certificate[] getAcceptedIssuers() {
                            X509Certificate[] myTrustedAnchors = new X509Certificate[0];
                            return myTrustedAnchors;
                        }

                        @Override
                        public void checkClientTrusted(X509Certificate[] certs, String authType) {
                        }

                        @Override
                        public void checkServerTrusted(X509Certificate[] certs, String authType) {
                        }
                    }
            };

            SSLContext sc = SSLContext.getInstance("TLS");
            sc.init(null, trustAllCerts, new SecureRandom());
            HttpsURLConnection.setDefaultSSLSocketFactory(sc.getSocketFactory());
            HttpsURLConnection.setDefaultHostnameVerifier(new HostnameVerifier() {
                @Override
                @SuppressLint("BadHostnameVerifier")
                public boolean verify(String arg0, SSLSession arg1) {
                    // TODO verify the hostname to return true only when we trust it.
                    return true;
                }
            });
        } catch (Exception e) {
            Log.e(TAG, "Error setting up default trust manager and host name verifier", e);
        }
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        // Inflate the menu; this adds items to the action bar if it is present.
        getMenuInflater().inflate(R.menu.menu_main, menu);
        return true;
    }

    @Override
    public boolean onPrepareOptionsMenu(Menu menu) {
        IdentityProvider identityProvider = ServiceContextFactory.getInstance().getIdentityProvider();
        if (null != identityProvider) {
            MenuItem itemAuthenticate = menu.findItem(R.id.action_authenticate);
            MenuItem itemUnAuthenticate = menu.findItem(R.id.action_unauthenticate);
            itemAuthenticate.setEnabled(true);
            menu.findItem(R.id.action_define_idp).setEnabled(false);
            // Is the identity provider authenticated?
            if ((null == ServiceContextFactory.getInstance().getRefreshToken(this)) &&
               (null == ServiceContextFactory.getInstance().getAccessToken(this)))
            {   // No, disable the un-authenticate menu item.
                itemUnAuthenticate.setEnabled(false);
                itemUnAuthenticate.setShowAsAction(MenuItem.SHOW_AS_ACTION_NEVER);
                itemAuthenticate.setShowAsAction(MenuItem.SHOW_AS_ACTION_IF_ROOM);
                menu.findItem(R.id.action_invoke).setEnabled(false);
            }
            else
            {   // Yes, prefer to put the "Un-Authenticate" action as the "shown"
                // action.
                itemUnAuthenticate.setEnabled(true);
                itemUnAuthenticate.setShowAsAction(MenuItem.SHOW_AS_ACTION_IF_ROOM);
                itemAuthenticate.setShowAsAction(MenuItem.SHOW_AS_ACTION_NEVER);
                menu.findItem(R.id.action_invoke).setEnabled( (null != identityProvider.getUserInfoEndPoint()) );
            }
        } else {
            MenuItem itemAuthenticate = menu.findItem(R.id.action_authenticate);
            itemAuthenticate.setEnabled(false);
            itemAuthenticate.setShowAsAction(MenuItem.SHOW_AS_ACTION_NEVER);
            MenuItem itemUnAuthenticate = menu.findItem(R.id.action_unauthenticate);
            itemUnAuthenticate.setEnabled(false);
            itemUnAuthenticate.setShowAsAction(MenuItem.SHOW_AS_ACTION_NEVER);
            MenuItem itemDefineIdp = menu.findItem(R.id.action_define_idp);
            itemUnAuthenticate.setEnabled(true);
            itemDefineIdp.setShowAsAction(MenuItem.SHOW_AS_ACTION_IF_ROOM);
            menu.findItem(R.id.action_delete_idp).setEnabled(false);
            menu.findItem(R.id.action_edit_idp).setEnabled(false);
            menu.findItem(R.id.action_invoke).setEnabled(false);
        }
        return super.onPrepareOptionsMenu(menu);
    }

    @Override
    public boolean onOptionsItemSelected(MenuItem item)
    {
        int id = item.getItemId();
        if (id == R.id.action_settings) {
            Intent settingsIntent = new Intent(MainActivity.this, SettingsActivity.class);
            Settings settings = new Settings(MainActivity.this);
            settingsIntent.putExtra(SettingsActivity.TAG_ACCESS_TOKEN_REFRESH_BEFORE_EXPIRATION_BY, settings.getAccessTokenRefreshBeforeExpirationBy());
            settingsIntent.putExtra(SettingsActivity.TAG_ENCRYPTION_KEY_AGE, DateTimeUtil.getDuration(ServiceContextFactory.getInstance().getKeyAge()));
            startActivityForResult(settingsIntent, SettingsActivity.COORELATIONID_SETTINGS);
        }
        else if (id == R.id.action_define_idp)
        {
            Intent identityProviderEditorIntent = new Intent(MainActivity.this, IdentityProviderEditorActivity.class);
            startActivityForResult(identityProviderEditorIntent, IdentityProviderEditorActivity.COORELATIONID_IDENTITY_PROVIDER_EDITOR);
        }
        else if (id == R.id.action_edit_idp) {
            IdentityProvider identityProvider = ServiceContextFactory.getInstance().getIdentityProvider();
            Intent identityProviderEditorIntent = new Intent(MainActivity.this, IdentityProviderEditorActivity.class);
            identityProviderEditorIntent.putExtra(IdentityProviderEditorActivity.TAG_IDENTITY_PROVIDER_EDITOR_JSON, identityProvider.toJson());
            MainActivity.this.startActivityForResult(identityProviderEditorIntent, IdentityProviderEditorActivity.COORELATIONID_IDENTITY_PROVIDER_EDITOR);

        }
        else if (id == R.id.action_delete_idp)
        {
            new AskDeleteIdentityProvider().show();
        }
        else if (id == R.id.action_authenticate)
        {
            IdentityProvider identityProvider = ServiceContextFactory.getInstance().getIdentityProvider();
            Intent intent = new Intent(MainActivity.this, AuthenticateActivity.class);
            Log.d(TAG, "initiating auth for " + identityProvider.getProviderUrl());
            intent.putExtra(AuthenticateActivity.AUTHENTICATE_IDENTITY_PROVIDER_JSON, identityProvider.toJson());
            startActivity(intent);
        }
        else if (id == R.id.action_unauthenticate)
        {
            IdentityProvider identityProvider = ServiceContextFactory.getInstance().getIdentityProvider();
            Intent intent = new Intent(MainActivity.this, UnAuthenticateActivity.class);
            Log.d(TAG, "initiating un-auth for " + identityProvider.getProviderUrl());
            intent.putExtra(UnAuthenticateActivity.UNAUTHENTICATE_IDENTITY_PROVIDER_JSON, identityProvider.toJson());
            MainActivity.this.startActivity(intent);
        }
        else if (id == R.id.action_invoke)
        {
            try
            {
                IdentityProvider identityProvider = ServiceContextFactory.getInstance().getIdentityProvider();
                if (null != identityProvider.getUserInfoEndPoint()) {
                    ServiceContextFactory.getInstance().invokeWithFreshToken(MainActivity.this,
                            new InvokeTaskHttpGet(identityProvider.getUserInfoEndPoint().toString(), null, new InvokeClientCallbackImpl()),
                            DateTimeUtil.MINUTE_IN_MILLIS);
                }
                else throw new Exception("Missing User Info End Point");
            }
            catch (Exception ex)
            {
                Toast.makeText(MainActivity.this, ex.getMessage(), Toast.LENGTH_LONG).show();
            }
        }
        return super.onOptionsItemSelected(item);
    }

    private class BrowserFetcher extends AsyncTask<Void, Void, Void> {

        @Override
        protected Void doInBackground(Void... voids)
        {
            browserDescriptors = BrowserSelector.getAllBrowsers(MainActivity.this);
            PackageManager pm = MainActivity.this.getPackageManager();
            for (BrowserDescriptor descriptor : browserDescriptors)
            {
                try
                {
                    ApplicationInfo info = pm.getApplicationInfo(descriptor.packageName, 0);
                    String label =  pm.getApplicationLabel(info).toString();

                    if (descriptor.useCustomTab)
                    {
                        browserList.add(label + " Custom Tab");
                    }
                    else
                    {
                        browserList.add(label);
                    }
                }
                catch (PackageManager.NameNotFoundException e)
                {
                    e.printStackTrace();
                    browserList.add(descriptor.packageName);
                }
            }
            return null;
        }
    }

    private class AskDeleteIdentityProvider {
        private AlertDialog deleteDialog;
        private IdentityProvider identityProvider;

        public AskDeleteIdentityProvider() {
            IdentityProvider identityProvider = ServiceContextFactory.getInstance().getIdentityProvider();
            deleteDialog = new AlertDialog.Builder(MainActivity.this)
                    .setTitle("Delete")
                    .setMessage("Delete Identity Provider: " + identityProvider.getHost())
                    .setIcon(R.drawable.ic_delete_forever)
                    .setPositiveButton("Delete", new DialogInterface.OnClickListener() {
                        public void onClick(DialogInterface dialog, int whichButton) {
                            ServiceContextFactory.getInstance().deleteIdentityProvider(MainActivity.this);
                            MainActivity.this.invalidateOptionsMenu();
                            MainActivity.this.refreshUI();
                            dialog.dismiss();
                        }
                    })
                    .setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
                        public void onClick(DialogInterface dialog, int which) {
                            dialog.dismiss();
                        }
                    })
                    .create();
        }

        public void show() {
            deleteDialog.show();
        }
    }

    private class InvokeClientCallbackImpl
        implements InvokeClientCallback
    {
        @Override
        public void onInvokeComplete(InvokeTask task)
        {
            if (null != task.getException()) {
                Toast.makeText(MainActivity.this, "Error invoking User Info URL: " + task.getException().getMessage(), Toast.LENGTH_LONG).show();
            }
            else
            {
                int httpResponseCode = task.getHttpResponseCode();
                ByteArrayOutputStream httpBinaryPayload = task.getHttpBinaryPayload();
                Toast.makeText(MainActivity.this, "Success invoking User Info URL: HTTP Code: " + httpResponseCode + ", Message size: " + httpBinaryPayload.size() + ", of type " + task.getContentType() +
                        " " + httpBinaryPayload.toString(), Toast.LENGTH_LONG).show();
                if (task.isAccessTokenRefreshed())
                {
                    refreshUI();
                }
            }
        }
    }

}
