/*
 * Copyright (c) 2017 Microfocus Corporation. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not use this file except
 * in compliance with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software distributed under the
 * License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 * express or implied. See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.microfocus.android.sampleapp.shyanimals;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.os.Bundle;
import android.support.v7.app.AppCompatActivity;
import android.text.Editable;
import android.text.TextWatcher;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.Button;
import android.widget.EditText;
import android.widget.TextView;

import com.microfocus.android.mobile.sdk.library.ServiceContext;
import com.microfocus.android.mobile.sdk.library.ServiceContextFactory;
import com.microfocus.android.mobile.sdk.library.util.DateTimeUtil;

/**
 * Activity that allows editing of settings associated with the sample application Shy Animals.
 */
public class SettingsActivity extends AppCompatActivity {
    private static final String TAG = "SettingsActivity";

    static final int COORELATIONID_SETTINGS = 15288;

    static final String TAG_ACCESS_TOKEN_REFRESH_BEFORE_EXPIRATION_BY = "accessTokenRefreshBeforeExpirationBy";
    static final String TAG_ENCRYPTION_KEY_AGE = "encryptionKeyAge";

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_settings);

        Intent intent = getIntent();
        long lAccessToken = intent.getLongExtra(TAG_ACCESS_TOKEN_REFRESH_BEFORE_EXPIRATION_BY, ServiceContext.ACCESS_TOKEN_REFRESH_BEFORE_EXPIRATION_BY_DEFAULT);
        String strEncryptionKeyAge = intent.getStringExtra(TAG_ENCRYPTION_KEY_AGE);

        long lAccessTokenHours = lAccessToken / DateTimeUtil.HOUR_IN_MILLIS;
        long lRemainder = lAccessToken % DateTimeUtil.HOUR_IN_MILLIS;
        long lAccessTokenMinutes = lRemainder / DateTimeUtil.MINUTE_IN_MILLIS;
        lRemainder = lRemainder % DateTimeUtil.MINUTE_IN_MILLIS;
        long lAccessTokenSeconds = lRemainder / DateTimeUtil.SECOND_IN_MILLIS;

        EditText accessTokenHours = (EditText)findViewById(R.id.accessTokenWindowHours);
        EditText accessTokenMinutes = (EditText)findViewById(R.id.accessTokenWindowMinutes);
        EditText accessTokenSeconds = (EditText)findViewById(R.id.accessTokenWindowSeconds);
        accessTokenHours.setText("" + lAccessTokenHours);
        accessTokenMinutes.setText("" + lAccessTokenMinutes);
        accessTokenSeconds.setText("" + lAccessTokenSeconds);

        accessTokenHours.addTextChangedListener(new NumberTextWatcher(0, 23));
        accessTokenMinutes.addTextChangedListener(new NumberTextWatcher(0, 59));
        accessTokenSeconds.addTextChangedListener(new NumberTextWatcher(0, 59));

        TextView encryptionKeyAge = (TextView)findViewById(R.id.encryptionKeyAge);
        encryptionKeyAge.setText("Age: " + strEncryptionKeyAge);

        Button rollEncryptionKeyButton = (Button)findViewById(R.id.rollEncryptionKeyButton);
        rollEncryptionKeyButton.setOnClickListener(new Button.OnClickListener(){
            @Override
            public void onClick(View view) {
                new SettingsActivity.AskRollEncryptionKey(SettingsActivity.this).show();
            }
        });
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu)
    {
        MenuItem add = menu.add("Save");
        add.setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS);
        add.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem menuItem) {
                Intent result = new Intent();

                EditText accessTokenHours = (EditText)findViewById(R.id.accessTokenWindowHours);
                EditText accessTokenMinutes = (EditText)findViewById(R.id.accessTokenWindowMinutes);
                EditText accessTokenSeconds = (EditText)findViewById(R.id.accessTokenWindowSeconds);

                long lAccessToken = 0;
                try {
                    int iAccessTokenHours = Integer.parseInt(accessTokenHours.getText().toString());
                    int iAccessTokenMinutes = Integer.parseInt(accessTokenMinutes.getText().toString());
                    int iAccessTokenSeconds = Integer.parseInt(accessTokenSeconds.getText().toString());
                    lAccessToken = (iAccessTokenHours * DateTimeUtil.HOUR_IN_MILLIS) + (iAccessTokenMinutes * DateTimeUtil.MINUTE_IN_MILLIS) + (iAccessTokenSeconds * DateTimeUtil.SECOND_IN_MILLIS);
                }
                catch (Exception e)
                {   // Error set to defaults
                    lAccessToken = ServiceContext.ACCESS_TOKEN_REFRESH_BEFORE_EXPIRATION_BY_DEFAULT;
                    Log.e(TAG, "Error parsing settings results. Reverting to defaults. Exception: " + e.toString());
                }

                result.putExtra(TAG_ACCESS_TOKEN_REFRESH_BEFORE_EXPIRATION_BY, lAccessToken);

                setResult(Activity.RESULT_OK, result);
                finish();
                return false;
            }
        });
        return true;
    }

    private class NumberTextWatcher
        implements TextWatcher
    {
        private int minimum;
        private int maximum;

        public NumberTextWatcher(int minimum, int maximum)
        {
            this.minimum = minimum;
            this.maximum = maximum;
        }

        @Override
        public void onTextChanged(CharSequence currentDigits, int start, int before, int count) {
        }

        @Override
        public void beforeTextChanged(CharSequence s, int start, int count, int after) {
        }

        @Override
        public void afterTextChanged(Editable s) {
            String strExistingValue = s.toString();
            if (null != strExistingValue) {
                int iReplacementValue = minimum;
                boolean bReplace = false;
                if (strExistingValue.isEmpty())
                {
                    bReplace = true;
                }
                else {
                    try {
                        int iValue = Integer.parseInt(s.toString());
                        if (iValue < minimum) {
                            bReplace = true;
                        } else if (iValue > maximum) {
                            bReplace = true;
                            iReplacementValue = maximum;
                        }
                    } catch (Exception e) {
                        bReplace = true;
                    }
                }
                if (bReplace) {
                    String strReplacementValue = "" + iReplacementValue;
                    s.replace(0, s.length(), strReplacementValue, 0, strReplacementValue.length());
                }
            }
        }
    }

    private class AskRollEncryptionKey {
        private Context context;
        private AlertDialog askDialog;

        public AskRollEncryptionKey(Context context) {
            this.context = context;
            askDialog = new AlertDialog.Builder(context)
                    .setTitle("Roll Encryption Key")
                    .setMessage("Roll encryption key? This cannot be reversed.")
                    .setIcon(R.drawable.ic_roll_encryption_key)
                    .setPositiveButton("Roll", new DialogInterface.OnClickListener() {
                        public void onClick(DialogInterface dialog, int whichButton) {
                            ServiceContextFactory.getInstance().rollKey(AskRollEncryptionKey.this.context, new RollEncryptionKeyCallBack());
                            dialog.dismiss();
                        }
                    })
                    .setNegativeButton("Cancel", new DialogInterface.OnClickListener() {
                        public void onClick(DialogInterface dialog, int which) {
                            dialog.dismiss();
                        }
                    })
                    .create();
        }

        public void show() {
            askDialog.show();
        }
    }

    public class RollEncryptionKeyCallBack implements ServiceContext.RollKeyCallback
    {
        public void onComplete(Context context)
        {
            TextView encryptionKeyAge = (TextView)findViewById(R.id.encryptionKeyAge);
            long lKeyAge = ServiceContextFactory.getInstance().getKeyAge();
            encryptionKeyAge.setText("Age: " + DateTimeUtil.getDuration(lKeyAge));
        }
    }
}
