package com.microfocus.android.sampleapp.shyanimals;

import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.support.v7.app.AppCompatActivity;
import android.view.Menu;
import android.view.MenuItem;
import android.widget.CheckBox;
import android.widget.RadioButton;
import android.widget.RadioGroup;
import android.widget.TextView;
import android.widget.Toast;

import com.microfocus.android.mobile.sdk.library.IdentityProvider;
import com.microfocus.android.mobile.sdk.library.ServiceContextFactory;
import com.microfocus.android.mobile.sdk.library.UnAuthenticator;
import com.microfocus.android.mobile.sdk.library.UnAuthenticatorBuilder;
import com.microfocus.android.mobile.sdk.library.UnAuthenticatorClientCallback;
import com.microfocus.android.mobile.sdk.library.UnAuthenticatorReport;
import com.microfocus.android.mobile.sdk.library.util.DateTimeUtil;

import net.openid.appauth.browser.BrowserDescriptor;
import net.openid.appauth.browser.BrowserSelector;

import java.util.List;

/**
 * The activity invoked to initiate an UnAuthentication. Presents the end user
 * with per-un-authentication options to customize the un-authentication process
 * and the 'Go' button to start the un-authentication.
 */
public class UnAuthenticateActivity extends AppCompatActivity {
    final static String UNAUTHENTICATE_IDENTITY_PROVIDER_JSON = "identityProviderJson";
    final static String TAG = "UnAuthenticateActivity";
    private IdentityProvider identityProvider;
    private List<BrowserDescriptor> browserDescriptors;
    private RadioGroup radioGroup = null;

    @Override
    protected void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_unauthenticate);
        Intent intent = getIntent();
        String json = intent.getStringExtra(UNAUTHENTICATE_IDENTITY_PROVIDER_JSON);
        try {
            if (null == json) {
                throw new Exception("Identity Provider JSON Missing");
            }
            identityProvider = ServiceContextFactory.getInstance().getIdentityProviderBuilder(json).build();
            ((TextView) findViewById(R.id.unauthenticate_domain_name)).setText(identityProvider.getHost());
            ((TextView) findViewById(R.id.unauthenticate_client_id)).setText(identityProvider.getClientId());
            ((TextView) findViewById(R.id.unauthenticate_timeout_value)).setText("" + (UnAuthenticatorBuilder.TIMEOUT_DEFAULT / DateTimeUtil.SECOND_IN_MILLIS));

            browserDescriptors = BrowserSelector.getAllBrowsers(UnAuthenticateActivity.this);
            PackageManager pm = UnAuthenticateActivity.this.getPackageManager();
            radioGroup = (RadioGroup) findViewById(R.id.unAuthBrowserRadioGroup);

            for (int i = 0; i < browserDescriptors.size(); i++) {
                String label;
                RadioButton rb = new RadioButton(this);
                try {
                    ApplicationInfo info = pm.getApplicationInfo(browserDescriptors.get(i).packageName, 0);
                    label = pm.getApplicationLabel(info).toString();

                    if (browserDescriptors.get(i).useCustomTab) {
                        label += " Custom Tab";
                    }
                } catch (PackageManager.NameNotFoundException e) {
                    e.printStackTrace();
                    label = browserDescriptors.get(i).packageName;
                }
                rb.setText(label);
                rb.setId(i);
                if (i == 0)
                    rb.setChecked(true);
                radioGroup.addView(rb);
            }
        } catch (Exception e) {
            ((TextView) findViewById(R.id.unauthenticate_domain_name)).setText("Identity Provider Not Available");
            ((TextView) findViewById(R.id.unauthenticate_client_id)).setText("Error loading Identity Provider");
        }
    }

    @Override
    public boolean onCreateOptionsMenu(Menu menu) {
        MenuItem go = menu.add("Go");
        go.setShowAsAction(MenuItem.SHOW_AS_ACTION_ALWAYS);
        go.setOnMenuItemClickListener(new MenuItem.OnMenuItemClickListener() {
            @Override
            public boolean onMenuItemClick(MenuItem menuItem) {

                try {
                    Settings settings = new Settings(UnAuthenticateActivity.this);
                    UnAuthenticatorBuilder builder = ServiceContextFactory.getInstance().getUnAuthenticatorBuilder(UnAuthenticateActivity.this, new UnAuthenticateActivity.UnAuthenticatorCallbackImpl());
                    builder.setBrowserDescriptor(browserDescriptors.get(radioGroup.getCheckedRadioButtonId()));
                    // If there is a specified timeout, place it in the builder
                    String strTimeoutSeconds = ((TextView) findViewById(R.id.unauthenticate_timeout_value)).getText().toString();
                    if ((null != strTimeoutSeconds) && (!strTimeoutSeconds.isEmpty()))
                    {
                        try
                        {
                            long lTimeoutSeconds = Long.parseLong(strTimeoutSeconds);
                            builder.setTimeout(lTimeoutSeconds * DateTimeUtil.SECOND_IN_MILLIS);
                        }
                        catch (Exception e)
                        {
                            // Ignore parse exception. Leave timeout at default value.
                        }
                    }
                    CheckBox checkBoxDisallowTokenRevocation = (CheckBox)findViewById(R.id.checkDisallowTokenRevocation);
                    builder.setDisallowTokenRevocation(checkBoxDisallowTokenRevocation.isChecked());
                    CheckBox checkBoxDisallowBroswerBasedLogout = (CheckBox)findViewById(R.id.checkDisallowBroswerBasedLogout);
                    builder.setDisallowBrowserBasedLogout(checkBoxDisallowBroswerBasedLogout.isChecked());

                    UnAuthenticator uaa = builder.build();
                    ServiceContextFactory.getInstance().unauthenticate(uaa);
                } catch (Exception e) {
                    Toast.makeText(UnAuthenticateActivity.this, e.getMessage(), Toast.LENGTH_LONG).show();
                }
                return false;
            }
        });
        return true;
    }

    private class UnAuthenticatorCallbackImpl
            implements UnAuthenticatorClientCallback {
        public void onUnAuthenticationComplete(UnAuthenticatorReport report) {
            Intent intent = new Intent(UnAuthenticateActivity.this, UnAuthenticateReportActivity.class);
            intent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
            String json = ServiceContextFactory.getInstance().toJson(report);
            intent.putExtra(AuthenticateReportActivity.REPORT_ACTIVITY_JSON, json);
            startActivity(intent);
        }
    }
}