/*
  Copyright (c) 2007 Novell, Inc. All Rights Reserved.

  Novell grants permission, free of charge, to any person obtaining copies
  of this software and its associated documentation files (the "Software"),
  to deal in the Software without restriction, including to use, copy, adapt, 
  publish, distribute, display, perform, sublicense, and sell copies of the 
  Software, subject to the following condition: You must include the above 
  copyright notice and this permission notice in all full or partial copies 
  of the Software.

  NOVELL PROVIDES THE SOFTWARE "AS IS," WITHOUT ANY EXPRESS OR IMPLIED WARRANTY,
  INCLUDING WITHOUT THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
  PARTICULAR PURPOSE, AND NON-INFRINGMENT.  NOVELL, THE AUTHORS OF THE SOFTWARE,
  AND THE OWNERS OF COPYRIGHT IN THE SOFTWARE ARE NOT LIABLE FOR ANY CLAIM, DAMAGES,
  OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT, OR OTHERWISE, ARISING
  FROM, OUT OF, OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/

package com.novell.nam.authentication;

import java.util.*;

import javax.servlet.http.*;

import org.eclipse.higgins.sts.api.*;

import com.novell.nidp.*;
import com.novell.nidp.authentication.*;
import com.novell.nidp.authentication.local.*;
import com.novell.nidp.common.authority.*;

public class PasswordClass extends LocalAuthenticationClass implements STSAuthenticationClass
{
    /**
     * Constructor for form based authentication
     *
     * @param props      Properties associated with the implementing class
     * @param uStores    List of ordered user stores to authenticate against
     */
    public PasswordClass(Properties props, ArrayList<UserAuthority> uStores)
    {
        super(props,uStores);
    }

    /**
     * Get the authentication type this class implements
     * 
     * @return	returns the authentication type represented by this class
     */
    public String getType()
    {
        return AuthnConstants.PASSWORD;
    }
 
    public void initializeRequest(HttpServletRequest    request,
			  					  HttpServletResponse   response,
			  					  NIDPSession           session,
			  					  NIDPSessionData		data,
			  					  boolean               following,
			  					  String                url)
	{
		super.initializeRequest(request, response, session, data, following, url);
	}

    /**
     * Perform form based authentication.  This method gets called on each response
     * during authentication process
     * 
     * @return  returns the status of the authentication process which is
     *          one of AUTHENTICATED, NOT_AUTHENTICATED, CANCELLED, HANDLED_REQUEST, 
     *          PWD_EXPIRING, PWD_EXPIRED
     */
    protected int doAuthenticate()
    {
        // If this is the first time the class is called following another method
        // we want to display the form that will get the credentials.  This method
        // prevents a previous form from providing data to the next form if any
        // parameter names end up being the same
        if (!isFirstCallAfterPrevMethod())
        {
            // This wasnt first time method was called, so see if data can be processed
            int status = handlePostedData();
            if (status != NOT_AUTHENTICATED)
                return status;
        }
        
        String jsp = getProperty(AuthnConstants.PROPERTY_JSP); 
        if (jsp == null || jsp.length() == 0)
            jsp = NIDPConstants.JSP_LOGIN;            

        m_PageToShow = new PageToShow(jsp);
        m_PageToShow.addAttribute(NIDPConstants.ATTR_URL, (getReturnURL() != null ? getReturnURL() : m_Request.getRequestURL().toString()));
        if (getAuthnRequest() != null && getAuthnRequest().getTarget() != null)
        	m_PageToShow.addAttribute("target", getAuthnRequest().getTarget());

        return SHOW_JSP;
    }
     
    /**
     * Get and process the data that is posted from the form
     *
     * @return  returns the status of the authentication process which is
     *          one of AUTHENTICATED, NOT_AUTHENTICATED, CANCELLED, HANDLED_REQUEST, 
     *          PWD_EXPIRING, PWD_EXPIRED
     */
    private int handlePostedData()
    {
        // Look for a name and password
        String id       = m_Request.getParameter(NIDPConstants.PARM_USERID);
        String password = m_Request.getParameter(NIDPConstants.PARM_PASSWORD);
        
        // Check to see if admin has setup for a custom query
    	String ldapQuery = checkForQuery();
    	
        try
        {
            // using admin defined attributes for query
            if (ldapQuery != null)
            {
            	if (authenticateWithQuery(ldapQuery,password)) 
            		return AUTHENTICATED;
            }
            
            // If using default of name and password
            else
            {
                if (id == null || id.length() == 0)
                    return NOT_AUTHENTICATED;
         
                if (authenticateWithPassword(id,password))
                    return AUTHENTICATED;
            }            
        }
        catch (PasswordExpiringException pe)
        {
            return PWD_EXPIRING;
        }
        catch (PasswordExpiredException pe)
        {
            return PWD_EXPIRED;
        }
        
        m_Request.setAttribute(NIDPConstants.ATTR_LOGIN_ERROR, getUserErrorMsg());
        return NOT_AUTHENTICATED;
    }

    public NIDPPrincipal handleSTSAuthentication(ISecurityInformation securityInformation)
    {
	   	IUsernameToken usernameToken = 
	   		(IUsernameToken)securityInformation.getFirst(IUsernameToken.class);
		   	
   		if (null != usernameToken)
   		{
   	        try
   	        {
   	            if (authenticateWithPassword(usernameToken.getUsername(),usernameToken.getPassword()))
   	                return getPrincipal();
   	        }
   	        catch (PasswordExpiringException pe)
   	        {
   	            return getPrincipal();
   	        }
   	        catch (PasswordExpiredException pe) {}
   		}
   		return null;
    }
}