/*
  Copyright (c) 2008, 2012 Novell, Inc. All Rights Reserved.

  Novell grants permission, free of charge, to any person obtaining copies
  of this software and its associated documentation files (the "Software"),
  to deal in the Software without restriction, including to use, copy, adapt, 
  publish, distribute, display, perform, sublicense, and sell copies of the 
  Software, subject to the following condition: You must include the above 
  copyright notice and this permission notice in all full or partial copies 
  of the Software.

  NOVELL PROVIDES THE SOFTWARE "AS IS," WITHOUT ANY EXPRESS OR IMPLIED WARRANTY,
  INCLUDING WITHOUT THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
  PARTICULAR PURPOSE, AND NON-INFRINGMENT.  NOVELL, THE AUTHORS OF THE SOFTWARE,
  AND THE OWNERS OF COPYRIGHT IN THE SOFTWARE ARE NOT LIABLE FOR ANY CLAIM, DAMAGES,
  OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT, OR OTHERWISE, ARISING
  FROM, OUT OF, OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/

package com.novell.nam.custom.policy.data;


import java.util.Hashtable;

import javax.naming.Context;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.Control;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;

import com.novell.nxpe.NxpeContextDataElement;
import com.novell.nxpe.NxpeException;
import com.novell.nxpe.NxpeInformationContext;
import com.novell.nxpe.NxpeParameter;
import com.novell.nxpe.NxpeParameterList;
import com.novell.nxpe.NxpeResponseContext;
import com.novell.nxpe.NxpeResult;


/**
 * 
 * @author Paladin Team <nxpe-dev@forge.provo.novell.com>
 * @since NXPE1.0
 */
public final class LDAPGroupDataElement implements NxpeContextDataElement
{

    private static final String USER_STORE_NAME = "User Store";
    private static final int EV_USER_STORE = 11;

    private static final String AUTHENTICATION_NAME = "Authentication";
    private static final int EV_AUTHENTICATION = 211;
    private static final String DEFAULT_AUTHENTICATION = "simple";

    private static final String DIRECTORY_TYPE_NAME = "Directory Type";
    private static final int EV_DIRECTORY_TYPE = 222;
    private static final String DEFAULT_DIRECTORY_TYPE = "unknown";

    private static final String PROVIDER_URL_NAME = "User Store Replica";
    private static final int EV_PROVIDER_URL = 31;
    private static final String DEFAULT_PROVIDER_URL = "ldap://localhost:389";

    private static final String LDAP_USER_DN_NAME = "LDAP User DN";
    private static final int EV_LDAP_USER_DN = 41;

    private static final String SECURITY_PRINCIPAL_NAME = "Security Principal";
    private static final int EV_SECURITY_PRINCIPAL = 51;

    private static final String SECURITY_CREDENTIALS_NAME = "Security Credentials";
    private static final int EV_SECURITY_CREDENTIALS = 52;

    private static final String SEARCH_CONTEXT_NAME = "Search Context";
    private static final int EV_SEARCH_CONTEXT = 61;

    private static final String DEBUG_NAME = "Debug";
    private static final int EV_DEBUG = 91;

    private static final String CLS_GROUP = "group";
    private static final String CLS_GROUPOFNAMES = "groupofnames";
    private static final String CLS_GROUPOFUNIQUENAMES = "groupofuniquenames";

    private static final String ATTR_CN = "cn";

    
    // NxpeContextDataElement values
    private final String strName;
    private final int iEnumerativeValue;
    private final String strParameter;

    private NxpeParameterList configurationValues;

    private boolean debug;

    private String strUserStore;
    private String strProviderURL;
    private String strAuthentication;
    private String strDirectoryType;
    private String strLdapFilter;
    private SearchControls searchControls;


    //******************** LDAPGroupDataElement/constructor ********************

    /**
     * 
     */
    public LDAPGroupDataElement(
            String strName,
            int iEnumerativeValue,
            String strParameter)
        throws NxpeException
    {
        this.strName = strName;
        this.iEnumerativeValue = iEnumerativeValue;
        this.strParameter = strParameter;

    }


    //******************** NxpeContextDataElement ********************

    /**
     * Called by Nxpe to initialize the condition in preparation for policy 
     * evaluation
     * 
     * Notes: Derived classes are required to override this method.                                                             
     *     This method is guaranteed to be called before any other method is 
     *     called, as this is part of object construction.
     * 
     * @param configurationParameters List of configuration data required by the 
     *      external condition Handler. This object is only valid during the 
     *      lifetime of the init() method.  If the condition wants to preserve 
     *      config data, it must be copied to the condition handler. 
     * 
     * @throws NxpeException
     */
    public void initialize(
            NxpeParameterList configurationValues)
        throws NxpeException
    {
        this.configurationValues = configurationValues;

        setDebug(configurationValues);

        strProviderURL = DEFAULT_PROVIDER_URL;
        strAuthentication = DEFAULT_AUTHENTICATION;
        strDirectoryType = DEFAULT_DIRECTORY_TYPE;

        StringBuffer sbLdapFilter = new StringBuffer(128);

        // setup filter
        sbLdapFilter.append("(|(objectClass=");
        sbLdapFilter.append(CLS_GROUP);
        sbLdapFilter.append(")(objectClass=");
        sbLdapFilter.append(CLS_GROUPOFNAMES);
        sbLdapFilter.append(")(objectClass=");
        sbLdapFilter.append(CLS_GROUPOFUNIQUENAMES);
        sbLdapFilter.append("))");

        strLdapFilter = new String(sbLdapFilter);

        // setup search controls
        searchControls = new SearchControls();
        searchControls.setTimeLimit(0);
        searchControls.setReturningObjFlag(true);
        searchControls.setSearchScope(SearchControls.SUBTREE_SCOPE);
        searchControls.setReturningAttributes(new String[] { ATTR_CN });

    }

    /**
     * 
     */
    public int getEnumerativeValue()
    {
        return (iEnumerativeValue);
    }

    /**
     * 
     */
    public String getName()
    {
        return (strName);
    }

    /**
     * 
     */
    public String getParameter()
    {
        return (strParameter);
    }

    /**
     * 
     */
    public synchronized Object getValue(
            NxpeInformationContext informationContext,
            NxpeResponseContext responseContext)
        throws NxpeException
    {
        LdapContext ldapContext = null;

        setUserStore(informationContext);
        setProviderURL(informationContext);
        setAuthentication(informationContext);
        setDirectoryType(informationContext);

        String strLDAPUserDN = getLDAPUserDN(informationContext);
        String strDN = getSecurityPrincipal(informationContext);

        if (strLDAPUserDN == null)
        {
            strLDAPUserDN = strDN;
        }
        
        String strPassword = getSecurityCredentials(informationContext);
        String strSearchContext = getSearchContext(informationContext);

        StringBuffer sbLdapFilter = new StringBuffer(128);

        // setup filter : It can be any filter condition e.g a simple condition like  "(ou=admin-console*)" 
        // or as following
        sbLdapFilter.append("(&(|(member=");
        sbLdapFilter.append(strLDAPUserDN);
        sbLdapFilter.append(")");
        sbLdapFilter.append("(uniqueMember=");
        sbLdapFilter.append(strLDAPUserDN);
        sbLdapFilter.append("))");
        sbLdapFilter.append(strLdapFilter);
        sbLdapFilter.append(")");

        String strLdapMemberFilter = new String(sbLdapFilter);
    
        try
        {
        	String strGroupDNs = "";
            ldapContext = newInitialLdapContext(strDN, strPassword);
            NamingEnumeration neGroups = ldapContext.search(strSearchContext, strLdapMemberFilter, searchControls);
           
            while (neGroups.hasMore())
            {

                SearchResult srGroup = (SearchResult) neGroups.next();
                String strGroupDN = srGroup.getNameInNamespace();

                strGroupDNs = strGroupDNs + strGroupDN + "    ";

                if (debug)
                {
                    System.out.println("LDAPGroupDataElement: \"" + strGroupDN + "\"");

                }
            }
            return (strGroupDNs);

        }
        catch (NamingException e)
        {
            if (debug)
            {
                e.printStackTrace();

            }

            throw (new NxpeException(NxpeResult.ErrorDataUnavailable, e));

        }
        finally
        {
            if (ldapContext != null)
            {
                try
                {
                    ldapContext.close();

                }
                catch (NamingException e)
                {
                    if (debug)
                    {
                        System.out.println(e.getMessage());

                    }
                }
            }
        }

    } /* getValue() */


    //******************** LDAPGroupDataElement/private ********************

    /**
     * Optional:
     * 
     * @param informationContext
     * 
     * @return
     * 
     * @throws com.novell.nxpe.NxpeException
     */
    private String getLDAPUserDN(
            NxpeInformationContext informationContext)
        throws NxpeException
    {
        NxpeParameter pLDAPUserDN;

        if ((pLDAPUserDN = configurationValues.getParameter(EV_LDAP_USER_DN)) != null)
        {
            String strLDAPUserDN = (String) informationContext.getData(pLDAPUserDN);

            if (debug)
            {
                System.out.println(LDAP_USER_DN_NAME + ": " + strLDAPUserDN);
            }

            return (strLDAPUserDN);

        }

        if (debug)
        {
            System.out.println(LDAP_USER_DN_NAME + ": not present.");
        }

        return (null);

    }

    /**
     * Required:
     * 
     * @param informationContext
     * 
     * @return
     * 
     * @throws com.novell.nxpe.NxpeException
     */
    private String getSecurityPrincipal(
            NxpeInformationContext informationContext)
        throws NxpeException
    {
        NxpeParameter pSecurityPrincipal;

        if ((pSecurityPrincipal = configurationValues.getParameter(EV_SECURITY_PRINCIPAL)) == null)
        {
            NxpeException nxpeException = new NxpeException(NxpeResult.ErrorDataUnavailable, SECURITY_PRINCIPAL_NAME + ": not present.");

            if (debug)
            {
                nxpeException.printStackTrace();
            }

            throw (nxpeException);

        }

        String strSecurityPrincipal = (String) informationContext.getData(pSecurityPrincipal);

        if (debug)
        {
            System.out.println(SECURITY_PRINCIPAL_NAME + ": " + strSecurityPrincipal);
        }

        return (strSecurityPrincipal);

    }

    /**
     * Required:
     * 
     * @param informationContext
     * 
     * @return
     * 
     * @throws com.novell.nxpe.NxpeException
     */
    private String getSecurityCredentials(
            NxpeInformationContext informationContext)
        throws NxpeException
    {
        NxpeParameter pSecurityCredentials = configurationValues.getParameter(EV_SECURITY_CREDENTIALS);

        if ((pSecurityCredentials = configurationValues.getParameter(EV_SECURITY_CREDENTIALS)) == null)
        {
            NxpeException nxpeException = new NxpeException(NxpeResult.ErrorDataUnavailable, SECURITY_CREDENTIALS_NAME + ": not present.");

            if (debug)
            {
                nxpeException.printStackTrace();
            }

            throw (nxpeException);

        }

        String strSecurityCredentials = (String) informationContext.getData(pSecurityCredentials);

        if (debug)
        {
            System.out.println(SECURITY_CREDENTIALS_NAME + ": " + strSecurityCredentials);
        }

        return (strSecurityCredentials);

    }

    /**
     * Optional:
     * 
     * @param informationContext
     * 
     * @return
     * 
     * @throws com.novell.nxpe.NxpeException
     */
    private String getSearchContext(
            NxpeInformationContext informationContext)
        throws NxpeException
    {
        String strSearchContext = "";
        NxpeParameter pSearchContext;

        if ((pSearchContext = configurationValues.getParameter(EV_SEARCH_CONTEXT)) != null)
        {
            strSearchContext = (String) informationContext.getData(pSearchContext);

        }

        if (debug)
        {
            System.out.println(SEARCH_CONTEXT_NAME + ": \"" + strSearchContext + "\"");
        }

        return (strSearchContext);

    }

    /**
     * Optional:
     * 
     * @param informationContext
     * 
     * @throws com.novell.nxpe.NxpeException
     */
    private void setUserStore(
            NxpeInformationContext informationContext)
        throws NxpeException
    {
        NxpeParameter parameter;

        if ((parameter = configurationValues.getParameter(EV_USER_STORE)) != null)
        {
            strUserStore = (String) informationContext.getData(parameter);

            if (debug)
            {
                System.out.println(USER_STORE_NAME + ": " + strUserStore);

            }
        }
    }

    /**
     * Required:
     * 
     * The value of the ProviderURL property should contain a URL string (e.g. "ldap://somehost:389").
     * 
     * @param informationContext
     * 
     * @throws com.novell.nxpe.NxpeException
     */
    private void setProviderURL(
            NxpeInformationContext informationContext)
        throws NxpeException
    {
        NxpeParameter pProviderURL;

        if ((pProviderURL = configurationValues.getParameter(EV_PROVIDER_URL)) != null)
        {
            strProviderURL = (String) informationContext.getData(pProviderURL);

        }

        if (debug)
        {
            System.out.println(PROVIDER_URL_NAME + ": " + strProviderURL);

        }
    }

    /**
     * Optional:
     * 
     * @param informationContext
     * 
     * @throws com.novell.nxpe.NxpeException
     */
    private void setDirectoryType(
            NxpeInformationContext informationContext)
        throws NxpeException
    {
        NxpeParameter pDirectoryType;

        if ((pDirectoryType = configurationValues.getParameter(EV_DIRECTORY_TYPE)) != null)
        {
            strDirectoryType = (String) informationContext.getData(pDirectoryType);

        }

        if (debug)
        {
            System.out.println(DIRECTORY_TYPE_NAME + ": " + strDirectoryType);

        }
    }

    /**
     * Optional:
     * 
     * @param informationContext
     * 
     * @throws com.novell.nxpe.NxpeException
     */
    private void setAuthentication(
            NxpeInformationContext informationContext)
        throws NxpeException
    {
        NxpeParameter pAuthentication;

        if ((pAuthentication = configurationValues.getParameter(EV_AUTHENTICATION)) != null)
        {
            strAuthentication = (String) informationContext.getData(pAuthentication);

        }

        if (debug)
        {
            System.out.println(AUTHENTICATION_NAME + ": " + strAuthentication);

        }
    }

    /**
     * Optional:
     * 
     * @param configurationValues
     * 
     * @throws com.novell.nxpe.NxpeException
     */
    private void setDebug(
            NxpeParameterList configurationValues)
        throws NxpeException
    {
        NxpeParameter parameter = configurationValues.getParameter(EV_DEBUG);

        if (parameter != null)
        {
            debug = Boolean.parseBoolean(parameter.getValue());

            System.out.println("Debug: " + debug);

        }
    }

    /**
     * 
     * @param principal
     * @param credentials
     * 
     * @return
     * 
     * @throws javax.naming.NamingException
     */
    private LdapContext newInitialLdapContext(String principal, String credentials)
        throws NamingException
    {
        Hashtable<String, String> environment = new Hashtable<String, String>();
        LdapContext ldapContext;

        environment.put(Context.INITIAL_CONTEXT_FACTORY, "com.sun.jndi.ldap.LdapCtxFactory");
        environment.put(Context.PROVIDER_URL, strProviderURL);

        if (strProviderURL.startsWith("ldaps://"))
        {
            environment.put(Context.SECURITY_PROTOCOL, "ssl");
        }

        environment.put(Context.SECURITY_AUTHENTICATION, strAuthentication);
        environment.put(Context.SECURITY_PRINCIPAL, principal);
        environment.put(Context.SECURITY_CREDENTIALS, credentials);

        environment.put(Context.REFERRAL, "follow");

        ldapContext = new InitialLdapContext(environment, (Control[]) null);

        return (ldapContext);

    }

} /* LDAPGroupDataElement */
