/*
  Copyright (c) 2012 Novell, Inc. All Rights Reserved.

  Novell grants permission, free of charge, to any person obtaining copies
  of this software and its associated documentation files (the "Software"),
  to deal in the Software without restriction, including to use, copy, adapt, 
  publish, distribute, display, perform, sublicense, and sell copies of the 
  Software, subject to the following condition: You must include the above 
  copyright notice and this permission notice in all full or partial copies 
  of the Software.

  NOVELL PROVIDES THE SOFTWARE "AS IS," WITHOUT ANY EXPRESS OR IMPLIED WARRANTY,
  INCLUDING WITHOUT THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
  PARTICULAR PURPOSE, AND NON-INFRINGMENT.  NOVELL, THE AUTHORS OF THE SOFTWARE,
  AND THE OWNERS OF COPYRIGHT IN THE SOFTWARE ARE NOT LIABLE FOR ANY CLAIM, DAMAGES,
  OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT, OR OTHERWISE, ARISING
  FROM, OUT OF, OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/

package com.novell.nam.custom.policy.condition;

import com.novell.nxpe.NxpeCondition;
import com.novell.nxpe.NxpeException;
import com.novell.nxpe.NxpeInformationContext;
import com.novell.nxpe.NxpeParameter;
import com.novell.nxpe.NxpeParameterList;
import com.novell.nxpe.NxpeResponseContext;
import com.novell.nxpe.NxpeResult;

/**
 * This is an example code for a "Condition" type of Policy Extension.
 * evaluate() is the method which has to be modified by teh user to code the
 * condition evaluation logic. Most of other code can be retained as it is.
 */
public class PolicyConditionExtnTemplate implements NxpeCondition {

	/**
	 * ----BEGIN---- Initializing the constants for parameters passed to this
	 * class Configuration parameter can be configured from the administrator
	 * console UI. Go to Policies Tab->Extensions-><class extention>
	 * ->Configuration Parameters:->New
	 */

	/**
	 * Name parameter is used for reference in the code, can be used for
	 * logging. EV(Enumeration Value) will be used for fetching the parameter.
	 * This number must match the number configured at UI as ID parameter
	 */

	/**
	 * Unique string value for the condition. This value is used for tracing
	 * evaluation by policy engine
	 */
	private String interfaceID;

	/** Variable used to turn on/off the logging for debugging */
	private boolean debug;

	/** NxpeParameterList will be initialized during initialize() call */
	private NxpeParameterList configurationValues;

	/**
	 * This parameter is a string constant set in policy by the admin to turn
	 * debug on/off
	 */
	private static final String DEBUG_NAME = "Debug";
	private static final int EV_DEBUG = 91;

	/** This parameter is Credential Profile:LDAP Credentials:LDAP User DN */
	private static final String LDAP_USER_DN_NAME = "LDAP User DN";
	private static final int EV_LDAP_USER_DN = 41;

	/**
	 * Initialize the configuration values
	 * 
	 * @required
	 * @param configurationValues
	 *            :NxpeParameterList: The configuration parameters passed by
	 *            policy engine are used to initialize the NxpeCondition object
	 *            and are the parameters that the extension needs for evaluating
	 *            the condition.
	 * 
	 * @throws com.novell.nxpe.NxpeException
	 */
	@Override
	public void initialize(NxpeParameterList configurationValues)
			throws NxpeException {

		this.configurationValues = configurationValues;
		/** set debug as true by Default */
		this.debug = true;
		setDebug(configurationValues);

	}

	/**
	 * Sets the unique string value for the condition. This value is used for
	 * tracing evaluation
	 */
	@Override
	public void setInterfaceId(String arg0) throws NxpeException {
		this.interfaceID = arg0;

	}

	/**
	 * Required: evaluate() is the method which has to be modified by user to
	 * code the condition evaluation logic required.This method is called by the
	 * policy engine when the condition extension needs to be evaluated for a
	 * policy.
	 * 
	 * @param informationContext
	 *            :The informationContext parameter contains the parameter
	 *            information the extension needs from the policy engine to
	 *            evaluate the condition.
	 * @param responseContext
	 *            :The responseCtx is a reflection object for communicating
	 *            detailed response information back to the application. This is
	 *            additional information and does not replace the need to place
	 *            an evaluation result in the return value from this call.
	 * 
	 * @return NxpeResult: returns result whether condition evaluation is
	 *         true,false, etc.
	 * 
	 * @throws com.novell.nxpe.NxpeException
	 */
	@Override
	public NxpeResult evaluate(NxpeInformationContext informationContext,
			NxpeResponseContext responseContext) throws NxpeException {

		String strLDAPUserDN = "";
		NxpeResult res = NxpeResult.ConditionFalse;

		if (debug) {
			System.out
					.println("PolicyConditionExtnTemplate : executing evaluate() of PolicyConditionExtnTemplate policy");
		}

		try {
			/**
			 * 
			 * USER NEEDS TO PUT THE CONDITON EVALUATION LOGIC HERE
			 * 
			 */

			/**
			 * sample logic : begins This is a simple sample logic that returns
			 * "True" when LDAP UserDN is not null or empty else returns false
			 * 
			 */
			strLDAPUserDN = getLDAPUserDN(informationContext);

			if (strLDAPUserDN != null && !strLDAPUserDN.isEmpty()) {
				if (debug)
					System.out
							.println("PolicyConditionExtnTemplate :strLDAPUserDN"
									+ strLDAPUserDN);
				res = NxpeResult.ConditionTrue;

			} else {
				res = NxpeResult.ConditionFalse;
			}
			/**
			 * sample logic: ends
			 */

		} catch (NxpeException ex) {
			if (debug) {
				ex.printStackTrace();
			}
			res = NxpeResult.ConditionFalse;

		}

		if (debug)
			System.out.println(" PolicyConditionExtnTemplate : Exiting...");

		return res;

	}

	/**
	 * Enable/Disable the Debug logging Optional:
	 * 
	 * @param configurationValues
	 *            :NxpeParameterList
	 * 
	 * @throws com.novell.nxpe.NxpeException
	 */
	private void setDebug(NxpeParameterList configurationValues)
			throws NxpeException {

		/**
		 * Retrieve the value of "Debug"( value = 91) parameter set for this
		 * policy via admin console
		 */
		NxpeParameter parameter = configurationValues.getParameter(EV_DEBUG);

		if (parameter != null) {
			this.debug = Boolean.parseBoolean(parameter.getValue());
			System.out.println("PolicyConditionExtnTemplate : " + DEBUG_NAME
					+ " : " + this.debug);
		}
	}

	/**
	 * This is a sample method to show how the values for configuration
	 * parameters can be retrieved at evaluation time from the
	 * NxpeInformationContext object that is passed by the policy engine. This
	 * method retrieves the LDAP UserDN as an example. Optional:
	 * 
	 * @param informationContext
	 *            : used to retrieve the values of configuration parameter
	 * @return String : value of the LDAP UserDN
	 * 
	 * @throws com.novell.nxpe.NxpeException
	 */
	private String getLDAPUserDN(NxpeInformationContext informationContext)
			throws NxpeException {
		NxpeParameter pLDAPUserDN;

		if ((pLDAPUserDN = this.configurationValues
				.getParameter(EV_LDAP_USER_DN)) != null) {
			/**
			 * Retrieve value of LDAP USER DN using informationContext NOTE: If
			 * your data is multi valued, then assign to an array
			 */
			String strLDAPUserDN = (String) informationContext
					.getData(pLDAPUserDN);

			if (this.debug) {
				System.out.println("PolicyConditionExtnTemplate : "
						+ LDAP_USER_DN_NAME + " : " + strLDAPUserDN);
			}

			return strLDAPUserDN;
		} else {
			/**
			 * LDAP UserDN configuration parameter is not set for the policy.
			 * So, couldn't retrieve it's value
			 */
			NxpeException nxpeException = new NxpeException(
					NxpeResult.ErrorDataUnavailable,
					"PolicyConditionExtnTemplate :" + LDAP_USER_DN_NAME
							+ ": not present.");
			throw (nxpeException);
		}

	}

}
