/*
  Copyright (c) 2012 Novell, Inc. All Rights Reserved.

  Novell grants permission, free of charge, to any person obtaining copies
  of this software and its associated documentation files (the "Software"),
  to deal in the Software without restriction, including to use, copy, adapt, 
  publish, distribute, display, perform, sublicense, and sell copies of the 
  Software, subject to the following condition: You must include the above 
  copyright notice and this permission notice in all full or partial copies 
  of the Software.

  NOVELL PROVIDES THE SOFTWARE "AS IS," WITHOUT ANY EXPRESS OR IMPLIED WARRANTY,
  INCLUDING WITHOUT THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
  PARTICULAR PURPOSE, AND NON-INFRINGMENT.  NOVELL, THE AUTHORS OF THE SOFTWARE,
  AND THE OWNERS OF COPYRIGHT IN THE SOFTWARE ARE NOT LIABLE FOR ANY CLAIM, DAMAGES,
  OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT, OR OTHERWISE, ARISING
  FROM, OUT OF, OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/

package com.novell.nam.custom.policy.data;

import com.novell.nxpe.NxpeContextDataElement;
import com.novell.nxpe.NxpeException;
import com.novell.nxpe.NxpeInformationContext;
import com.novell.nxpe.NxpeParameter;
import com.novell.nxpe.NxpeParameterList;
import com.novell.nxpe.NxpeResponseContext;
import com.novell.nxpe.NxpeResult;

/**
 * This is an example code for a "Data" type of Policy Extension. getValue() is
 * the method which has to be modified by the user to code the data logic. Most
 * of other code can be retained as it is.
 */
public class PolicyDataExtnTemplate implements NxpeContextDataElement {

	/**
	 * ----BEGIN---- Initializing the constants for parameters passed to this
	 * class Configuration parameter can be configured from the administrator
	 * console UI. Go to Policies Tab->Extensions-><class extention>
	 * ->Configuration Parameters:->New
	 */

	/**
	 * Name parameter is used for reference in the code, can be used for
	 * logging. EV(Enumeration Value) will be used for fetching the parameter.
	 * This number must match the number configured at UI as ID parameter
	 */

	/** This parameter is Credential Profile:LDAP Credentials:LDAP User DN */
	private static final String LDAP_USER_DN_NAME = "LDAP User DN";
	private static final int EV_LDAP_USER_DN = 41;

	/**
	 * This parameter is a string constant set in policy by the admin to turn
	 * debug on/off
	 */
	private static final String DEBUG_NAME = "Debug";
	private static final int EV_DEBUG = 91;

	/** NxpeContextDataElement values */
	private final String strName;
	private final int iEnumerativeValue;
	private final String strParameter;

	/** Variable used to turn on/off the logging for debugging */
	private boolean debug;

	/** NxpeParameterList will be initialized during initialize() call */
	private NxpeParameterList configurationValues;

	/**
	 * Used for printing whether we are using single instance of this class or
	 * multiple instance of this class.
	 */
	private boolean singleInstance = false;

	/** Constructor. Called by PolicyDataExtTemplateFactory class */
	public PolicyDataExtnTemplate(String name, int enumerationValue,
			String parameter) {
		this.strName = name;
		this.iEnumerativeValue = enumerationValue;
		this.strParameter = parameter;
	}

	/**
	 * Initialize the configuration values
	 * 
	 * @required
	 * @param configurationValues
	 *            :NxpeParameterList: The configuration parameters passed by
	 *            policy engine are used to initialize the
	 *            NxpeContextDataElement object and are the parameters that the
	 *            extension requires to retrieve data.
	 * 
	 * @throws com.novell.nxpe.NxpeException
	 */
	@Override
	public void initialize(NxpeParameterList configurationValues)
			throws NxpeException {
		this.configurationValues = configurationValues;
		this.debug = true;
		setDebug(configurationValues);
	}

	@Override
	public int getEnumerativeValue() {
		return this.iEnumerativeValue;
	}

	@Override
	public String getName() {
		return this.strName;
	}

	@Override
	public String getParameter() {
		return this.strParameter;
	}

	/**
	 * Required: getValue() is the method which has to be modified by user to
	 * code the logic required.This method is called by the policy engine when
	 * the data extension needs to be evaluated for a policy.
	 * 
	 * @param informationContext
	 *            :The informationContext parameter contains the parameter
	 *            information the extension needs from the policy engine for the
	 *            required data related logic.
	 * @param responseContext
	 *            :The responseCtx is a reflection object for communicating
	 *            detailed response information back to the application. This is
	 *            additional information and does not replace the need to place
	 *            an result in the return value from this call.
	 * 
	 * @return Object: returns result Object
	 * 
	 * @throws com.novell.nxpe.NxpeException
	 */
	@Override
	public Object getValue(NxpeInformationContext informationContext,
			NxpeResponseContext responseContext) throws NxpeException {

		Object res = " Default";
		if (debug)
			System.out
					.println("PolicyDataExtnTemplate : executing getValue() of PolicyDataExtnTemplate policy");

		try {
			/**
			 * 
			 * USER NEEDS TO PUT THE DATA Extension EVALUATION LOGIC HERE
			 * 
			 */

			/**
			 * sample logic : begins This is a simple sample logic that returns
			 * "A" if LDAP UserDN is not empty or null otherwise returns "B"
			 */
			String strLDAPUserDN = "";
			String workerType;
			strLDAPUserDN = getLDAPUserDN(informationContext);

			if (strLDAPUserDN != null && !strLDAPUserDN.isEmpty()) {
				if (debug)
					System.out.println("PolicyDataExtnTemplate :strLDAPUserDN"
							+ strLDAPUserDN);
				res = "A";
			} else {
				res = "B";
				// Some error handling can be done here
			}
			/**
			 * sample logic: ends
			 */

		} catch (NxpeException ex) {
			if (debug) {
				ex.printStackTrace();
			}
			res = "Default";

		}

		if (debug)
			System.out.println("PolicyDataExtnTemplate : Exiting...");

		return res;

	}

	/**
	 * Enable/Disable the Debug logging Optional:
	 * 
	 * @param configurationValues
	 *            :NxpeParameterList
	 * 
	 * @throws com.novell.nxpe.NxpeException
	 */
	private void setDebug(NxpeParameterList configurationValues)
			throws NxpeException {

		/**
		 * Retrieve the value of "Debug"( value = 91) parameter set for this
		 * policy via admin console
		 */
		NxpeParameter parameter = configurationValues.getParameter(EV_DEBUG);

		if (parameter != null) {
			this.debug = Boolean.parseBoolean(parameter.getValue());
			System.out.println("PolicyDataExtnTemplate : " + DEBUG_NAME + " : "
					+ this.debug);
		}
	}

	/**
	 * This is a sample method to show how the values for configuration
	 * parameters can be retrieved at evaluation time from the
	 * NxpeInformationContext object that is passed by the policy engine. This
	 * method retrieves the LDAP UserDN as an example. Optional:
	 * 
	 * @param informationContext
	 *            : used to retrieve the values of configuration parameter
	 * @return String : value of the LDAP UserDN
	 * 
	 * @throws com.novell.nxpe.NxpeException
	 */
	private String getLDAPUserDN(NxpeInformationContext informationContext)
			throws NxpeException {
		NxpeParameter pLDAPUserDN;

		if ((pLDAPUserDN = this.configurationValues
				.getParameter(EV_LDAP_USER_DN)) != null) {
			/**
			 * Retrieve value of LDAP USER DN using informationContext NOTE: If
			 * your data is multi valued, then assign to an array
			 */
			String strLDAPUserDN = (String) informationContext
					.getData(pLDAPUserDN);

			if (this.debug) {
				System.out.println("PolicyDataExtnTemplate : "
						+ LDAP_USER_DN_NAME + " : " + strLDAPUserDN);
			}

			return strLDAPUserDN;
		} else {
			/**
			 * LDAP UserDN configuration parameter is not set for the policy.
			 * So, couldn't retrieve it's value
			 */
			NxpeException nxpeException = new NxpeException(
					NxpeResult.ErrorDataUnavailable, "PolicyDataExtnTemplate :"
							+ LDAP_USER_DN_NAME + ": not present.");

			throw (nxpeException);
		}

	}
}
