/*
  Copyright (c) 2007 Novell, Inc. All Rights Reserved.

  Novell grants permission, free of charge, to any person obtaining copies
  of this software and its associated documentation files (the "Software"),
  to deal in the Software without restriction, including to use, copy, adapt, 
  publish, distribute, display, perform, sublicense, and sell copies of the 
  Software, subject to the following condition: You must include the above 
  copyright notice and this permission notice in all full or partial copies 
  of the Software.

  NOVELL PROVIDES THE SOFTWARE "AS IS," WITHOUT ANY EXPRESS OR IMPLIED WARRANTY,
  INCLUDING WITHOUT THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A 
  PARTICULAR PURPOSE, AND NON-INFRINGMENT.  NOVELL, THE AUTHORS OF THE SOFTWARE,
  AND THE OWNERS OF COPYRIGHT IN THE SOFTWARE ARE NOT LIABLE FOR ANY CLAIM, DAMAGES,
  OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT, OR OTHERWISE, ARISING
  FROM, OUT OF, OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/

package com.novell.nxpe;


/**
 * Defines an interface you need to extend. It contains the methods required
 * to evaluate the condition for a policy.
 * @author Paladin Team <nxpe-dev@forge.provo.novell.com>
 * @since NXPE1.0
 */
public interface NxpeCondition
{
    /**
     * Called by the policy engine to initialize the condition in preparation
     * for policy evaluation.
     *
     * <p>Derived classes are required to override this method.
     *     This method is guaranteed to be called before any other method is
     *     called, as this is part of object construction.</p>
     *
     * @param configurationValues A list of configuration data required by the
     *      external condition handler. This object is only valid during the
     *      lifetime of the initialize method.  If you want to preserve
     *      config data, it must be copied to the condition handler.
     *
     * @throws NxpeException
     */
    void initialize(
            NxpeParameterList configurationValues)
        throws NxpeException;

    /**
     * Called by the policy engine to evaluate the condition.
     * 
     * <p>Derived classes are require to override this method to
     * implement the supported condition.</p>
     * 
     * @param informationContext The parameter information the extension needs 
     *      from the policy engine to evaluate the condition.
     * @param responseContext The object for communicating detailed response 
     *      information back to the application. This does not replace the need 
     *      to notify the policy engine of the condition completion status 
     *      via the return value from this call.
     * @return The results of the evaluation.
     * @throws NxpeException 
     */
    NxpeResult evaluate(
			NxpeInformationContext informationContext, 
			NxpeResponseContext responseContext)
		throws NxpeException;

    /**
     * Sets the unique string value for the condition.
     *
     * <p>This is currently the InterfaceId found in the condition
     *     element. This data is used for tracing evaluation. </p>
     *
     * @param interfaceId The unique string value for the condition.
     *
     * @throws NxpeException
     */
    void setInterfaceId(
            String interfaceId)
        throws NxpeException;

} /* NxpeCondition */

