
/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/nds/NdsTypedName.java $
  $Revision: 9 $
  $Modtime: 11/15/00 3:43p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import java.util.Enumeration;
import java.util.Vector;

import com.novell.service.schema.SchemaValue;
import com.novell.service.schema.SchemaComposite;


/**
 * Provides access to attribute values of the TypedName
 * syntax. It is used for attributes whose values represent
 * a level and an interval associated with an object. This
 * syntax names an NDS object and attaches two numeric values
 * to it:
 * <ol><li>The level of the attribute indicates the priority.
 * <li>The interval indicates the frequency of reference.
 * </ol>
 * <p>The objectName or Distinguished Name identifies the NDS
 * object referred to by the Typed Name. The values of level
 * and interval are user-assigned and relative. To be effective
 * they must be implemented by the user. The user can use them
 * to implement iterative intervals or to enforce priorities.
 * <p>In comparisons and searches using equality, all three
 * parts of the field must be equal for attributes to be equal.
 * 
 * <p>Matching Rules: Equality</p>
 * 
 * @see NdsSyntaxId
 */
public class NdsTypedName 
   implements NdsAttributeValue, SchemaComposite, Cloneable, Serializable
{
   private static final int matchingRules = equality;

   private static String[] compositeIds;

   static {
      compositeIds = new String[3];
      compositeIds[0] = "objectName";
      compositeIds[1] = "level";
      compositeIds[2] = "interval";
   }

  /**
   * @internal
   */
   protected String objectName;

  /**
   * @internal
   */
   protected int level;

  /**
   * @internal
   */
   protected int interval;

   private String name;

   /**
   * Constructs an NdsTypedName object based on the specified
   * objectName, level, and interval parameters. The level
   * of the attribute indicates the priority, and the interval
   * represents the number of seconds between certain events or
   * the frequency of reference.
   * 
   * @param objectName The object name stored as a String value.
   * @param level      The level of the attribute indicating the
   *                   priority stored as an long value.
   * @param interval   The interval indicating the frequency of
   *                   reference stored as an long value.
   */
   public NdsTypedName (
         String objectName,
         long level,
         long interval)
   {
      this.objectName = objectName;
      this.level = (int) (level & INT_MASK);
      this.interval = (int) (interval & INT_MASK);
   }

  /**
   * @internal
   */
   protected NdsTypedName ()
   {
      this.level = -1;
      this.interval = -1;
   }

   /*
    * Private constructor; not documented!
    */
   private NdsTypedName (
         String name,
         String objectName,
         int level,
         int interval)
   {
      this.name = name;
      this.objectName = objectName;
      this.level = level;
      this.interval = interval;
   }

  /**
   * @internal
   */
   protected NdsTypedName (
         NdsTypedName aTypedName)
   {
      this.objectName = aTypedName.objectName;
      this.level = aTypedName.level;
      this.interval = aTypedName.interval;
      this.name = aTypedName.name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically.
   *
   * The compareTo method does not apply to the TypedName
   * syntax because this syntax does not support the ordering and
   * substrings matching rules.
   */
   public int compareTo (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals (
         Object anObject)
   {
      if ((anObject != null) && (anObject instanceof NdsTypedName))
      {
         NdsTypedName anotherTypedName = (NdsTypedName) anObject;

         if (false == objectName.equals (anotherTypedName.objectName))
         {
            return (false);
         }
         if (level != anotherTypedName.level)
         {
            return (false);
         }
         if (interval != anotherTypedName.interval)
         {
            return (false);
         }
         return (true);
      }
      return (false);
   }

  /**
   * @internal
   */
   public String getName ()
   {
      return (name);
   }

  /**
   * @internal
   */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.TYPED_NAME);
   }


   // ******************** SchemaComposite Interface ********************

   /**
    * Returns the number of elements in this sequence.
    *
    * @return The number of elements in this sequence as an int.
    */
   public int count ()
   {
      return (compositeIds.length);
   }

   /**
   * Returns a SchemaValue object based on the given passed-in
   * string containing the element value. The SchemaValue object
   * is a Java language data type.
   *
   * @param compositeId The passed-in string containing the element
   *                    value.
   *
   * @return The SchemaValue object containing the objectName,
   *         level, and interval, or NULL.
   */
   public SchemaValue getValue (
         String compositeId)
   {
      if (compositeId.equalsIgnoreCase (compositeIds[0]))
      {
         // getObjectName ();
         return (new NdsString (compositeIds[0], objectName));
      }
      else if (compositeId.equalsIgnoreCase (compositeIds[1]))
      {
         // getLevel ();
         return (new NdsInteger (compositeIds[1], level));
      }
      else if (compositeId.equalsIgnoreCase (compositeIds[2]))
      {
         // getInterval ();
         return (new NdsInteger (compositeIds[2], interval));
      }
      return (null);
   }

  /**
   * Returns an enumeration of composit IDs.
   *
   * @return An enumeration of composite IDs.
   */
   public Enumeration getValueIds ()
   {
      Vector vector = new Vector ();

      for (int i = 0; i < compositeIds.length; i++)
         vector.addElement (compositeIds[i]);
      return (vector.elements ());
   }

   /**
    * Returns an enumeration of the elements in this composite.
    *
    * @return An enumeration of the composite elements
    *         (objectName, level and interval).
    */
   public Enumeration getValues ()
   {
      Vector values = new Vector ();

      values.addElement (new NdsString (compositeIds[0], objectName));
      values.addElement (new NdsInteger (compositeIds[1], level));
      values.addElement (new NdsInteger (compositeIds[2], interval));
      return (values.elements ());
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Compares two Objects using the approximate matching rule.
   * 
   * The approximate method does not apply to the Typed Name
   * syntax because this syntax does not support the approximate
   * equals matching rule.
   */
   public boolean approximate (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Returns the int that represents the NDS syntax ID.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.TYPED_NAME_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules. The Typed Name syntax supports only the
   * equality matching rule.
   *
   * @param matchingRules The set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
         return (true);
      return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone ()
   {
      try
      {
         NdsTypedName typedName = (NdsTypedName) super.clone ();

         typedName.objectName = this.objectName;
         typedName.level = this.level;
         typedName.interval = this.interval;

         typedName.name = this.name;

         return (typedName);
      }
      catch (CloneNotSupportedException e)
      {
         // this shouldn't happen, since we are Cloneable
         throw (new InternalError ());
      }

   } /* clone () */

  /**
   * Generates a string representation of the object. It
   * returns a string that textually represents the object.
   * The result should be a concise but informative
   * representation that is easy to read.
   *
   * @return The String representation of the object.
   */
   public String toString ()
   {
      return (
         objectName + ";" +
         String.valueOf (level) + ";" +
         String.valueOf (interval));
   }


   // ******************** NdsTypedName Class ********************

  /**
   * Returns the object name stored as a String value in the
   * object data member.
   * 
   * @return The object name as a String value.
   */
   public String getObjectName ()
   {
      return (objectName);
   }

  /**
   * Returns the attribute level stored as a long value in the
   * object data member.
   * 
   * @return The attribute level as a long value.
   */
   public long getLevel ()
   {
      return (level & INT_MASK);
   }

  /**
   * Returns the interval stored as a long value in the
   * object data member.
   * 
   * @return The interval as a long value.
   */
   public long getInterval ()
   {
      return (interval & INT_MASK);
   }

} /* NdsTypedName */


