/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/server/ServerDiagnostics.java $
  $Revision: 12 $
  $Modtime: 1/28/00 12:46p $

  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/
package com.novell.service.server;

import com.novell.service.jncp.NSIException;

/**
 * Provides a specific attribute of a ServerDirContext. The ID of
 * the attribute is equal to ATTR_ID, and the value is an object
 * that implements this interface.
 *
 * @see ServerDiagnosticsImpl
 */
public interface ServerDiagnostics
{
  /**
   *  The attribute ID of this attribute.
   *
   *  <p>(ATTR_ID = "ServerDiagnostics")
   */
   public final static String ATTR_ID = "ServerDiagnostics";

   /**
   *  @deprecated
   *  This member will be removed. Use ATTR_ID instead.
   */
   public final static String attrID = ATTR_ID;

  /**
   * Returns the time in ticks (about 1/18 second) since the server
   * was started. When ServerUpTime reaches 0xFFFFFFFF, it wraps
   * to 0.
   *
   * @return The time since the server was started.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getServerUpTime() throws NSIException;

  /**
   * Returns the current percentage of CPU utilization for the server.
   *
   * @return Current CPU utilization.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getCurrentCPUUtilization() throws NSIException;

  /**
   * Returns the number of replies that were canceled because the
   * connection was reallocated while the request was being processed.
   *
   * @return The number of canceled replies.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getReplyCanceledCount() throws NSIException;

  /**
   * Returns the number of times that writes were delayed because of
   * a pending TTS(tm) transaction or cache busy condition.
   *
   * @return The number of times write delays occurred.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getWriteHeldOffCount() throws NSIException;

  /**
   * Returns the number of times that writes were canceled since a
   * duplicate request was received.
   *
   * @return The number of times writes were canceled.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getWriteHeldOffWithDupRequest() throws NSIException;

  /**
   * Returns the number of packets received which had an invalid request
   * type or were received after the server was shut down.
   *
   * @return The number of invalid request type packets received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getInvalidRequestTypeCount() throws NSIException;

  /**
   * Returns the number of packets received for a connection being
   * terminated.
   *
   * @return The number of packets received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getBeingAbortedCount() throws NSIException;

  /**
   * Returns the number of times a connection is requested when
   * a connection already exists.
   *
   * @return The number of times a connection is requested.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getAlreadyDoingReallocCount() throws NSIException;

  /**
   * Returns the number of times an attempt was made to deallocate a
   * connection slot that was not valid.
   *
   * @return The number of attempts made to deallocate a connection
   *         slot.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getDeAllocInvalidSlotCount() throws NSIException;

  /**
   * Returns the number of times the server was deallocated because
   * requests were still being processed.
   *
   * @return The number of times the server was deallocated.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getDeAllocBeingProcessedCount() throws NSIException;

  /**
   * Returns the number of times the server was deallocated because a
   * forged packet was received.
   *
   * @return The number of times the server was deallocated.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getDeAllocForgedPacketCount() throws NSIException;

  /**
   * Returns the number of times the server was deallocated because
   * information was still being transmitted.
   *
   * @return The number of times the server was deallocated.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getDeAllocStillTransmittingCount() throws NSIException;

  /**
   * Returns the number of times the server was unable to allocate
   * a connection for any reason.
   *
   * @return The number of times the server was unable to allocate
   *         a connection.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getStartStationErrorCount() throws NSIException;

  /**
   * Returns the number of requests received for an invalid connection
   * slot.
   *
   * @return The number of requests received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getInvalidSlotCount() throws NSIException;

  /**
   * Returns the number of times a duplicate request was received
   * during processing of the first request.
   *
   * @return The number of times a duplicate request was received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getBeingProcessedCount() throws NSIException;

  /**
   * Returns the number of suspicious invalid packets received.
   *
   * @return The number of suspicious invalid packets received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getForgedPacketCount() throws NSIException;

  /**
   * Returns the number of times a new request is received before
   * a reply to a previous request has been sent.
   *
   * @return The number of times a new request is received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getStillTransmittingCount() throws NSIException;

  /**
   * Returns the number of times the requester did not receive the reply
   * and the request had to be reprocessed.
   *
   * @return The number of times the requester did not receive the reply.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getReExecuteRequestCount() throws NSIException;

  /**
   * Returns the number of request packets the server received from
   * a connection where the sequence number in the packet did not
   * match the current sequence number or the next sequence number.
   *
   * @return The number of invalid sequence number packets.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getInvalidSequenceNumCount() throws NSIException;

  /**
   * Returns the number of times a duplicate reply was requested when
   * the reply had already been sent.
   *
   * @return The number of times a duplicate reply was requested.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getDuplicateIsBeingSentAlreadyCount() throws NSIException;

  /**
   * Returns the number of acknowledgments sent by the server (sent
   * when a connection repeats a request being serviced).
   *
   * @return The number of acknowledgments sent by the server.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getSentPositiveAcknowledgeCount() throws NSIException;

  /**
   * Returns the number of request packets for which the server had
   * to send a duplicate reply (sent only for requests the server
   * cannot process).
   *
   * @return The number of request packets for which the server had
   * to send a duplicate reply.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getSentDuplicateReplyCount() throws NSIException;

  /**
   * Returns the number of times the server could not allocate
   * memory to expand the connection table for a new connection.
   *
   * @return The number of times the server could not allocate
   *         memory.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getNoMemForStationCtrlCount() throws NSIException;

  /**
   * Returns the number of times no slots were available in the
   * connection table for a new connection.
   *
   * @return The number of times no slots were available.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getNoAvailableConnsCount() throws NSIException;

  /**
   * Returns the number of times the server reallocated the same
   * slot in the connection table for a client that logged out and
   * relogged in.
   *
   * @return The number of times the server reallocated the same
   * slot in the connection table.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getReallocSlotCount() throws NSIException;

  /**
   * Returns the number of times that a request came from a client
   * to relog in before that client had been completely logged out.
   *
   * @return The number of times that a request came from a client
   * to relog in.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getReallocSlotCameTooSoonCount() throws NSIException;

  /**
   * Returns the number of packets discarded because they had passed
   * through more than 16 bridges without reaching their destination.
   *
   * @return The number of packets discarded.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public short getTooManyHops() throws NSIException;

  /**
   * Returns the number of packets discarded because their destination
   * network was unknown to the server.
   *
   * @return The number of packets discarded.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public short getUnknownNetwork() throws NSIException;

  /**
   * Reserved (pass 0).
   *
   * @return Zero (0).
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public short getNoSpaceForService() throws NSIException;

  /**
   * Returns the number of times a packet was discarded because no
   * buffers existed to receive it.
   *
   * @return The number of times a packet was discarded.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public short getNoReceiveBuffers() throws NSIException;

  /**
   * Returns the number of received packets not destined for
   * the server.
   *
   * @return The number of received packets.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public short getNotMyNetwork() throws NSIException;

  /**
   * Returns the number of NetBIOS packets received that were
   * sent forward.
   *
   * @return The number of NetBIOS packets received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getNetBIOSProgatedCount() throws NSIException;

  /**
   * Returns the total packets received by the server.
   *
   * @return The total packets received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getTotalPacketsServiced() throws NSIException;

  /**
   * Returns the number of all packets forwarded by the server.
   *
   * @return The number of all packets forwarded.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public int getTotalPacketsRouted() throws NSIException;

  /**
   * Sets as the search value the time in ticks (about 1/18 second)
   * since the server was started. This is used to search for
   * all servers matching the upTime parameter.
   *
   * @param upTime The time since the server was started.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setServerUpTime(
         int upTime)
      throws NSIException;

  /**
   * Sets as the search value the current percentage of CPU
   * utilization for the server. This is used to search for
   * all servers matching the upTime parameter.
   *
   * @param utilization The current percentage of CPU utilization
   *                    for the server.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setCurrentCPUUtilization(
         int utilization)
      throws NSIException;

  /**
   * Sets as the search value the number of replies that were
   * canceled because the connection was reallocated while the
   * request was being processed. This is used to search for
   * all servers matching the value parameter.
   *
   * @param value The number of canceled replies.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setReplyCanceledCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times that writes were
   * delayed because of a pending TTS(tm) transaction or cache busy
   * condition. This is used to search for all servers matching the
   * value parameter.
   *
   * @param value the number of times that writes were delayed.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setWriteHeldOffCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times that writes were
   * canceled since a duplicate request was received. This is used
   * to search for all servers matching the value parameter.
   *
   * @param value The number of times that writes were canceled.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setWriteHeldOffWithDupRequest(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of packets received that
   * had an invalid request type or were received after the server
   * was shut down. This is used to search for all servers matching
   * the value parameter.
   *
   * @param value The number of packets received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setInvalidRequestTypeCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of packets received for
   * a connection being terminated. This is used to search for all
   * servers matching the value parameter.
   *
   * @param value The number of packets received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setBeingAbortedCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times that a connection
   * is requested when a connection already exists. This is used to
   * search for all servers matching the value parameter.
   *
   * @param value The number of times a connection is requested.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setAlreadyDoingReallocCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times an attempt was
   * made to deallocate a connection slot that was not valid. This
   * is used to search for all servers matching the value parameter.
   *
   * @param value The number of times an attempt was made to
   *              deallocate a connection slot.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setDeAllocInvalidSlotCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times the server was
   * deallocated because requests were still being processed. This
   * is used to search for all servers matching the value parameter.
   *
   * @param value The number of times the server was deallocated.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setDeAllocBeingProcessedCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times the server was
   * deallocated because a forged packet was received. This is used
   * to search for all servers matching the value parameter.
   *
   * @param value The number of times the server was deallocated.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setDeAllocForgedPacketCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times the server was
   * deallocated because information was still being transmitted.
   * This is used to search for all servers matching the value
   * parameter.
   *
   * @param value The number of times the server was deallocated.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setDeAllocStillTransmittingCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times the server was
   * unable to allocate a connection for any reason. This is used
   * to search for all servers matching the value parameter.
   *
   * @param value The number of times the server was unable to
   *              allocate a connection.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setStartStationErrorCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of requests received
   * for an invalid connection slot. This is used to search for
   * all servers matching the value parameter.
   *
   * @param value The number of requests received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setInvalidSlotCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times a duplicate request
   * was received during processing of the first request. This is used
   * to search for all servers matching the value parameter.
   *
   * @param value The number of times a duplicate request was received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setBeingProcessedCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of suspicious invalid
   * packets received. This is used to search for all servers
   * matching the value parameter.
   *
   * @param value The number of suspicious invalid packets received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setForgedPacketCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times a new request is
   * received before a reply to a previous request has been sent.
   * This is used to search for all servers matching the value parameter.
   *
   * @param value The number of times a new request is received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setStillTransmittingCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times the requester did
   * not receive the reply and the request had to be reprocessed.
   * This is used to search for all servers matching the value parameter.
   *
   * @param value The number of times the requester did not receive
   *              the reply.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setReExecuteRequestCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of request packets the server
   * received from a connection where the sequence number in the packet
   * did not match the current or the next sequence number. This is
   * used to search for all servers matching the value parameter.
   *
   * @param value The number of request packets the server received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setInvalidSequenceNumCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times a duplicate reply
   * was requested when the reply had already been sent. This is
   * used to search for all servers matching the value parameter.
   *
   * @param value The number of times a duplicate reply was
   *              requested.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setDuplicateIsBeingSentAlreadyCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of acknowledgments sent
   * by the server (sent when a connection repeats a request being
   * serviced). This is used to search for all servers matching
   * the value parameter.
   *
   * @param value The number of acknowledgments sent by the server.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setSentPositiveAcknowledgeCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of request packets for
   * which the server had to send a duplicate reply (sent only
   * for requests the server cannot process). This is used to
   * search for all servers matching the value parameter.
   *
   * @param value The number of request packets.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setSentDuplicateReplyCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times the server could
   * not allocate memory to expand the connection table for a new
   * connection. This is used to search for all servers matching
   * the value parameter.
   *
   * @param value The number of times the server could not allocate
   *              memory.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setNoMemForStationCtrlCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times no slots were
   * available in the connection table for a new connection. This
   * is used to search for all servers matching the value parameter.
   *
   * @param value The number of times no slots were available in the
   *              connection table for a new connection.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setNoAvailableConnsCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times the server
   * reallocated the same slot in the connection table for a client
   * that logged out and relogged in. This is used to search for
   * all servers matching the value parameter.
   *
   * @param value The number of times the server reallocated the
   *              same slot.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setReallocSlotCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of times that a request
   * came from a client to relog in before that client had been
   * completely logged out. This is used to search for all servers
   * matching the value parameter.
   *
   * @param value The number of times that a request came from a
   *              client to relog in.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setReallocSlotCameTooSoonCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of packets discarded because
   * they had passed through more than 16 bridges without reaching
   * their destination. This is used to search for all servers
   * matching the value parameter.
   *
   * @param value The number of packets discarded.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setTooManyHops(
         short value)
      throws NSIException;

  /**
   * Sets as the search value the number of packets discarded
   * because their destination network was unknown to the server.
   * This is used to search for all servers matching the value
   * parameter.
   *
   * @param value The number of packets discarded.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setUnknownNetwork(
         short value)
      throws NSIException;

  /**
   * Sets as the search value zero (0)because it is reserved.
   *
   * @param value Zero because it is reserved.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setNoSpaceForService(
         short value)
      throws NSIException;

  /**
   * Sets as the search value the number of times a packet was
   * discarded because no buffers existed to receive it. This is
   * used to search for all servers matching the value parameter.
   *
   * @param value The number of times a packet was discarded.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setNoReceiveBuffers(
         short value)
      throws NSIException;

  /**
   * Sets as the search value the number of received packets not
   * destined for the server. This is used to search for all
   * servers matching the value parameter.
   *
   * @param value The number of received packets.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setNotMyNetwork(
         short value)
      throws NSIException;

  /**
   * Sets as the search value the number of NetBIOS packets
   * received that were sent forward. This is used to search for
   * all servers matching the value parameter.
   *
   * @param value The number of NetBIOS packets received.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setNetBIOSProgatedCount(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the total packets received by the
   * server. This is used to search for all servers matching the
   * value parameter.
   *
   * @param value The total packets received by the server.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setTotalPacketsServiced(
         int value)
      throws NSIException;

  /**
   * Sets as the search value the number of all packets forwarded
   * by the server. This is used to search for all servers matching
   * the value parameter.
   *
   * @param value The number of all packets forwarded by the server.
   *
   * @exception NSIException  When an error occurs retrieving the
   *                          information.
   */
   public void setTotalPacketsRouted(
         int value)
      throws NSIException;
}
