
/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/nds/NdsAttributeRights.java $
  $Revision: 13 $
  $Modtime: 11/15/00 3:43p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;

import javax.naming.NamingException;
import javax.naming.directory.*;

import com.novell.service.jncp.NSIException;

/**
 * Provides definitions of the effective rights for attributes,
 * and methods for checking attribute effective rights.
 *
 * @see NdsRights
 */
public final class NdsAttributeRights extends NdsRights
{
   private String attrName = null;

  /**
   * Allows a trustee to compare a value with an attribute's
   * value.
   *
   * <p>This allows the trustee to see if the attribute 
   * contains the value without having rights to see what
   * the value is.
   *
   * <p>(DS_ATTR_COMPARE     = 0x1)
   */   
   public static final int DS_ATTR_COMPARE     = 0x1;

  /**
   * Allows a trustee to read an attribute.
   * 
   * <p>This right confers the Compare right as well as the Read
   * right.
   *
   * <p>(DS_ATTR_READ     = 0x2)
   */   
   public static final int DS_ATTR_READ        = 0x2;

  /**
   * Allows a trustee to add, delete, or modify an attribute
   * value.
   * 
   * <p>This right also gives the trustee the Self (Add or
   * Delete Self) right.
   *
   * <p>(DS_ATTR_WRITE     = 0x4)
   */   
   public static final int DS_ATTR_WRITE       = 0x4;

  /**
   * Allows a trustee to add or delete its name as an
   * attribute value on those attributes that take
   * object names as their values.
   *
   * <p>(DS_ATTR_SELF     = 0x8)
   */   
   public static final int DS_ATTR_SELF        = 0x8;

  /**
   * Gives a trustee all rights to the object's attributes.
   *
   * <p>(DS_ATTR_SUPERVISOR     = 0x20)
   */   
   public static final int DS_ATTR_SUPERVISOR  = 0x20;

  /**
   * Allows a trustee to inherit the rights granted in
   * the ACL and exercise these attribute rights on
   * subordinate objects.
   *
   * <p>(DS_ATTR_INHERIT_CTL     = 0x40)
   */   
   public static final int DS_ATTR_INHERIT_CTL = 0x40;

  /**
   * Constructs an NdsAttributeRights object and sets the 
   * attribute effective rights.
   *
   * @param rights The attribute effective rights as
   *               defined by the above variables.
   */
   public NdsAttributeRights (
         int rights)
   {
		this.rights = rights;
   }
           
  /**
   * Constructs an NdsAttributeRights object and sets the 
   * attribute name and associated effective rights.
   *
   * @param attr The Attribute for which to get the
   *             attribute effective rights.
   * 
   * @exception NSIException Standard NJCL exception for working on
   *                         top of XPlat libraries.
   */
   public NdsAttributeRights (
              Attribute attr)
        throws NSIException
   {
		this.attrName = attr.getID ();
      try			  
      {
		   this.rights = (((NdsInteger)attr.get ()).intValue ());
      }
      catch (NamingException e)
      {
         throw new NSIException ();
      }
   }

  /**
   * Returns the name of the current attribute as a String.
   *
   * @return A string containing the name of the attribute.
   */
   public String getAttrName ()
   {
      return attrName;
   }
   
  /**
   * Checks to see if an attribute has Compare rights.
   *
   * <p>Calls getRights() for the current attribute and then
   * checks the Compare rights status.
   *
   * @return A Boolean set to TRUE if the attribute has Compare
   *         rights, otherwise set to FALSE.
   *
   * @see NdsRights#getRights()
   */
   public boolean hasCompare ()
   {
      return ((getRights() & DS_ATTR_COMPARE) != 0);
   }

  /**
   * Checks to see if an attribute has Read rights.
   *
   * <p>Calls getRights() for the current attribute and then
   * checks the Read rights status.
   *
   * @return A Boolean set to TRUE if the attribute has Read
   *         rights, otherwise set to FALSE.
   *
   * @see NdsRights#getRights()
   */
   public boolean hasRead ()
   {
      return ((getRights() & DS_ATTR_READ) != 0);
   }

  /**
   * Checks to see if an attribute has Write rights.
   *
   * <p>Calls getRights() for the current attribute and then
   * checks the Write rights status.
   *
   * @return A Boolean set to TRUE if the attribute has Write
   *         rights, otherwise set to FALSE.
   *
   * @see NdsRights#getRights()
   */
   public boolean hasWrite ()
   {
      return ((getRights() & DS_ATTR_WRITE) != 0);
   }

  /**
   * Checks to see if an attribute has Self rights.
   *
   * <p>Calls getRights() for the current attribute and then
   * checks the Self rights status.
   *
   * @return A Boolean set to TRUE if the attribute has Self
   *         rights, otherwise set to FALSE.
   *
   * @see NdsRights#getRights()
   */
   public boolean hasSelf ()
   {
      return ((getRights() & DS_ATTR_SELF) != 0);
   }

  /**
   * Checks to see if an attribute has Supervisor rights.
   *
   * <p>Calls getRights() for the current attribute and then
   * checks the Supervisor rights status.
   *
   * @return A Boolean set to TRUE if the attribute has Supervisor
   *         rights, otherwise set to FALSE.
   *
   * @see NdsRights#getRights()
   */
   public boolean hasSupervisor ()
   {
      return ((getRights() & DS_ATTR_SUPERVISOR) != 0);
   }

  /**
   * Checks to see if an attribute has Inherit CTL rights.
   *
   * <p>Calls getRights() for the current attribute and then
   * checks the Inherit CTL rights status.
   *
   * @return A Boolean set to TRUE if the attribute has Inherit CTL
   *         rights, otherwise set to FALSE.
   *
   * @see NdsRights#getRights()
   */
   public boolean hasInheritCtl ()
   {
      return ((getRights() & DS_ATTR_INHERIT_CTL) != 0);
   }

} /* NdsAttributeRights */


