
/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/nds/NdsCaseIgnoreList.java $
  $Revision: 16 $
  $Modtime: 11/15/00 3:43p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import java.util.Enumeration;
import java.util.Vector;

import com.novell.service.schema.SchemaSequence;

/**
 * Provides access to values of the Case Ignore List attribute.
 * It is used for attributes whose values are ordered sequences of
 * strings for which the case (upper or lower) is not significant
 * when performing comparisons.
 *
 * <p>Two Case Ignore Lists match for equality if and only if the
 * number of strings in each is the same, and all corresponding
 * strings match. For two corresponding strings in the list to
 * match they must be the same length and their corresponding
 * characters must be identical. In matching attributes that
 * conform to this syntax, NDS omits those spaces that are not
 * significant (leading spaces, trailing spaces, and multiple
 * consecutive internal spaces). The NDS stores insignificant
 * spaces with the attribute value.
 * 
 * <p>Matching Rules: Equality</p>
 *
 * @see NdsSyntaxId
 */
public class NdsCaseIgnoreList
   implements NdsAttributeValue, SchemaSequence, Cloneable, Serializable
{
   private static final int matchingRules = equality;

  /**
   * @internal
   */
  protected String[] values;

   private String name;


  /**
   * Constructs an NdsCaseIgnoreList object from a String array
   * of values. This array is copied to make the attribute
   * value immutable.
   *
   * <p>Two case ignore lists match for equality if and only if the
   * number of strings in each is the same, and all corresponding
   * strings match. For two corresponding strings in the list to match
   * they must be the same length and their corresponding characters
   * must be identical (according to the rules for case ignore strings).
   * 
   * @param value The String array of stored values.
   * 
   */
   public NdsCaseIgnoreList (
        String[] value)
   {
      this ("", value);
   }

  /**
   * @internal
   */
   protected NdsCaseIgnoreList ()
   {
      values = null;
      name = null;
   }

   /*
    * Private constructor; not documented!
    */
   private NdsCaseIgnoreList (
        String name, String[] value)
   {
      this.values = (String[]) value.clone ();
      this.name = name;
   }

  /**
   * @internal
   */
   protected NdsCaseIgnoreList (
         NdsCaseIgnoreList aCaseIgnoreList)
   {
      values = aCaseIgnoreList.values;
      name = aCaseIgnoreList.name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically.
   *
   * The compareTo method does not apply to the NdsCaseIgnoreList
   * syntax because this syntax does not support the ordering and
   * substrings matching rules.
   */
   public int compareTo (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals (
         Object anObject)
   {
      if ((anObject != null) && (anObject instanceof NdsCaseIgnoreList))
      {
         int length = values.length;
         NdsCaseIgnoreList anotherCIList = (NdsCaseIgnoreList) anObject;

         for (int i = 0; i < length; i++)
         {
            String pattern = values[i];
            String target = anotherCIList.values[i];

            if (false == pattern.equals (target))
            {
               return (false);
            }
         }
         return (true);
      }

      return (false);
   }

  /**
   * @internal
   */
   public String getName ()
   {
      return (name);
   }

  /**
   * @internal
   */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.CASE_IGNORE_LIST);
   }


   // ******************** SchemaSequence Interface ********************

  /**
   * Returns the number of elements in this sequence.
   *
   * @return The number of elements in this sequence as an int.
   */
   public int count ()
   {
      return (values.length);
   }

  /**
   * Returns an enumeration of element values in this sequence.
   *
   * @return An enumeration of element values.
   */
   public Enumeration getValues ()
   {
      Vector vector = new Vector ();

      for (int i = 0; i < values.length; i++)
      {
         vector.addElement (values[i]);
      }
      return (vector.elements ());
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Compares two Objects using the approximate matching rule.
   * 
   * The approximate method does not apply to the NdsCaseIgnoreList
   * syntax because this syntax does not support the approximate
   * equals matching rule.
   */
   public boolean approximate (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Returns the int that represents the NDS syntax ID.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.CASE_IGNORE_LIST_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules.
   *
   * <p>The NdsCaseIgnoreList syntax supports only the equality
   * matching rule.
   *
   * @param matchingRules The set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
         return (true);
      else
         return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone()
   {
      try
      {
         NdsCaseIgnoreList list = (NdsCaseIgnoreList) super.clone ();

         list.values = (String[]) this.values.clone ();
         list.name = this.name;

         return (list);
      }
      catch (CloneNotSupportedException e)
      {
         // this shouldn't happen, since we are Cloneable
         throw (new InternalError ());
      }

   } /* clone () */

  /**
   * Generates and returns a string representation of the object. In
   * general, the toString method returns a string that textually 
   * represents this object. The result should be a concise but
   * informative representation that is easy to read.
   *
   * @return A String representation of the object.
   */
   public String toString ()
   {
      return (values.toString());
   }


   // ******************** NdsCaseIgnoreList Class ********************

  /**
   * Returns the String array containing a list of strings stored
   * in the NDS attribute.
   *
   * @return The String array of stored attribute values.
   */
   public String[] getCaseIgnoreList ()
   {
      return ((String[]) values.clone ());
   }

} /* NdsCaseIgnoreList */
