
/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/nds/NdsFAXNumber.java $
  $Revision: 10 $
  $Modtime: 11/15/00 3:43p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import java.util.Enumeration;
import java.util.Vector;

import com.novell.service.schema.SchemaValue;
import com.novell.service.schema.SchemaComposite;


/**
 * Provides access to values of the FAX Number attribute.
 * It is used for attributes whose values are strings that
 * comply with the format for international telephone numbers,
 * E.123, and an optional bit string formatted according to
 * Recommendation T.30.
 *
 * <p>Fax telephone number values are matched based on the
 * telephone number field. The rules for matching fax telephone
 * numbers are identical to those for the Case Exact String
 * syntax except that all space and hyphen (-) characters are
 * skipped during the comparison.
 *
 * <p>Matching Rules: Equality</p>
 * 
 * @see NdsSyntaxId
 */
public class NdsFAXNumber
   implements NdsAttributeValue, SchemaComposite, Cloneable, Serializable
{
   private static final int matchingRules = equality;

   private static String[] compositeIds;

   static {
      compositeIds = new String[3];
      compositeIds[0] = "telephoneNumber";
      compositeIds[1] = "bitCount";
      compositeIds[2] = "parameters";
   }

  /**
   * @internal
   */
   protected String telephoneNumber;

  /**
   * @internal
   */
   protected int bitCount;

  /**
   * @internal
   */
   protected byte[] parameters;

   private String name;

   /**
   * Constructs an NdsFAXNumber object based on the specified
   * telephoneNumber, bitCount, and parameters variables.
   * 
   * @param telephoneNumber Telephone number to be stored as a String.
   * @param bitCount        Bit count to be stored as a long.
   * @param parameters      Byte array of parameters(bit string and its
   *                        length).
   */
   public NdsFAXNumber (
         String telephoneNumber,
         long bitCount,
         byte[] parameters)
   {
      this ("", telephoneNumber, bitCount, parameters);
   }

  /**
   * @internal
   */
   protected NdsFAXNumber ()
   {
      this.telephoneNumber = null;
      this.bitCount = -1;
      parameters = null;
   }

   private NdsFAXNumber (
         String name,
         String telephoneNumber,
         long bitCount,
         byte[] parameters)
   {
      int length = parameters.length;

      this.telephoneNumber = telephoneNumber;
      this.bitCount = (int) (bitCount & INT_MASK);
      this.parameters = new byte[length];
      System.arraycopy (parameters, 0, this.parameters, 0, length);
      this.name = name;
   }

  /**
   * @internal
   */
   protected NdsFAXNumber (
         NdsFAXNumber aFAXNumber)
   {
      int length = aFAXNumber.parameters.length;

      this.telephoneNumber = aFAXNumber.telephoneNumber;
      this.bitCount = aFAXNumber.bitCount;
      this.parameters = new byte[length];
      System.arraycopy (aFAXNumber.parameters, 0, this.parameters, 0, length);
      this.parameters = (byte[]) aFAXNumber.parameters.clone ();
      this.name = name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically.
   *
   * The compareTo method does not apply to the NdsFAXNumber syntax
   * because this syntax does not support the ordering and substrings
   * matching rules.
   */
   public int compareTo (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals (
         Object anObject)
   {
      if ((anObject != null) && (anObject instanceof NdsFAXNumber))
      {
         byte[] anotherParameters;
         NdsFAXNumber anotherFAXNumber;

         anotherFAXNumber = (NdsFAXNumber) anObject;
         if (false == telephoneNumber.equals (anotherFAXNumber.telephoneNumber))
         {
            return (false);
         }
         if (this.bitCount != anotherFAXNumber.getBitCount ())
         {
            return (false);
         }
         anotherParameters = anotherFAXNumber.getParameters ();
         for (int i = 0; i < parameters.length; i++)
         {
            if (parameters[i] != anotherParameters[i])
            {
               return (false);
            }
         }
         return (true);
      }
      return (false);
   }

   /**
    * @internal
    */
   public String getName ()
   {
      return (name);
   }

   /**
    * @internal
    */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.FAX_NUMBER);
   }


   // ******************** SchemaComposite Interface ********************

  /**
   * Returns the number of elements in this sequence.
   *
   * @return The number of elements in this sequence as an int.
   */
   public int count ()
   {
      return (compositeIds.length);
   }

   /**
   * Returns a SchemaValue object based on the given passed-in
   * string containing the element value. The SchemaValue object
   * is a Java language data type.
   *
   * @param compositeId Passed-in string containing the element
   *                    value.
   *
   * @return A SchemaValue object containing the telephoneNumber,
   *         bitCount and parameters, or NULL.
   */
   public SchemaValue getValue (
         String compositeId)
   {
      if (compositeId.equalsIgnoreCase (compositeIds[0]))
      {
         // getTelephoneNumber ();
         return (new NdsString (compositeIds[0], telephoneNumber));
      }
      else if (compositeId.equalsIgnoreCase (compositeIds[1]))
      {
         // getBitCount ();
         return (new NdsInteger (compositeIds[1], bitCount));
      }
      else if (compositeId.equalsIgnoreCase (compositeIds[2]))
      {
         // getParameters ();
/*
         return (new SchemaElement (
                        compositeIds[2],
                        address.clone (),
                        "byte[]"));
*/
      }
      return (null);

   } /* getValue () */

  /**
   * Returns an enumeration of element IDs in this composite.
   *
   * @return An enumeration of element IDs.
   */
   public Enumeration getValueIds ()
   {
      Vector vector = new Vector ();

      for (int i=0; i<compositeIds.length; i++)
      {
         vector.addElement(compositeIds[i]);
      }
      return (vector.elements ());
   }

  /**
   * Returns an enumeration of element values in this composite.
   *
   * @return An enumeration of element values telephoneNumber
   *         and bitCount.
   */
   public Enumeration getValues ()
   {
      Vector values = new Vector ();

      values.addElement (new NdsString (compositeIds[0], telephoneNumber));
      values.addElement (new NdsInteger (compositeIds[1], bitCount));
/*
      values.addElement (new SchemaElement (
                                    compositeIds[2],
                                    address.clone (),
                                    "byte[]"));
*/
      return (values.elements ());
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Compares two Objects using the approximate matching rule.
   * 
   * The approximate method does not apply to the NdsFAXNumber
   * syntax because this syntax does not support the approximate
   * equals matching rule.
   */
   public boolean approximate (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Returns the int that represents the NDS syntaxId.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.FAX_NUMBER_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules.
   *
   * <p>The NdsFaxNumber syntax supports only the equality
   *  matching rule.
   *
   * @param matchingRules The set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
      {
         return (true);
      }
      return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone ()
   {
      try
      {
         NdsFAXNumber fax = (NdsFAXNumber) super.clone ();

         fax.telephoneNumber = this.telephoneNumber;
         fax.bitCount = this.bitCount;
         fax.parameters = (byte[]) this.parameters.clone ();

         fax.name = this.name;

         return (fax);
      }
      catch (CloneNotSupportedException e)
      {
         // this shouldn't happen, since we are Cloneable
         throw (new InternalError ());
      }

   } /* clone () */

  /**
   * Generates a string representation of the object. It
   * returns a string that textually represents the object.
   * The result should be a concise but informative
   * representation that is easy to read.
   *
   * @return The String representation of the object.
   */
   public String toString ()
   {
      return (telephoneNumber + ";" + String.valueOf (bitCount) + ";" );
   }


   // ******************** NdsFAXNumber Class ********************

  /**
   * Returns the String value of telephoneNumber stored in the
   * NDS attribute.
   *
   * @return The String value of telephoneNumber.
   */
   public String getTelephoneNumber ()
   {
      return (telephoneNumber);
   }

  /**
   * Returns the long value of bitCount stored in the
   * NDS attribute.
   *
   * @return The long value of bitCount.
   */
   public long getBitCount ()
   {
      return (bitCount & INT_MASK);
   }

  /**
   * Returns the byte array of parameters stored in the
   * NDS attribute.
   *
   * @return The byte array of parameters.
   */
   public byte[] getParameters ()
   {
      return (parameters);
   }

} /* NdsFAXNumber */


