
/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/nds/NdsCounter.java $
  $Revision: 16 $
  $Modtime: 11/15/00 3:43p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import com.novell.service.schema.SchemaNumber;


/**
 * Provides access to values of the Counter attribute. It is
 * used for attributes whose values are incrementally modified
 * numeric signed integers. Any attribute defined using Counter is
 * single-valued. Any value added to an attribute of the Counter
 * syntax is arithmetically added to the total, and any value 
 * deleted is arithmetically subtracted from the total.
 * 
 * <p>Matching Rules: Equality and Ordering </p>
 *
 * @see NdsSyntaxId
 */
public class NdsCounter
   implements NdsAttributeValue, SchemaNumber, Cloneable, Serializable
{
   private static final int matchingRules = equality|ordering;

 /**
  * @internal
  */
   protected int value;
   private String name;

  /**
   * Constructs an NdsCounter object from an int value. Any integer added
   * to the value is arithmetically added to the total, and any integer
   * deleted is arithmetically subtracted from the total.
   *
   * @param value An int stored value.
   * 
   */
   public NdsCounter (
         int value)
   {
      this ("", value);
   }

  /**
   * @internal
   */
   protected NdsCounter ()
   {
      this.value = -1;
      this.name = null;
   }

   /*
    * Private constructor; not documented!
    */
   private NdsCounter (
         String name,
         long value)
   {
      this.value = (int) (value & INT_MASK);
      this.name = name;
   }

  /**
   * @internal
   */
   protected NdsCounter (
         NdsCounter aCounter)
   {
      this.value = aCounter.value;
      this.name = aCounter.name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically. For the Case Exact String syntax, the
   * compareTo method compares this NDS Object with a reference
   * NDS Object contained in the anObject parameter. The
   * comparison is done according to the substrings matching
   * rule for this syntax.
   * 
   * @param anObject The reference NDS Object with which
   *                 to compare.
   *
   * @return An int set to 0 if the reference Object is equal
   *         to this Object; an int <0 if the reference Object
   *         is less than this Object; an int >0 if the
   *         reference Object is greater than this Object.
   *
   * @exception IllegalArgumentException The passed-in object is of
   *            a different type than this NDS Object.
   */
   public int compareTo (
         Object anObject)
      throws Exception
   {
      if (anObject instanceof NdsCounter)
      {
         int value;

         value = ((NdsCounter) anObject).intValue ();
         return (this.value > value ? 1 : this.value < value ? -1 : 0);
      }
      throw (new IllegalArgumentException ());
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals (
         Object anObject)
   {
      if ((anObject != null) && (anObject instanceof NdsCounter))
      {
         return (value == ((NdsCounter) anObject).intValue ());
      }
      return (false);
   }

  /**
   * @internal
   */
   public String getName ()
   {
      return (name);
   }

  /**
   * @internal
   */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.COUNTER);
   }


   // ******************** SchemaNumber Interface ********************

   /**
    * Returns the value of this NDS object as an int.
    * 
    * @return The int value of this object.
    */
   public int intValue ()
   {
      return (value);
   }

   /**
    * Returns the value of this NDS object as a long.
    * 
    * @return The long value of this object.
    */
   public long longValue ()
   {
      return ((long) value);
   }

   /**
    * Returns the value of this NDS object as a float.
    * 
    * @return The float value of this object.
    */
   public float floatValue ()
   {
      return ((float) value);
   }

   /**
    * Returns the value of this NDS object as a double.
    * 
    * @return The double value of this object.
    */
   public double doubleValue ()
   {
      return ((double) value);
   }

   /**
    * Returns the value of this NDS object as a byte.
    * 
    * @return The byte value of this object.
    */
   public byte byteValue ()
   {
      return ((byte) value);
   }

   /**
    * Returns the value of this NDS object as a short.
    * 
    * @return The short value of this object.
    */
   public short shortValue ()
   {
      return ((short) value);
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Compares two Objects using the approximate matching rule.
   * 
   * The approximate method does not apply to the NdsCounter syntax
   * because this syntax does not support the approximate equals
   * matching rule.
   */
   public boolean approximate (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Returns the int that represents the NDS syntax ID.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.COUNTER_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules.
   *
   * <p>The NdsCounter syntax supports the equality
   *  and ordering matching rules.
   *
   * @param matchingRules The set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
      {
         return (true);
      }
      return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone ()
   {
      try
      {
         NdsCounter counter = (NdsCounter) super.clone ();

         counter.value = this.value;
         counter.name = this.name;

         return (counter);
      }
      catch (CloneNotSupportedException e)
      {
         // this shouldn't happen, since we are Cloneable
         throw (new InternalError ());
      }

   } /* clone () */

  /**
   * Generates and returns a string representation of the object. In
   * general, the toString method returns a string that textually 
   * represent this object. The result should be a concise but
   * informative representation that is easy to read.
   *
   * @return The String representation of the object.
   */
   public String toString ()
   {
      return (String.valueOf (value));
   }


   // ******************** NdsCounter Class ********************

  /**
   * Returns the long value of Counter stored in the object
   * data member.
   *
   * @return The stored long value.
   */
   public long getCounter ()
   {
      return (value & INT_MASK);
   }

} /* NdsCounter */
 