
/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/nds/NdsTime.java $
  $Revision: 10 $
  $Modtime: 11/15/00 3:43p $

  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import java.util.Date;
import java.util.Enumeration;
import java.util.GregorianCalendar;
import java.util.TimeZone;


/**
 * Provides access to values of the Time attribute.
 * It is used for attributes whose values represent time. A
 * Time value consists of a whole number of seconds, where
 * zero equals 12:00 midnight, January 1, 1970, UTC. Two
 * Time values are compared by comparing the Integer_T values.
 *
 * <p>Matching Rules: Equality and Ordering</p>
 * 
 * @see NdsSyntaxId
 */
public class NdsTime
   extends NdsInteger
   implements NdsAttributeValue, Cloneable, Serializable
{
   private static final int matchingRules = equality|ordering;

   /**
   * Constructs an NdsTime object based on the specified
   * long value parameter.
   * 
   * @param value The time in a whole number of seconds to be stored
   *              as a long value.
   */
   public NdsTime (
         long value)
   {
      super (value);
   }

   /**
   * Constructs an NdsTime object based on the specified
   * Date value parameter. It converts time from milliseconds
   * to seconds.
   * 
   * @param value The NdsTime Date value to be stored.
   */
   public NdsTime (
         Date value)
   {
      // convert from milliseconds to seconds
      super (value.getTime () / 1000);
   }

  /**
   * @internal
   */
   protected NdsTime ()
   {

   }

   /*
    * Private constructor; not documented!
    */
   private NdsTime (
         String name,
         long value)
   {
      super (name, value);
   }

  /**
   * @internal
   */
   protected NdsTime (
         NdsTime aTime)
   {
      this.value = aTime.value;
      this.name = aTime.name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically. For the Time syntax, the compareTo method
   * compares this NDS Object with a reference NDS Object contained
   * in the anObject parameter. The comparison is done according to
   * the ordering matching rule for this syntax.
   * 
   * @param anObject The reference NDS Object with which
   *                 to compare.
   *
   * @return An int set to 0 if the reference Object is equal to this
   *         Object; an int <0 if the reference Object is less than
   *         this Object; an int >0 if the reference Object is greater
   *         than this Object.
   *
   * @exception IllegalArgumentException The passed-in object is of
   *            a different type than this NDS Object.
   */
   public int compareTo (
         Object anObject)
      throws Exception
   {
      if (anObject instanceof NdsTime)
      {
         int value;

         value = ((NdsTime) anObject).intValue ();
         return (this.value > value ? 1 : this.value < value ? -1 : 0);
      }
      throw (new IllegalArgumentException ());
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals (
         Object anObject)
   {
      if ((anObject != null) && (anObject instanceof NdsTime))
      {
         return (value == ((NdsTime) anObject).intValue ());
      }
      return (false);
   }

  /**
   * @internal
   */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.TIME);
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Returns the int that represents the NDS syntax ID.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.TIME_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules.
   *
   * <p>The NdsTime syntax supports the equality and
   *  ordering matching rules.
   *
   * @param matchingRules The set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
      {
         return (true);
      }
      return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone ()
   {
      NdsTime time = (NdsTime) super.clone ();

      return (time);

   } /* clone () */


   // ******************** NdsTime Class ********************

  /**
   * Returns the time stored as a long value in the object
   * data member.
   * 
   * @return The time stored as a long value.
   */
   public long getTime ()
   {
      return (value & INT_MASK);
   }

  /**
   * Returns the time Date value stored in the object data member.
   * It converts time from milliseconds to seconds.
   *
   * @return The Date value with time in seconds.
   */
   public Date getDate ()
   {
      Date date = new Date ();

      date.setTime ((long)value * 1000);
      return (date);
   }

  /**
   * Returns the GregorianCalendar value stored in the object
   * data member. It converts the date from standard calendar
   * to Gregorian calendar.
   *
   * @return The GregorianCalendar date value.
   */
   public GregorianCalendar getGregorianCalendar ()
   {
      GregorianCalendar calendar;

      calendar = new GregorianCalendar (TimeZone.getTimeZone ("UTC"));
      calendar.setTime (getDate ());
      return (calendar);
   }

} /* NdsTime */

