/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/file/nw/calls/BinderyNameImpl.java $
  $Revision: 8 $
  $Modtime: 1/28/00 1:48p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.file.nw.calls;

import java.rmi.RemoteException;

import com.novell.service.jncp.NSIException;

import com.novell.service.session.SessionException;
import com.novell.service.session.xplat.CallsService;

import com.novell.service.bindery.BinderyUtil;
import com.novell.service.toolkit.jcl.NWLong;

import com.novell.java.lang.IntegerBuffer;

/** @internal
 * BinderyName implementation
 *
 * <p>This class extends BinderyName by providing conversions from a bindery
 * object ID to a name and from a name to a bindery object ID.
 * </p>
 *
 * @see com.novell.service.file.nw.calls.BinderyName
 */

public class BinderyNameImpl extends BinderyName
{

   private CallsService callsService = null;

/* **************************************************************************
* public constructors (not published)
****************************************************************************/

   /**
    * Minimum constructor
    *
    * <p>Construct this object.  The bindery object name, type and objectId
    * is currently unknown.  The getName method will return null until the
    * setName method is called.  Type and objectId will be initialized to
    * -1.
    *
    * @param callsService         calls service for NWDSCalls functions
    *
    * </p>
    */

   public BinderyNameImpl(CallsService callsService)
   {
      this(null, -1, -1, callsService);
   }

   /**
    * Bindery Object Name known constructor
    *
    * <p>Construct this object, with the bindery object type, and objectId
    * currently unknown.  Type and objectId will be initialized to  -1.
    * </p>
    *
    * @param name                Bindery Object Name
    * @param callsService         calls service for NWDSCalls functions
    */

   public BinderyNameImpl(String name, CallsService callsService)
   {
      this(name, -1, -1, callsService);
   }

   /**
    * Bindery Object Type known constructor
    *
    * <p>Construct this object, with the bindery object name currently
    * unknown.  The getName method will return null until the setName method
    * is called.  objectId will be initialized to -1.
    * </p>
    * @param type                bindery object type
    * @param callsService         calls service for NWDSCalls functions
    */

   public BinderyNameImpl(int type, CallsService callsService)
   {
      this(null, type, -1, callsService);
   }

   /**
    * Bindery Object Name and Bindery Object Type known constructor
    *
    * <p>The objectId will be iniitlized to -1.
    * </p>
    * @param name                bindery object name
    * @param type                bindery object type
    * @param callsService         calls service for NWDSCalls functions
    */

   public BinderyNameImpl(String name, int type, CallsService callsService)
   {
      this(name, type, -1, callsService);
   }


   /**
    * Full Constructor
    *
    * <p>The name, type and objectId state data members will all be set to
    * the appropriate input parameters.
    * </p>
    * @param name                bindery object name
    * @param type                bindery object type
    * @param objectId            bindery object ID
    * @param callsService         calls service for NWDSCalls functions
    */

   public BinderyNameImpl(
      String name,
      int type,
      int objectId,
      CallsService callsService)
   {
      super(name, type, objectId);
      this.callsService = callsService;
   }

/* **************************************************************************
* public accessors (overridden from superclass, not published)
****************************************************************************/

   /**
    * Returns the Object Name
    *
    * <p>Returns the object name which is the name handed into the
    * constructor or set with the last call to the setName method.  This
    * method overrides the ObjectName.getName method, and is intended for
    * spi provider use only, since we decided not to use the objectName in
    * the various attribute values (but a string instead), the spi needs to
    * supply all of the possible information into the string name (name+type
    * in this case).  This method will simply call the getNameType method of
    * his BinderyName super class and return the name+type string.
    * </p>
    *
    * @return                    The Object name
    */

   public String getName()
   {
      return getNameType();
   }

   /**
    * Sets the Object Name.
    *
    * <p>Sets the Bindery name.  This method will  cause the bindery name to
    * Id conversion to be attempted (NWGetObjectID).  objectId will be set to
    * the conversion value if the conversion is successfull, it will be set
    * to -1 otherwise.  The Conversion is dependant on the type having been
    * set, either by a constructor, or by the setType method.
    * </p>
    *
    * @param name                The object name
    *
    * @see #nameToObjectId
    * @see #setType
    */

   public void setName(String name)
   {
      super.setName(name);
      if (name != null && getType() != -1 && callsService != null)
      {
         try
         {
            setObjectId(nameToObjectId(name, getType(), callsService));
         } catch (NSIException nsie)
         {
            setObjectId(-1);
         }
      }else
         setObjectId(-1);
   }

   /**
    * Sets the Object Type.
    *
    * <p>Sets the Bindery type.  This method will  cause the bindery name to
    * Id conversion to be attempted (NWGetObjectID).  objectId will be set to
    * the conversion value if the conversion is successfull, it will be set
    * to -1 otherwise.  The Conversion is dependant on the name having been
    * set, either by a constructor, or by the setName method.
    * </p>
    *
    * @param name                The object name
    *
    * @see #nameToObjectId
    * @see #setName
    */

   public void setType(int type)
   {
      super.setType(type);
      if (getName() != null)
      {
         try
         {
            setObjectId(nameToObjectId(getName(), type, callsService));
         } catch (NSIException nsie)
         {
            setObjectId(-1);
         }
      }else
         setObjectId(-1);
   }

/* **************************************************************************
* public static conversion methods
****************************************************************************/


   /**
    * Given an object name and type, return the objectId
    *
    * <p>Converts from the given bindery object name and type to the bindery
    * object ID.  Type is always expected to come in in human readable form.
    * this method will alway byte swap the type before making the native
    * call.
    * </p>
    *
    * @param name                The bindery object name desired
    * @param type                The bindery object type desired
    * @param callsService         calls service for NWCalls functions
    *
    * @return                    The corrisponding bindery object ID.
    *
    * @exception                 com.novell.service.jncp.NSIException
    */

   public static int nameToObjectId(
      String name,
      int type,
      CallsService callsService)
      throws NSIException
   {
      type = BinderyUtil.byteswapID(type);
      IntegerBuffer objId = new IntegerBuffer();

      try
      {
         callsService.getObjectID(name, type, objId);
      } catch (SessionException e)
      {
         throw new NSIException(e.getMessage(), 0, e);
      }

      return objId.intValue();
   }

   /**
    * Given an objectID, return a BinderyNameImpl object
    *
    * <p>Converts objectId to a bindery name and type and instantiates a
    * BinderyNameImpl object with the converted bindery name and type
    * parameters.  The bindery type will always be in human readable form in
    * the BinderyNameImpl object.
    * </p>
    *
    * @param objectId            The desired objectId
    * @param conn                The connection to use for nwcalls
    *
    * @return                    BinderyNameImpl object if the objectId is
    *                            found, null, otherwise.
    *
    * @exception                 com.novell.service.jncp.NSIException
    */

   public static BinderyNameImpl objectIdToBinderyName(
      int objectId,
      CallsService callsService)
      throws NSIException
   {
      String objName;
      IntegerBuffer objType = new IntegerBuffer();

      try
      {
         objName = callsService.getObjectName(objectId, objType);
      } catch (SessionException e)
      {
         throw new NSIException(e.getMessage(), 0, e);
      }

      return new BinderyNameImpl(
                     objName.toString(),
                     BinderyUtil.byteswapID(objType.intValue()),
                     objectId, callsService);
   }
}
