
/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/java/io/EndianOutputStream.java $
  $Revision: 4 $
  $Modtime: 1/18/00 9:07a $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.java.io;


import java.io.OutputStream;
import java.io.DataOutputStream;
import java.io.IOException;


/**
 * Provides for writing primitive Java data types to a stream in an 
 * endian specific format. 
 *
 * <p>Primitive data types are well understood types with associated
 * operations. For example, an int is considered to be a good primitive
 * data type. The data can be converted back using a DataInputStream.
 * A prior knowledge of stream contents is needed.</p>
 *
 * @see EndianOutput
 */
public class EndianOutputStream 
   extends DataOutputStream 
   implements EndianOutput
{
   /**
    * Constructs a new EndianOutputStream object with an
    * OutputStream parameter.
    *
    * @param out The Output stream.
    */
   public EndianOutputStream (
         OutputStream out)
   {
      super (out);
   }

   // ******************** EndianOutput Interface ********************

   /**
    * Writes a String as a NULL-terminated sequence of chars in 
    * high-low order.
    *
    * @param s The String of chars to be written.
    * 
    * @exception IOException If an I/O error has occurred.
    */
   public final void writeHiLoChars (
         String s)
      throws IOException 
   {
      writeChars (s);
      writeChar (0);
   }

   /**
    * Writes a String as a NULL-terminated sequence of chars in
    * low-high order.
    *
    * @param s The String of chars to be written.
    * 
    * @exception IOException If an I/O error has occurred.
    */
   public final void writeLoHiChars (
         String s) 
      throws IOException 
   {
      OutputStream out = this.out;
      int len = s.length ();

      for (int i = 0 ; i < len ; i++) 
      {
         int v = s.charAt (i);

         out.write ((v >>> 0) & 0xFF);
         out.write ((v >>> 8) & 0xFF);
      }
      written += len * 2;
      writeChar (0);
   }

   /**
    * Writes a 32-bit int in high-low order.
    *
    * @param v The integer value to be written.
    * 
    * @exception IOException If an I/O error has occurred.
    */
   public final void writeHiLoInt (
         int v)
      throws IOException 
   {
      writeInt (v);
   }

   /**
    * Writes a 32-bit int in low-high order.
    *
    * @param v The integer value to be written.
    * 
    * @exception IOException If an I/O error has occurred.
    */
   public final void writeLoHiInt (
         int v)
      throws IOException 
   {
      OutputStream out = this.out;

      out.write ((v >>>  0) & 0xFF);
      out.write ((v >>>  8) & 0xFF);
      out.write ((v >>> 16) & 0xFF);
      out.write ((v >>> 24) & 0xFF);
      written += 4;
   }

   /**
    * Writes a 16-bit short in high-low order.
    *
    * @param v The short value to be written.
    * 
    * @exception IOException If an I/O error has occurred.
    */
   public final void writeHiLoShort (
         int v)
      throws IOException 
   {
      writeShort (v);
   }

   /**
    * Writes a 16-bit short in low-high order.
    *
    * @param v The short value to be written.
    * 
    * @exception IOException If an I/O error has occurred.
    */
   public final void writeLoHiShort (
         int v)
      throws IOException 
   {
      OutputStream out = this.out;

      out.write ((v >>> 0) & 0xFF);
      out.write ((v >>> 8) & 0xFF);
      written += 2;
   }

   /**
    * Writes a 64-bit long in high-low order.
    *
    * @param v The long value to be written.
    * 
    * @exception IOException If an I/O error has occurred.
    */
   public final void writeHiLoLong (
         long v)
      throws IOException 
   {
      writeLong (v);
   }

   /**
    * Writes a 64-bit long in low-high order.
    *
    * @param v The long value to be written.
    * 
    * @exception IOException If an I/O error has occurred.
    */
   public final void writeLoHiLong (
         long v)
      throws IOException 
   {
      OutputStream out = this.out;

      out.write ((int)(v >>>  0) & 0xFF);
      out.write ((int)(v >>>  8) & 0xFF);
      out.write ((int)(v >>> 16) & 0xFF);
      out.write ((int)(v >>> 24) & 0xFF);
      out.write ((int)(v >>> 32) & 0xFF);
      out.write ((int)(v >>> 40) & 0xFF);
      out.write ((int)(v >>> 48) & 0xFF);
      out.write ((int)(v >>> 56) & 0xFF);
      written += 8;
   }

   /**
    * Writes empty 8-bit bytes to the specified byte boundary.
    *
    * @param n The byte boundary.
    * 
    * @exception IOException If an I/O error has occurred.
    */
   public final void pad (
         int n)
      throws IOException 
   {
      OutputStream out = this.out;
      int len = written % n;

      for (int i = 0 ; i < len ; i++) 
      {
         out.write (0);
      }
      written += len;
   }

} /* EndianOutputStream */


