
/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/nds/NdsEMailAddress.java $
  $Revision: 8 $
  $Modtime: 10/15/99 3:52p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import java.util.Enumeration;
import java.util.Vector;

import com.novell.service.schema.SchemaValue;
import com.novell.service.schema.SchemaComposite;


/**
 * Provides access to attribute values of the NdsEMailAddress
 * syntax. It is used for attributes whose values represent
 * e-mail addresses.
 *
 * <p>Two e-mail addresses match for equality when their lengths
 * and corresponding characters are identical in all respects
 * except that case is ignored and extra spaces are not
 * significant. In matching attributes that conform to this
 * syntax, NDS omits those spaces that are not significant
 * (leading spaces, trailing spaces, and multiple consecutive
 * internal spaces). NDS stores insignificant spaces with the
 * attribute value.
 *
 * <p>Matching Rules: Equality</p>
 *
 * @see NdsSyntaxId
 */
public class NdsEMailAddress
   implements NdsAttributeValue, SchemaComposite, Cloneable, Serializable
{
   private static final int matchingRules = equality;

   private static String[] compositeIds;

   static
   {
      compositeIds = new String[2];
      compositeIds[0] = "type";
      compositeIds[1] = "address";
   }

   /**
    * @internal
    */
   protected int type;

   /**
    * @internal
    */
   protected String address;

   private String name;


   /**
   * Constructs an NdsEMailAddress object based on the specified
   * type and address parameters.
   * 
   * @param type    The e-mail address type to be stored as a long.
   * @param address The e-mail address to be stored as a String.
   */
   public NdsEMailAddress (
         long type,
         String address)
   {
      this ("", type, address);
   }

   /**
    * @internal
    */
   protected NdsEMailAddress ()
   {
      this.type = -1;
      this.address = null;
      this.name = null;
   }

   /**
    * @internal
    */
   protected NdsEMailAddress (
         String name,
         long type,
         String address)
   {
      this.type = (int) (type & INT_MASK);
      this.address = address;
      this.name = name;
   }

   /**
    * @internal
    */
   protected NdsEMailAddress (
         NdsEMailAddress anEMailAddress)
   {
      this.type = anEMailAddress.type;
      this.address = anEMailAddress.address;
      this.name = anEMailAddress.name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically.
   *
   * The compareTo method does not apply to the NdsEMailAddress syntax
   * because this syntax does not support the ordering and substrings
   * matching rules.
   */
    public int compareTo (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals(Object anObject)
   {
      if ((anObject != null) && (anObject instanceof NdsEMailAddress))
      {
         NdsEMailAddress anotherEMailAddress;

         anotherEMailAddress = (NdsEMailAddress) anObject;

         if (type == anotherEMailAddress.getType ())
         {
            return (false);
         }
         if (false == address.equals (anotherEMailAddress.getAddress ()))
         {
            return (false);
         }
         return (true);
      }
      return (false);
   }

  /**
   * @internal
   */
   public String getName ()
   {
      return (name);
   }

  /**
   * @internal
   */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.EMAIL_ADDRESS);
   }


   // ******************** SchemaComposite Interface ********************

  /**
   * Returns the number of elements in this sequence.
   *
   * @return The number of elements in this sequence as an int.
   */
   public int count ()
   {
      return (compositeIds.length);
   }

   /**
   * Returns a SchemaValue object based on the given passed-in
   * string containing the element value.
   *
   * <p>The SchemaValue object is a Java language data type.
   *
   * @param compositeId The passed-in string containing the
   *                    element value.
   *
   * @return SchemaValue The object containing the type
   *         and address, or NULL.
   */
   public SchemaValue getValue (
         String compositeId)
   {
      if (compositeId.equalsIgnoreCase (compositeIds[0]))
      {
         // getType ();
         return (new NdsInteger (compositeIds[0], type));
      }
      else if (compositeId.equalsIgnoreCase (compositeIds[1]))
      {
         // getAddress ();
         return (new NdsString (compositeIds[1], address));
      }
      return (null);
   }

  /**
   * Returns an enumeration of element IDs in this composite.
   *
   * @return The enumeration of element IDs.
   */
   public Enumeration getValueIds ()
   {
      Vector vector = new Vector ();

      for (int i=0; i<compositeIds.length; i++)
         vector.addElement(compositeIds[i]);
      return (vector.elements ());
   }

  /**
   * Returns an enumeration of element values in this composite.
   *
   * @return The enumeration of element values (type and address)
   *         in this composite.
   */
   public Enumeration getValues ()
   {
      Vector values = new Vector ();

      values.addElement (new NdsInteger (compositeIds[0], type));
      values.addElement (new NdsString (compositeIds[1], address));
      return (values.elements ());
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Compares two Objects using the approximate matching rule.
   * 
   * The approximate method does not apply to the NdsEMailAddress
   * syntax because this syntax does not support the approximate
   * equals matching rule.
   */
   public boolean approximate (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Returns the int that represents the NDS syntax ID.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.EMAIL_ADDRESS_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules.
   *
   * <p>The NdsEMailAddress syntax supports only the equality
   *  matching rule.
   *
   * @param matchingRules The set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
         return (true);
      else
         return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone ()
   {
      try
      {
         NdsEMailAddress eMail = (NdsEMailAddress) super.clone ();

         eMail.type = this.type;
         eMail.address = this.address;

         eMail.name = this.name;

         return (eMail);
      }
      catch (CloneNotSupportedException e)
      {
         // this shouldn't happen, since we are Cloneable
         throw (new InternalError ());
      }

   } /* clone () */

  /**
   * Generates a string representation of the object. It
   * returns a string that textually represents the object.
   * The result should be a concise but informative
   * representation that is easy to read.
   *
   * @return The String representation of the object.
   */
   public String toString ()
   {
      return (String.valueOf (type) + ";" + address);
   }


   // ******************** NdsEMailAddress Class ********************

  /**
   * Returns the long value of e-mail address type stored in the
   * NDS attribute.
   *
   * @return The long value of e-mail address type.
   */
   public long getType ()
   {
      return (type & INT_MASK);
   }

  /**
   * Returns the String value of e-mail address stored in the NDS
   * attribute.
   *
   * @return The String value of e-mail address.
   */
   public String getAddress ()
   {
      return (address);
   }

} /* NdsEMailAddress */


