/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/qms/QMSQueue.java $
  $Revision: 12 $
  $Modtime: 1/28/00 12:46p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.qms;


import com.novell.service.jncp.NSIException;
import java.util.*;

/**
 * Provides a central storage and queueing mechanism that enables users
 * to create jobs that are added to the queue. QMSQueue is responsible
 * for managing jobs placed in the queue. The jobs can be serviced
 * remotely by an application at another node on the network.
 *
 * <p>Attribute IDs of a Queue include only the State attribute. There
 * are only three State flags defined: QS_CANT_ADD_JOBS, QS_CANT_SERVICE_JOBS,
 * AND QS_SERVERS_CANT_ATTACH.</p>
 *
 * @see QMSJob
 */

public interface QMSQueue
{
   /**
    * Queue state flag (if set) indicating that adding jobs is not allowed.
    * This flag is part of the State attribute.
    *
    * <p>(QS_CANT_ADD_JOBS = 0x01)</p>
    *
    * @see #getState
    */
   public static final int QS_CANT_ADD_JOBS        = 0x01;

   /**
    * Queue state flag (if set) indicating that servers are not allowed to
    * attach to the queue. This flag is part of the State attribute.
    *
    * <p>(QS_SERVERS_CANT_ATTACH = 0x02)</p>
    *
    * @see #getState
    */
   public static final int QS_SERVERS_CANT_ATTACH  = 0x02;

   /**
    * Queue state flag (if set) indicating that servers are not allowed to
    * service jobs in the queue. This flag is part of the State attribute.
    *
    * <p>(QS_CANT_SERVICE_JOBS = 0x04)</p>
    *
    * @see #getState
    */
   public static final int QS_CANT_SERVICE_JOBS    = 0x04;

   /**
    * Bindery object type associated with a print queue (used by NetWare
    * print queue services).
    *
    * <p>(OT_PRINT_QUEUE = 0x03)</p>
    */
   public static final int OT_PRINT_QUEUE          = 0x03;

   /**
    * Bindery object type associated with an archive queue.
    *
    * <p>(OT_ARCHIVE_QUEUE = 0x08)</p>
    */
   public static final int OT_ARCHIVE_QUEUE        = 0x08;

   /**
    * Bindery object type associated with a job queue.
    *
    * <p>(OT_JOB_QUEUE = 0x0A)</p>
    */
   public static final int OT_JOB_QUEUE            = 0x0A;

   // QMSQueue public accessor methods =======================================

   /**
    * Returns the state of the queue as a bit mask. This is a convenience
    * method for accessing the queue State attribute. It returns the
    * State attribute just as the JNDI DirContext.getAttributes() method.
    *
    * @return  The current state of the queue.
    *
    * @exception NSIException  A network exception occurred.
    *
    * @see #QS_CANT_ADD_JOBS
    * @see #QS_SERVERS_CANT_ATTACH
    * @see #QS_CANT_SERVICE_JOBS
    * @see #setState
    */
   public int getState ()
      throws NSIException;

   /**
    * Sets the state of the queue given the new settings. This is a
    * convenience method for accessing the queue State attribute.
    *
    * @param newState  The desired state.
    *
    * @exception NSIException  A network exception occurred.
    *
    * @see #QS_CANT_ADD_JOBS
    * @see #QS_SERVERS_CANT_ATTACH
    * @see #QS_CANT_SERVICE_JOBS
    * @see #getState
    */
   public void setState (
         int newState)
      throws NSIException;

   /**
    * Returns the object ID for this queue.
    *
    * @return  The object ID for this queue.
    */
   public int getID ();

   // QMSQueue job related methods ===========================================

   /**
    * Returns an enumeration of jobs in this queue. This is a
    * convenience method for listing jobs, and is equivalent to
    * the JNDI Context.list() method
    *
    * @return  An enumeration of the jobs in this queue.
    *          The objects returned will implement the
    *          QMSJob interface.
    *
    * @see QMSJob
    */
   public Enumeration listJobs ();

   /**
    * Creates a new job for this queue.
    *
    * <p>This job does not yet exist in the queue. In order for the queue
    * to recognize the job, it must be submitted to the queue using the
    * submit() method. The queue will wait until the output stream returned
    * from the submit() method is closed before it begins processing the job.
    * </p>
    *
    * @return  A new QMSJob object.
    */
   public QMSJob createJob ();

   /**
    * Creates a QMSJob for an existing job specified by its job ID.
    *
    * <p>This job already exists and may be in the process of being
    *  serviced.</p>
    *
    * @return  A QMSJob object.
    */
   public QMSJob createJob (int ID);

} /* interface QMSQueue */
