/* **************************************************************************

  $Archive: /njcl/src/com/novell/service/toolkit/jcl/NWLong.java $
  $Revision: 1 $
  $Modtime: 3/05/98 2:48p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.toolkit.jcl;

/**
 * <p>Provides a changeable long value that can be passed by
 * reference.
 *
 * <p>NWLong helps avoid the problem with Java long base types
 * being passed by value, and Java Long objects not having a set
 * function.
 *
 * <p>The intended use for NWLong is for passing it to a native
 * routine that sets the value. The value can then be retrieved
 * and stored in a long base type value.
 * </p>
 */


 /*
 * Public Method List (in order of their apperance in the file).
 * <br>
 * CTD   Method name - Description
 * ---   --------------------------------------------------------------------
 * c d   NWLong - Constructor that takes a long for the new value.
 * c d   NWLong - Constructor that takes an int for the new value.
 * c d   getValue - Retrieves the current value as a long.
 * c d   getIntValue - Retrieves the current value as an integer.
 * c d   setValue - Sets/overwrites the current value using a long.
 * c d   setValue - Sets/overwrites the current value using an integer.
 * c d   setValue - Sets/overwrites the current value using a java.lang.Long.
 * c d   toString - Returns a string representing the NWLong object.
 * c d   intToLong - Converts an int to long (unsigned).
 * c d   byteToLong - Converts a byte to long (unsigned).
 * </pre>
 ***************************************************************************/

public final class NWLong
{
   private long _value;


   /**
    * Constructs an NWLong with a default value of zero.
    * </p>
    */
   public NWLong ()
   {
      _value = 0;
   }

   /**
    * Constructs an NWLong using a value specified by a long.
    * </p>
    *
    * @param <i>l</i>            (in) Initial value.
    */
   public NWLong (long l)
   {
      _value = l;
   }

   /**
    * Constructs an NWLong using a value specified by an int.
    * </p>
    *
    * @param <i>i</i>            (in) Initial value.
    */
   public NWLong (int i)
   {
      _value = NWLong.intToLong (i);
   }

   /**
    * Returns the current value as a long.
    * </p>
    *
    * @return                    The current value as a long.
    */
   public long getValue ()
   {
      return (_value);
   }

   /**
    * Returns the current value as an int.
    *
    * <p>NOTE: This could cause the sign bit of the returned integer to
    * be set.  Use getIntValue() only for flags or similar values,
    * which will not be compared.
    * </p>
    *
    * @return                    The current value as an int.
    */
   public int getIntValue ()
   {
      return ((int) _value);
   }

   /**
    * Sets the current value using a long.
    * </p>
    *
    * @param <i>l</i>            (in) New value.
    */
   public void setValue (
         long l)
   {
      _value = l;
   }

   /**
    * Sets the current value using an unsigned integer.
    *
    * <p>The integer is actually signed, but this routine treats it as an
    * unsigned 32-bit value.
    * </p>
    *
    * @param   <i>i</i>          (in) The new value as an int.
    */
   public void setValue (
         int i)
   {
      _value = NWLong.intToLong (i);
   }

   /**
    * Sets the current value using a java.lang.Long.
    *
    * <p>If <i>l</i> parameter is <tt>null</tt>, the value is not changed.
    * </p>
    *
    * @param <i>l</i>            (in) The new value as a java.lang.Long.
    */
   public void setValue (
         java.lang.Long l)
   {
      if (null != l)
         _value = l.longValue ();
   }

   /**
    * Returns a String representing the NWLong.
    * </p>
    *
    * @return                    String form of the NWLong.
    */
   public String toString ()
   {
      String s;

      s = ":" + (Long.toString (_value) + " or 0x" + Long.toString (_value, 16));
      return (getClass().getName() + s);
   }

   /**
    * Converts an int to long (unsigned).
    *
    * <p>Simply casting the value does not work, because int is signed.
    * If the sign bit is set, a negative number is produced.
    * </p>
    *
    * @param   <i>i</i>          (in) Integer to be converted.
    *
    * @return                    Converted form of <i>i</i> to a long.
    */
   public static long intToLong (
         int i)
   {
      long uValue;

      if (i < 0)
      {
         i &= 0x7fffffff;
         uValue = i;
         uValue += 0x7fffffff;
         uValue++;
      }
      else
         uValue = i;
      return (uValue);
   } /* intToLong () */

   /**
    * Converts a byte to long (unsigned).
    *
    * <p>Simply casting the value does not work, because byte is signed.
    * If the sign bit is set, a negative number is produced.
    * </p>
    *
    * @param   <i>b</i>          (in) Byte to be converted.
    *
    * @return                    Converted form of <i>b</i> to a long.
    */
   public static long byteToLong (
         byte b)
   {
      long uValue;

      if (b < 0)
      {
         b &= 0x7f;
         uValue = b;
         uValue += 0x7f;
         uValue++;
      }
      else
         uValue = b;

      return (uValue);
   } /* byteToLong () */
}

