/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/file/nw/DataAccessableParameters.java $
  $Revision: 7 $
  $Modtime: 12/13/99 3:42p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.file.nw;

/**
 * Provides a DataAccessor implementation using the custom data
 * openFlags and dataSelector parameters.
 * 
 * <p>DataAccessableParameters is the file system's implementation of
 * the custom data parameter passed as the custom parameter to the
 * various NInputStream, NOutputStream and RandomAccess constructors.
 * This class is used in conjunction with the NFileInputStream,
 * NFileOutputStream and NRandomAccessFile classes. It defines constants
 * that can be used for the various constructors in these classes, or
 * this object can be instantiated and passed into the custom parameter
 * constructors.</p>
 *
 * <p>When opening a file stream, you can use any of the openFlags
 * outlined in the variable section. However, all streams do not support
 * all of the open flags specified below (extended attributes for example).
 * The only flags that are guaranteed to be supported for any accessor
 * are the READ and WRITE flags (and READ_WRITE for random access).
 *
 * <p>The dataSelector allows selection of different data forks on a device.  
 * For example, a file can have a data fork and a resource fork. To choose
 * the default fork, independent of platform and device type, specify 
 * DEFAULT_DATASELECTOR for this parameter. 
 *
 * @see com.novell.java.io.DataAccessable
 * @see com.novell.java.io.NInputStream
 * @see com.novell.java.io.NOutputStream
 * @see com.novell.java.io.RandomAccess
 * @see NFileInputStream
 * @see NFileOutputStream
 * @see NRandomAccessFile
 * @see #DEFAULT_DATASELECTOR
 */
 

public class DataAccessableParameters implements Cloneable
{

/* **************************************************************************
   openFlag constants
****************************************************************************/

   /**
    * The openFlag variable allowing the data accessor to be opened in
    * read mode.
    *
    * <p>(READ = 0x0001)</p>
    */
   public static final int READ              = 0x0001;

   /**
    * The openFlag variable allowing the data accessor to be opened in
    * write mode.
    *
    * <p>(WRITE = 0x0002)</p>
    */
   public static final int WRITE             = 0x0002;

   /**
    * The openFlag variable allowing the data accessor to be opened but
    * deny any shared read requests.
    *
    * <p>(DENY_READ = 0x0004)</p>
    */
   public static final int DENY_READ         = 0x0004;

   /**
    * The openFlag variable allowing the data accessor to be opened but
    * deny any shared write requests.
    *
    * <p>(DENY_WRITE = 0x0008)</p>
    */
   public static final int DENY_WRITE        = 0x0008;

   /**
    * The openFlag variable allowing the data accessor to be opened in
    * NetWare compatability mode.
    *
    * <p>(COMPATABILITY = 0x0010)</p>
    */
   public static final int COMPATABILITY     = 0x0010;

   /**
    * The openFlag variable allowing the data accessor to be opened in
    * a mode where any writes are immediately flushed to the device.
    *
    * <p>(WRITE_THROUGH = 0x0040)</p>
    */
   public static final int WRITE_THROUGH     = 0x0040;

   /**
    * The openFlag variable allowing the data accessor to be opened
    * in compressed mode.
    *
    * <p>(OPEN_COMPRESSED = 0x0100)</p>
    */
   public static final int OPEN_COMPRESSED   = 0x0100;

   /**
    * The openFlag variable allowing the data accessor to be opened
    * in delete-after-close mode.
    *
    * <p>Beginning with NetWare 4.10, a file can be opened as a temporary
    * file that will be deleted when the file is closed. To create a
    * temporary file, the caller must set the DELETE_FILE_CLOSE bit, the
    * DENY_READ bit, and the DENY_WRITE bit in the mode parameter of
    * openStream(). Temporary files by definition do not allow shared I/O
    * access. This file will exist until the caller closes the file, which
    * will cause the file to be deleted.  NetWare will create temporary files
    * with the HIDDEN bit set so that they will not be visible.</p>
    *
    * <p>(DELETE_FILE_CLOSE = 0x0400)</p>
    */
   public static final int DELETE_FILE_CLOSE = 0x0400;

   /**
    * The openFlag variable allowing the data accessor to be opened
    * in read/write mode.
    *
    * <p>(READ_WRITE = READ | WRITE)</p>
    */
   public static final int READ_WRITE = READ | WRITE;

   /**
    * The default NFileInputFile openFlags value.
    *
    * <p>(DEFAULT_INPUT_OPENFLAGS = READ)</p>
    */
    public static final int DEFAULT_INPUT_OPENFLAGS = READ;
    

   /**
    * The default NFileOutputFile openFlags value.
    *
    * <p>(DEFAULT_OUTPUT_OPENFLAGS = WRITE)</p>
    */
    public static final int DEFAULT_OUTPUT_OPENFLAGS = WRITE;
    
/* **************************************************************************
   dataSelector constants
****************************************************************************/

   /**
    * The dataSelector variable allowing an EA stream to be opened on
    * a directory entry. Pass this constant in as the dataSelector on
    * an open relative to a directory or any directory entry and an
    * EA stream will be opened. This is valid for use only with the
    * various subordinate constructors.</p>
    *
    * <p>(EA_STREAM_SELECTOR = 0xFF)</p>
    */
   public static final int EA_STREAM_SELECTOR = 0xFF;

   /**
    * The dataSelector variable with the default value.
    *
    * <p>(DEFAULT_DATASELECTOR = -1)</p>
    */
    public static final int DEFAULT_DATASELECTOR = -1;
    
/* **************************************************************************
   Private data members
****************************************************************************/

   private int openFlags;
   private int dataSelector;

/* **************************************************************************
   Constructor
****************************************************************************/

   /**
    * Constructs a DataAccessableParameters object.
    *
    * <p>The DataAccessableParameters object is suitable for the custom
    * parameter of the various NInputStream, NOutputStream, RandomAccess,
    * NFileInputStream, NFileOutputStream and NRandomAccessFile class
    * constructors.<p/>
    *
    * @param openFlags    The open flags to use when opening the 
    *                     stream.
    *
    * @param dataSelector The dataSelector to use when opening the 
    *                     stream.
    */

   public DataAccessableParameters(
      int openFlags,
      int dataSelector)
   {
      this.openFlags = openFlags;
      this.dataSelector = dataSelector;
   }

/* **************************************************************************
   public accessors
****************************************************************************/

   /**
    * Returns the openFlags value that was set in the constructor.
    *
    * @return  The openFlags value.
    */
    
   public int getOpenFlags()
   {
      return openFlags;
   }

   /**
    * Returns the dataSelector value that was set in the constructor.
    *
    * @return The dataSelector value.
    */
    
   public int getDataSelector()
   {
      return dataSelector;
   }

/* **************************************************************************
   Overridden Object methods
****************************************************************************/

   /**
    * Compares the input parameter object to this DataAccessableParameters
    * object.
    * 
    * @return A boolean set to TRUE if the objects are the same;
    *         otherwise, FALSE is returned if they are different.
    */

   public boolean equals(Object obj)
   {
      if (obj instanceof DataAccessableParameters)
      {
         DataAccessableParameters dap = (DataAccessableParameters) obj;
         if (dap.getOpenFlags() != openFlags)
            return false;
         if (dap.getDataSelector() != dataSelector)
            return false;
         return true;
      }
      return false;
   }

   /**
    * Returns a debug string representation of this DataAccessableParameters
    * object.
    * 
    * <p>The debug string is in the following format: nameSpace: s,     
    * openFlags: i, dataSelector: i.</p>
    *
    * @return The debug string representation.
    */
    
   public String toString()
   {
      return new String(
                  "openFlags: " + openFlags +
                  ", dataSelector: " + dataSelector);   
   
   }

   /**
    * Returns a clone of this DataAccessableParameters object.
    * 
    * @return The cloned DataAccessableParameters object.
    */

   public Object clone()
   {
      DataAccessableParameters dap = null;
      try
      {
         dap = (DataAccessableParameters) super.clone();
      }
      catch (CloneNotSupportedException e)
      {
         // this shouldn't happen, since we are Cloneable
         throw (new InternalError());
      }

      dap.openFlags = openFlags;
      dap.dataSelector = dataSelector;
      return dap;
   }

/* **************************************************************************
   public  spi methods
****************************************************************************/

   /**
    * Opens a file stream with the passed in subordinate name.
    * Set up default parameters as needed.
    * 
    * <p>This method should be used only by SPI provider developers;
    * it is not intended for the user application interface. The
    * implementors of the DataAccessable openStream custom data methods
    * should call this method on the custom data object received before
    * accessing any of the custom data paramters.</p>
    *
    * @param name The subordinate name to be opened, or NULL for
    *             non-subordinate opens.
    *
    * @return The subordinate name that should be used to open the
    *         stream, or NULL for non-subordinate opens.
    */
        
   public String checkParameters(
      String name, 
      String nameSpace)
   {
/*
      if (nameSpace == null)
      {
         // calculate default name space
         // if we're talking about a dir entry, then get the name space
         //   from it -- otherwise go to the JVM to calculate the NS
         if (opener instanceof DirEntryDirContext)
            nameSpace = ((DirEntryDirContext)opener).getNameSpace();
         else
            nameSpace = NameSpace.numberToName(
                                    NameSpace.getPlatformDefaultNameSpace());
      }
*/
      // calculate default file content stream depening on NS
      if (dataSelector == -1)
      {
         String macNS = NameSpace.numberToName(NameSpace.MAC_INT);

         // for MAC NS only, the file contents are in data stream 1
         if (nameSpace.compareTo(macNS) == 0)
            dataSelector = 1;
         else
            dataSelector = 0;
      }

      // force the name uppercase if in the DOS name space, because of a
      //   bug that allows a lower-case file name to be created in the DOS
      //   name space
      if (nameSpace.compareTo(NameSpace.numberToName(
          NameSpace.DOS_INT)) == 0 && dataSelector != EA_STREAM_SELECTOR)
      {
         if (name != null)
            name = name.toUpperCase();
      }
      return name;
   }
}   
