
/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/nds/NdsHold.java $
  $Revision: 9 $
  $Modtime: 10/15/99 3:23p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import java.util.Enumeration;
import java.util.Vector;

import com.novell.service.schema.SchemaValue;
import com.novell.service.schema.SchemaComposite;


/**
 * Provides access to values of the Hold attribute. It is used for
 * attributes whose values represent an object name/level pair.
 *
 * <p>The Hold syntax is an accounting quantity, which is an amount
 * tentatively held against a subject's credit limit, pending
 * completion of a transaction. NDS treates the Hold amount
 * similarly to the Counter syntax, with new values added to or
 * subtracted from the base total. If the evaluated hold amount
 * goes to 0 (zero), the Hold record is deleted.
 * 
 * <p>Matching Rules: Equality</p>
 *
 * @see NdsSyntaxId
 */
public class NdsHold
   implements NdsAttributeValue, SchemaComposite, Cloneable, Serializable
{
   private static final int matchingRules = equality;

   private static String[] compositeIds;

   static
   {
      compositeIds = new String[2];
      compositeIds[0] = "objectName";
      compositeIds[1] = "amount";
   }

  /**
   * @internal
   */
   protected String objectName;

  /**
   * @internal
   */
   protected int amount;

   private String name;

  /**
   * Constructs an NdsHold object based on the specified objectName
   * and amount parameters.
   *
   * @param objectName The distinguished name of the target object stored
   *                   as a String.
   * @param amount     The Accounting quantity tentatively held against
   *                   the target object's credit limit.
   */
   public NdsHold (
         String objectName,
         long amount)
   {
      this ("", objectName, amount);
   }

  /**
   * @internal
   */
   protected NdsHold ()
   {
      this.objectName = null;
      this.amount = -1;
      this.name = null;
   }

   /*
    * Private constructor; not documented!
    */
   private NdsHold (
         String name,
         String objectName,
         long amount)
   {
      this.objectName = objectName;
      this.amount = (int) (amount & INT_MASK);
      this.name = name;
   }

  /**
   * @internal
   */
   protected NdsHold (
         NdsHold aHold)
   {
      this.objectName = aHold.objectName;
      this.amount = aHold.amount;
      this.name = aHold.name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically.
   *
   * The compareTo method does not apply to the NdsHold syntax
   * because this syntax does not support the ordering and substrings
   * matching rules.
   */
   public int compareTo (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals (
         Object anObject)
   {
      if ((anObject != null) && (anObject instanceof NdsHold))
      {
         NdsHold anotherHold;

         anotherHold = (NdsHold) anObject;

         if (false == objectName.equals (anotherHold.objectName))
         {
            return (false);
         }
         if (amount == anotherHold.amount)
         {
            return (false);
         }
         return (true);
      }
      return (false);

   }

  /**
   * @internal
   */
   public String getName ()
   {
      return (name);
   }

  /**
   * @internal
   */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.HOLD);
   }


   // ******************** SchemaComposite Interface ********************

  /**
   * Returns the number of elements in this sequence.
   *
   * @return The number of elements in this sequence as an int.
   */
   public int count ()
   {
      return (compositeIds.length);
   }

   /**
   * Returns a SchemaValue object based on the given passed-in
   * string containing the element value. The SchemaValue object
   * is a Java language data type.
   *
   * @param compositeId The passed-in string containing the element
   *                    value.
   *
   * @return The SchemaValue object containing the objectName,
   *         and amount, or NULL.
   */
   public SchemaValue getValue (
         String compositeId)
   {
      if (compositeId.equalsIgnoreCase (compositeIds[0]))
      {
         // getObjectName ();
         return (new NdsString (compositeIds[0], objectName));
      }
      else if (compositeId.equalsIgnoreCase (compositeIds[1]))
      {
         // getAmount ();
         return (new NdsInteger (compositeIds[1], amount));
      }
      return (null);
   }

  /**
   * Returns an enumeration of composite IDs.
   *
   * @return An enumeration of composite IDs.
   */
   public Enumeration getValueIds ()
   {
      Vector vector = new Vector ();

      for (int i=0; i<compositeIds.length; i++)
         vector.addElement(compositeIds[i]);
      return (vector.elements ());
   }

  /**
   * Returns an enumeration of the elements in this composite.
   *
   * @return An enumeration of the composite elements objectName
   *         and amount.
   */
   public Enumeration getValues ()
   {
      Vector values = new Vector ();

      values.addElement (new NdsString (compositeIds[0], objectName));
      values.addElement (new NdsInteger (compositeIds[1], amount));
      return (values.elements ());
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Compares two Objects using the approximate matching rule.
   * 
   * The approximate method does not apply to the NdsHold syntax
   * because this syntax does not support the approximate equals
   * matching rule.
   */
   public boolean approximate (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Returns the int that represents this NDS syntax ID.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.HOLD_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules. The NdsHold syntax supports only the
   * Equality matching rule.
   *
   * @param matchingRules The set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
      {
         return (true);
      }
      return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone ()
   {
      try
      {
         NdsHold hold = (NdsHold) super.clone ();

         hold.objectName = this.objectName;
         hold.amount = this.amount;

         hold.name = this.name;

         return (hold);
      }
      catch (CloneNotSupportedException e)
      {
         // this shouldn't happen, since we are Cloneable
         throw (new InternalError ());
      }

   } /* clone () */

  /**
   * Generates a string representation of the object. It
   * returns a string that textually represents the object.
   * The result should be a concise but informative
   * representation that is easy to read.
   *
   * @return The String representation of the object.
   */
   public String toString ()
   {
      return (objectName + ";" + String.valueOf (amount));
   }


   // ******************** NdsHold Class ********************

  /**
   * Returns the String reference to the target object.
   *
   * @return The name of the target object as a String.
   */
   public String getObjectName ()
   {
      return (objectName);
   }

  /**
   * Returns the accounting quantity held against the target object's
   * credit limit.
   *
   * @return The accounting quantity held against the target object's
   *         credit limit.
   */
   public long getAmount ()
   {
      return (amount & INT_MASK);
   }

} /* NdsHold */


