/*************************************************************************
Copyright  1999-2002 Novell, Inc. All Rights Reserved. 

THIS WORK IS SUBJECT TO U.S. AND INTERNATIONAL COPYRIGHT LAWS AND 
TREATIES. USE AND REDISTRIBUTION OF THIS WORK IS SUBJECT TO THE LICENSE
AGREEMENT ACCOMPANYING THE SOFTWARE DEVELOPMENT KIT (SDK) THAT CONTAINS
THIS WORK. PURSUANT TO THE SDK LICENSE AGREEMENT, NOVELL HEREBY GRANTS
TO DEVELOPER A ROYALTY-FREE, NON-EXCLUSIVE LICENSE TO INCLUDE NOVELL'S
SAMPLE CODE IN ITS PRODUCT. NOVELL GRANTS DEVELOPER WORLDWIDE DISTRIBUTION
RIGHTS TO MARKET, DISTRIBUTE, OR SELL NOVELL'S SAMPLE CODE AS A COMPONENT
OF DEVELOPER'S PRODUCTS. NOVELL SHALL HAVE NO OBLIGATIONS TO DEVELOPER OR
DEVELOPER'S CUSTOMERS WITH RESPECT TO THIS CODE.

***************************************************************************/
/*******************************************************************
* Name: native_interface.h
* Description: define a C/C++ interface for DirXML drivers
* Tabs: 4
*******************************************************************/
#ifndef _NATIVE_INTERFACE_H_
#define	_NATIVE_INTERFACE_H_

#include	"define_interface.h"
#include	"dom.h"
#include	"sax.h"
#include	"OutputStream.h"
#include	"XmlWriter.h"

//NOTE: can't use wchar_t because it is 32-bits on Solaris
typedef unsigned short unicode;

#ifdef	__osf__
#include<alpha/endian.h>
#endif

#ifndef	DIRXML_EXPORT
#if defined(_MSC_VER)
//define these for Win32
#if	defined(DIRXML)
#define	DIRXML_EXPORT _declspec(dllexport)
#else
#define	DIRXML_EXPORT _declspec(dllimport)
#endif	//DIRXML
#else	//_MSC_VER
#define	DIRXML_EXPORT
#endif	//_MSC_VER
#endif	//!defined(DIRXML_EXPORT)

//use these namespaces so we don't have to 
//qualify everything 
#if defined(__cplusplus)
	using namespace DOM;
	using namespace SAX;

//parameter values for XmlDocument.getStream() endian parameter

#if defined(_AIX)

// AIX defines these screwy

#ifndef LITTLE_ENDIAN
#define LITTLE_ENDIAN	1234
#endif

#ifndef BIG_ENDIAN
#define	BIG_ENDIAN		4321
#endif

#elif !defined(linux) && !defined(__osf__) // on linux and tru64 these are already defined

	const int LITTLE_ENDIAN = 0;
	const int BIG_ENDIAN	= 1;
#endif
#else	// C

//parameter values for XmlDocument.getStream() endian parameter
#define LITTLE_ENDIAN	0
#define	BIG_ENDIAN		1

#endif	//if defined(__cplusplus)

//++
/*****************************************************************
Interface: XmlDocument
Desc:	Allows access to an XML document through three methods, a 
		DOM tree view, a serialized stream view, and a SAX event
		view.
Notes: 
*****************************************************************/
INTERFACE(XmlDocument)
//--
	//++
	//=========================================================================
	// Method:	getDocumentBytes
	// Desc:	Return the XML document as a serialized byte array
	// 
	// Notes:
	//			The encoding must be a standard XML encoding string:
	//				Example: "ASCII", "UTF-8", "UTF-16"
	//
	//			This will return 0 if there is no data or the desired encoding
	//			is not supported. "US-ASCII" shall always be supported.
	//=========================================================================
	METHOD(XmlDocument,getDocumentBytes,const unsigned char *)	//array of bytes
		METHOD_PARM1(const unicode * encoding)		//desired encoding
		METHOD_PARM(int endian)						//
		METHOD_PARM(int * length)					//pointer to a variable to receive length of array in bytes
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getDocument
	// Desc:	Return the XML document as a DOM tree.
	// 
	// Notes:	The returned Document object belongs to the XmlDocument object.
	//			No attempt must be made to delete it.
	//=========================================================================
	METHOD(XmlDocument,getDocument,Document *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getDocumentSAX
	// Desc:	Return a SAX Parser interface through which the caller can get
	//			a series of events describing the document.
	// 
	// Notes:	The caller must also call getDocumentInputSource() to get
	//			the InputSource for use with the Parser's parse() method.
	//=========================================================================
	METHOD(XmlDocument,getDocumentSAX,Parser *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getDocumentInputSource
	// Desc:	Return an InputSource object for use with the Parser returned
	//			from getDocumentSAX().
	// 
	// Notes:	The returned InputSource must only be used with the Parser
	//			returned from getDocumentSAX(). The InputSource belongs to
	//			the XMLDocument object and no attempt must be made to delete it.
	//=========================================================================
	METHOD(XmlDocument,getDocumentInputSource,InputSource *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	writeDocument
	// Desc:	Serialize the document to the passed OutputStream using
	//			default settings
	// 
	// Notes:
	//			outputStream may not be 0
	//			if encoding is zero, the default encoding will be used (UTF-8)
	//=========================================================================
	METHOD(XmlDocument, writeDocument, int)			//non-zero return indicates success
		METHOD_PARM1(OutputStream * outputStream)	//stream to which to write
		METHOD_PARM(const unicode * encoding)		//character encoding to use, or 0
		METHOD_PARM(int endian)						//0 for little-endian, 1 for big-endian
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getXmlWriter
	// Desc:	Return an XmlWriter interface that can be used to serialize
	//			the XML document.
	//
	//			The XmlWriter interface has methods for more closely controlling
	//			how the XML is output. For example, if the document is really
	//			just text, XML output escaping can be disabled.
	//
	//			After the various attributes have been set, calling to
	//			XmlWriter's write() method will cause the serialization to occur.
	// 
	// Notes:
	//			outputStream may not be 0
	//=========================================================================
	METHOD(XmlDocument, getXmlWriter, XmlWriter *)
		METHOD_PARM1(OutputStream * outputStream)	//stream to which to write
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	releaseXmlWriter
	// Desc:	Notify the XmlDocument object that the XmlWriter object
	//			returned from getXmlWriter may be freed.
	// 
	// Notes:
	//			This must be called before the OutputStream that was passed
	//			to getXmlWriter is released
	//=========================================================================
	METHOD(XmlDocument, releaseXmlWriter, void)
	END_METHOD
	//--

END_INTERFACE(XmlDocument)

//++
/*****************************************************************
Interface: XmlQueryProcessor
Desc:	Executes an XML-encoded query and returns the XML-encoded
		result. This is used for the SubscriptionShim to query NDS
		and for NDS to query the PublicationShim.
Notes: 
*****************************************************************/
INTERFACE(XmlQueryProcessor)
//--

	//++
	//=========================================================================
	// Method:	query
	// Desc:	Execute an XML-encoded query and return the XML-encoding result
	// 
	// Notes:	The passed query document is only valid during the context
	//			of the call to this method.
	//
	//			The returned result document is only valid until the next call
	//			to this method.
	//=========================================================================
	METHOD(Query, query, XmlDocument *)			//Result document or 0
		METHOD_PARM1(XmlDocument * document)	//Xml-encoded query
	END_METHOD
	//--

END_INTERFACE(XmlQueryProcessor)

//++
/*****************************************************************
Interface: XmlCommandProcessor
Desc:	Executes an XDS-encoded commandand returns the XDS-encoded
		result. This is used for NDS to send events to the
		SubscriptionShim and for the PublicationShim to send events to
		NDS.
Notes: 
*****************************************************************/
INTERFACE(XmlCommandProcessor)
//--

	//++
	//=========================================================================
	// Method:	execute
	// Desc:	Execute an XML-encoded request and return the XML-encoded result.
	// 
	// Notes:	The passed request document is only valid during the context
	//			of the call to this method.
	//
	//			The passed Query interface is only valid during the context of
	//			the call to this method.
	//
	//			The returned result document is only valid until the next call
	//			to this method.
	//=========================================================================
	METHOD(Execute, execute, XmlDocument *)				//result document or 0
		METHOD_PARM1(XmlDocument * document)			//request document
		METHOD_PARM(XmlQueryProcessor * queryInterface)	//Interface through which callee may query caller
	END_METHOD
	//--

END_INTERFACE(XmlCommandProcessor)

//++
/*****************************************************************
Interface: SubscriptionShim
Desc:	The interface through which NDS sends events to an
		application.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(SubscriptionShim, XmlCommandProcessor)
//--

	//++
	//=========================================================================
	// Method:	init
	// Desc:	Allow the shim to perform any necessary initialization before
	//			accepting requests.
	// 
	// Notes:	
	//=========================================================================
	METHOD(SubscriptionShim, init, XmlDocument *)	//result document or 0
		METHOD_PARM1(XmlDocument * shimParameters)	//shim initialization parms
	END_METHOD
	//--

END_INTERFACE(SubscriptionShim)

//++
/*****************************************************************
Interface: PublicationShim
Desc:	The interface through which an application sends events
		to NDS.
Notes: 
*****************************************************************/
INTERFACE(PublicationShim)
//--

	//++
	//=========================================================================
	// Method:	init
	// Desc:	Allow the shim to perform any necessary initialization before
	//			accepting requests.
	// 
	// Notes:	This will be called on a different thread than the 
	//			DriverShim::init() and SubscriptionShim::init() and 
	//			SubscriptionShim::execute() calls (but the same thread as
	//			PublicationShim::start())
	//=========================================================================
	METHOD(PublicationShim, init, XmlDocument *)	//result document or 0
		METHOD_PARM1(XmlDocument * shimParameters)	//shim initialization parms
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	start
	// Desc:	Pass control to the publisher shim so that the shim can start
	//			publishing application events.
	// 
	// Notes:	This method must not return until driver shutdown, unless it
	//			returns with a fatal error status.
	//			This will be called on a different thread than the 
	//			DriverShim::init() and SubscriptionShim::init() and 
	//			SubscriptionShim::execute() calls (but the same thread as
	//			PublicationShim::init())
	//=========================================================================
	METHOD(PublicationShim, start, XmlDocument *)			//result document or 0
			METHOD_PARM1(XmlCommandProcessor * ndsExecute)		//The NDS XmlCommandProcessor interface for publication
	END_METHOD
	//--

END_INTERFACE(PublicationShim)

//++
/*****************************************************************
Interface: DriverShim
Desc:	The abstraction of a DirXML driver for a single application.
Notes: 
*****************************************************************/
INTERFACE(DriverShim)
//--

	//++
	//=========================================================================
	// Method:	init
	// Desc:	Allow the driver to perform an shim-independent initialization.
	// 
	// Notes:	
	//=========================================================================
	METHOD(DriverShim, init, XmlDocument *)				//result document or 0
		METHOD_PARM1(XmlDocument * shimParameters)	//Parameters for shims
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	shutdown
	// Desc:	Command the driver to shutdown in preparation for being unloaded.
	// 
	// Notes:	
	//=========================================================================
	METHOD(DriverShim, shutdown, XmlDocument *)		//result document or 0
		METHOD_PARM1(XmlDocument * reason)		//XML-encoded reason for shutdown
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getSubscriptionShim
	// Desc:	Get a pointer to the SubscriptionShim interface from the 
	//			driver.
	// 
	// Notes:	Return must not be null.
	//=========================================================================
	METHOD(DriverShim, getSubscriptionShim, SubscriptionShim *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getPublicationShim
	// Desc:	Get a pointer to the PublicationShim interface from the 
	//			driver.
	// 
	// Notes:	Return must not be null.
	//=========================================================================
	METHOD(DriverShim, getPublicationShim, PublicationShim *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getSchema
	// Desc:	Get an XML representation of the application schema from the
	//			driver. This will not be called if init()/shutdown() are called
	//			for a driver instance, and vice-versa.
	// 
	// Notes:	
	//=========================================================================
	METHOD(DriverShim, getSchema, XmlDocument *)	//document containing schema description, or error return
		METHOD_PARM1(XmlDocument * initParameters)	//Parameters for (driver init, subscriber init, and publisher init params)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	destroy
	// Desc:	Free any resources allocated by the driver, including the
	//			driver object itself.
	// 
	// Notes:	This will be called after shutdown() (assuming init() succeeds)
	//=========================================================================
	METHOD(DriverShim, destroy, void)
	END_METHOD
	//--

END_INTERFACE(DriverShim)

//function prototype for DirXML driver .DLL entry point
//@return A pointer to a DriverShim object (if null, driver will
//be unloaded)
typedef DriverShim * (METHOD_CALL * CreateDriverProc)(void);

#if !defined(__cplusplus)
// macros for calling the interface from C

//XmlDocument
#define XmlDocument_getDocumentBytes(_this,enc,end,len)\
	(*_this->vTbl->getDocumentBytes)(_this,enc,end,len)

#define XmlDocument_getDocument(_this)\
	(*_this->vTbl->getDocument)(_this)

#define XmlDocument_getDocumentSAX(_this)\
	(*_this->vTbl->getDocumentSAX(_this)

#define XmlDocument_getDocumentInputSource(_this)\
	(*_this->vTbl->getDocumentInputSource(_this)

#define XmlDocument_writeDocument(_this,str,enc,end)\
	(*_this->vTbl->writeDocument)(_this,str,enc,end)

#define XmlDocument_getXmlWriter(_this,str)\
	(*_this->vTbl->getXmlWriter)(_this,str)

#define XmlDocument_releaseXmlWriter(_this)\
	(*_this->vTbl->releaseXmlWriter)(_this)

//Query
#define Query_query(_this,document)\
	(*this->vTbl->query)(_this,document)

//Execute
#define Execute_execute(_this,document)\
	(*this->vTbl->execute)(_this,document)

//SubscriptionShim
//	Execute-defined methods
#define SubscriptionShim_execute(_this,command)\
	(*((Execute *)this)->vTbl->execute)(_this,command)
//	SubscriptionShim-defined methods
#define SubscriptionShim_init(_this,parms)\
	(*this->vTbl->execute)(_this,parms)

//PublicationShim
#define PublicationShim_init(_this,parms)\
	(*this->vTbl->init)(_this,parms)

#define PublicationShim_start(_this,execute)\
	(*this->vTbl->start)(_this,execute)

//DriverShim
#define DriverShim_init(_this,parms)\
	(*this->vTbl->init)(_this,parms)

#define DriverShim_shutdown(_this,reason)\
	(*this->vTbl->shutdown)(_this,reason)

#define DriverShim_getSubscriber(_this)\
	(*this->vTbl->getSubscriber)(_this)

#define DriverShim_getPublisher(_this)\
	(*this->vTbl->getPublisher)(_this)


#endif



#endif //_NATIVE_INTERFACE_H_
