/*************************************************************************
Copyright  1999-2002 Novell, Inc. All Rights Reserved. 

THIS WORK IS SUBJECT TO U.S. AND INTERNATIONAL COPYRIGHT LAWS AND 
TREATIES. USE AND REDISTRIBUTION OF THIS WORK IS SUBJECT TO THE LICENSE
AGREEMENT ACCOMPANYING THE SOFTWARE DEVELOPMENT KIT (SDK) THAT CONTAINS
THIS WORK. PURSUANT TO THE SDK LICENSE AGREEMENT, NOVELL HEREBY GRANTS
TO DEVELOPER A ROYALTY-FREE, NON-EXCLUSIVE LICENSE TO INCLUDE NOVELL'S
SAMPLE CODE IN ITS PRODUCT. NOVELL GRANTS DEVELOPER WORLDWIDE DISTRIBUTION
RIGHTS TO MARKET, DISTRIBUTE, OR SELL NOVELL'S SAMPLE CODE AS A COMPONENT
OF DEVELOPER'S PRODUCTS. NOVELL SHALL HAVE NO OBLIGATIONS TO DEVELOPER OR
DEVELOPER'S CUSTOMERS WITH RESPECT TO THIS CODE.

***************************************************************************/
/*******************************************************************
* Name: dom.h
* Description: define a C/C++ interface for the Document Object Model
* level 1 core functionality
* Tabs: 4
*******************************************************************/
#ifndef _DOM_H_
#define	_DOM_H_

#include	"define_interface.h"

/*******************************************************************
	This is a C/C++ definition of the W3C Document Object Model for
	manipulating XML documents.

	See http://www.w3.org/TR/REC-DOM-Level-1/level-one-core.html for
	complete documentation.

	Note that the W3C has not defined a binding for the DOM to C or
	C++, so this probably won't interoperate with anyone other than
	DirXML's implementation.

	Because this must support C as well as C++ return values as well
	as exceptions are supported. To enable exceptions, the enableExceptions()
	method on the Document interface must be called with a non-zero
	parameter. All methods that are defined in the Java-binding with a
	void return are defined in this interface to return a pointer to
	a DOMException interface. If the return pointer is null, there was
	no error. If the return pointer is not null, the DOMException interface
	can be queried for the error. For all other methods, if the return
	pointer is null, there was an error and the getLastError() method
	of the Document interface can be called to receive a DOMException interface
	that can be queried for the error.

	Note that the interface and method names follow the Java-binding
	names exactly.

	For C++, the interfaces are in the namespace "DOM"
*********************************************************************/

//define a 16-bit unicode character (we can't use wchar_t because wchar_t
//is 32-bits on Solaris)
typedef unsigned short unicode;

#if	defined(__cplusplus)
namespace DOM
{
#endif


INTERFACE_FWD_DECL(Attr);
INTERFACE_FWD_DECL(CDATASection);
INTERFACE_FWD_DECL(Comment);
INTERFACE_FWD_DECL(Document);
INTERFACE_FWD_DECL(DocumentFragment);
INTERFACE_FWD_DECL(DocumentType);
INTERFACE_FWD_DECL(DOMException);
INTERFACE_FWD_DECL(DOMImplementation);
INTERFACE_FWD_DECL(Element);
INTERFACE_FWD_DECL(EntityReference);
INTERFACE_FWD_DECL(NamedNodeMap);
INTERFACE_FWD_DECL(Node);
INTERFACE_FWD_DECL(NodeList);
INTERFACE_FWD_DECL(ProcessingInstruction);
INTERFACE_FWD_DECL(Text);

//++
/*****************************************************************
Interface: Node
Desc:
	From the DOM spec:

	The <code>Node</code> interface is the primary datatype for the entire 
	Document Object Model. It represents a single node in the document tree. 
	While all objects implementing the <code>Node</code> interface expose 
	methods for dealing with children, not all objects implementing the 
	<code>Node</code> interface may have children. For example, 
	<code>Text</code> nodes may not have children, and adding children to such 
	nodes results in a <code>DOMException</code> being raised.  
	<p>The attributes <code>nodeName</code>, <code>nodeValue</code>  and 
	<code>attributes</code> are  included as a mechanism to get at node 
	information without  casting down to the specific derived interface. In 
	cases where  there is no obvious mapping of these attributes for a specific
	<code>nodeType</code> (e.g., <code>nodeValue</code> for an Element  or 
	<code>attributes</code>  for a Comment), this returns <code>null</code>. 
	Note that the  specialized interfaces may contain additional and more 
	convenient mechanisms to get and set the relevant information.
Notes: 
	This C/C++ binding adds the following methods to the DOM
	level 1 specified methods:
	
		void enableExceptions(int enable);
		DOMException * getLastError();
		void destroy();
*****************************************************************/
INTERFACE(Node)
//--

	//++
	//=========================================================================
	// Method:	getNodeName
	// Desc:	Return the name of this node.
	// 
	// Notes:	Not all node types have a name
	//=========================================================================
	METHOD(Node,getNodeName,const unicode *)	//unicode name of node or 0
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getNodeValue
	// Desc:	Return the value of this node
	// 
	// Notes:
	//=========================================================================
	METHOD(Node,getNodeValue,const unicode *)	//unicode value of the node or 0
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	setNodeValue
	// Desc:	Set the value of this node.
	// 
	//			DOMException returned or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	// Notes:	Will return an error if the node is readonly
	//=========================================================================
	METHOD(Node,setNodeValue,DOMException *)
		METHOD_PARM1(const unicode * value)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getNodeType
	// Desc:	Return the type of this node, see enum NodeType.
	// 
	// Notes:
	//=========================================================================
	METHOD(Node, getNodeType,short)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getParentNode
	// Desc:	Return the parent of this node. All nodes, except Document,
	//			DocumentFragment, and Attr may have a parent. However, if a 
	//			node has been created but not yet added to the tree this is
	//			null.
	// 
	// Notes:
	//=========================================================================
	METHOD(Node, getParentNode, Node *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getChildNodes
	// Desc:	Return a NodeList with the children of this node. If the node
	//			has no children this return an empty NodeList.
	// 
	// Notes:	The returned NodeList must be deleted when it is no longer needed.
	//			This must be done by calling the NodeList::destroy() method.
	//
	//			If this Node is deleted before the NodeList is destroyed, the
	//			NodeList becomes invalid and results are unpredictable.
	//=========================================================================
	METHOD(Node, getChildNodes, NodeList *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getFirstChild
	// Desc:	Return the first child of this node. If there is no such node
	//			the return is null.
	// 
	// Notes:
	//=========================================================================
	METHOD(Node, getFirstChild, Node *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getLastChild
	// Desc:	Return the last child of this node. If there is no such node
	//			the return is null.
	// 
	// Notes:
	//=========================================================================
	METHOD(Node, getLastChild, Node *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getPreviousSibling
	// Desc:	Return the node immediately preceding this node. If there is no
	//			such node the return is null.
	// 
	// Notes:
	//=========================================================================
	METHOD(Node, getPreviousSibling, Node *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getNextSibling
	// Desc:	Return the node immediately following this node. If there is no
	//			such node the return is null.
	// 
	// Notes:
	//=========================================================================
	METHOD(Node, getNextSibling, Node *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getAttributes
	// Desc:	Return a NamedNodeMap with a list of the attributes of this
	//			node (if it is an Element), or null otherwise.
	// 
	// Notes:	Note that the returned NamedNodeMap belongs to this node. No
	//			attempt must be made to delete it.
	//=========================================================================
	METHOD(Node, getAttributes, NamedNodeMap *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getOwnerDocument
	// Desc:	Return the Document object associated with this node. This is
	//			also the Document object used to create new nodes. When this
	//			node is a Document this returns null.
	// 
	// Notes:
	//=========================================================================
	METHOD(Node, getOwnerDocument, Document *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	insertBefore
	// Desc:	Inserts the node newChild before the existing child node 
	//			refChild. If refChild is null, insert newChild at the end of
	//			the list of children.
	//
	//			If newChild is a DocumentFragment object, all of its children
	//			are inserted, in the same order, before refChild. if the 
	//			newChild is already in the tree, it is first removed.
	//
	//			DOMException registered or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	//				WRONG_DOCUMENT_ERR if this Node is of a type that does not
	//				allow children of the newChild type, or if newChild is one
	//				of this Node's ancestors
	//				HIERARCHY_REQUEST_ERR if newChild is of a type that is not
	//				allowed as a child of this Node.
	//				NOT_FOUND_ERR if refChild is not a child of this Node
	// 
	// Notes:
	//=========================================================================
	METHOD(Node, insertBefore, Node *)
		METHOD_PARM1(Node * newChild)
		METHOD_PARM(Node * refChild)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	replaceChild
	// Desc:	Replaces the child node oldChild with the newChild in the list
	//			of children and returns the oldChild node. If the newChild is
	//			already in the tree, it is first removed.
	//
	//			DOMException registered or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	//				WRONG_DOCUMENT_ERR if this Node is of a type that does not
	//				allow children of the newChild type, or if newChild is one
	//				of this Node's ancestors
	//				HIERARCHY_REQUEST_ERR if newChild is of a type that is not
	//				allowed as a child of this Node.
	//				NOT_FOUND_ERR if refChild is not a child of this Node
	// 
	// Notes:	C/C++ specific: The destroy() method must be called on the
	//			returned node.
	//=========================================================================
	METHOD(Node, replaceChild, Node *)
		METHOD_PARM1(Node * newChild)
		METHOD_PARM(Node * oldChild)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	removeChild
	// Desc::	Removes the child indicated by oldChild from the list of 
	//			children and returns it.
	//
	//			DOMException registered or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	//				NOT_FOUND_ERR if refChild is not a child of this Node
	// 
	// Notes:	C/C++ specific: The destroy() method must be called on the
	//			returned node.
	//=========================================================================
	METHOD(Node, removeChild, Node *)
		METHOD_PARM1(Node * oldChild)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	appendChild
	// Desc:	Adds the node newChild to the end of the list of children of
	//			this node. If newChild is already in the tree, it is first
	//			removed.
	//
	//			DOMException registered or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	//				WRONG_DOCUMENT_ERR if this Node is of a type that does not
	//				allow children of the newChild type, or if newChild is one
	//				of this Node's ancestors
	//				HIERARCHY_REQUEST_ERR if newChild is of a type that is not
	//				allowed as a child of this Node.
	// 
	// Notes:
	//=========================================================================
	METHOD(Node, appendChild, Node *)
		METHOD_PARM1(Node * newChild)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	hasChildNodes
	// Desc:	This is a convenience method to allow easy determination of
	//			whether a node has any children. Returns non-zero if this node
	//			has children.
	// 
	// Notes:
	//=========================================================================
	METHOD(Node, hasChildNodes, int)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	cloneNode
	// Desc:	Returns a duplicate of this node, i.e., serves as a generic
	//			copy constructor for nodes. The duplicate node has no parent
	//			(getParentNode() returns null).
	//
	//			Cloning an Element copies all attributes and their values,
	//			including those generated by an XML processor to represent
	//			defaulted attributes, but this method does not copy any text
	//			it contains unless it is a deep clone, since the text is 
	//			contained in a child Text node. Cloning any other type of node
	//			simply returns a copy of this node.
	// 
	// Notes:
	//=========================================================================
	METHOD(Node, cloneNode, Node *)
		METHOD_PARM1(int deep)		//if non-zero recursively clone the subtree under this node
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getLastError
	// Desc:	Return the DOMException generated by the last operation on this
	//			node. If no exception was generated, return null.
	// 
	// Notes:	C Extension to DOM level 1
	//			The DOMException returned by this method belongs to the Node.
	//			No attempt should be made to delete it.
	//=========================================================================
	METHOD(Node, getLastError, DOMException *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	destroy
	// Desc:	Delete this node. If this node is currently in the tree, then
	//			this is equivalent to calling removeChild() on this node's
	//			parent then destroy() on this node. If this node is a Document
	//			object then the entire tree is destroyed.
	// 
	// Notes:	C/C++ extension to DOM level 1
	//=========================================================================
	METHOD(Node, destroy, void)
	END_METHOD
	//--

#if defined(__cplusplus)
	//These types the return values for Node.getNodeType()
	enum NodeType
	{
		ELEMENT_NODE				= 1,
		ATTRIBUTE_NODE				= 2,
		TEXT_NODE					= 3,
		CDATA_SECTION_NODE			= 4,
		ENTITY_REFERENCE_NODE		= 5,
		ENTITY_NODE					= 6,
		PROCESSING_INSTRUCTION_NODE	= 7,
		COMMENT_NODE				= 8,
		DOCUMENT_NODE				= 9,
		DOCUMENT_TYPE_NODE			= 10,
		DOCUMENT_FRAGMENT_NODE		= 11,
		NOTATION_NODE				= 12
	};

#endif

END_INTERFACE(Node)

#if !defined(__cplusplus)
//These types the return values for Node.getNodeType()
enum NodeType
{
	ELEMENT_NODE				= 1,
	ATTRIBUTE_NODE				= 2,
	TEXT_NODE					= 3,
	CDATA_SECTION_NODE			= 4,
	ENTITY_REFERENCE_NODE		= 5,
	ENTITY_NODE					= 6,
	PROCESSING_INSTRUCTION_NODE	= 7,
	COMMENT_NODE				= 8,
	DOCUMENT_NODE				= 9,
	DOCUMENT_TYPE_NODE			= 10,
	DOCUMENT_FRAGMENT_NODE		= 11,
	NOTATION_NODE				= 12
};
#endif	//#if !defined(__cplusplus)

//++
/*****************************************************************
Interface: Document
Desc:
	From the DOM spec:

	The <code>Document</code> interface represents the entire HTML or XML 
	document. Conceptually, it is the root of the document tree, and provides 
	the  primary access to the document's data.
	<p>Since elements, text nodes, comments, processing instructions, etc. 
	cannot exist outside the context of a <code>Document</code>, the 
	<code>Document</code> interface also contains the factory methods needed 
	to create these objects.  The <code>Node</code> objects created have a 
	<code>ownerDocument</code> attribute which associates them with the 
	<code>Document</code> within whose  context they were created.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(Document,Node)
//--

	//++
	//=========================================================================
	// Method:	getDoctype
	// Desc:	Returns the DocumentType node associated with this document.
	//			For documents without a Document Type Declaration this returns
	//			null.
	//
	//			The DOM level does not support editing the Document Type
	//			Declaration, therefore the DocumentType cannot be altered in
	//			any way.
	// 
	// Notes:
	//=========================================================================
	METHOD(Document, getDoctype, DocumentType *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getImplementation
	// Desc:	Return the DOMImplementation object that handles this document.
	//			A DOM application may use objects from multiple implementations.
	// 
	// Notes:
	//=========================================================================
	METHOD(Document, getImplementation, DOMImplementation *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getDocumentElement
	// Desc:	This is a convenience method that allows direct acess to the
	//			child node that is the root element of the document. If there
	//			is more than one element as a child of the Document object then
	//			the first element child is returned. If there is no document
	//			element, null is returned.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Document, getDocumentElement, Element *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	createElement
	// Desc:	Creates an element of the type specified. Note that the 
	//			instance returned implements the Element interface, so
	//			attributes can be specified directly on the returned object.
	//
	//			DOMException registered or thrown:
	//				INVALID_CHARACTER_ERROR: If tagName is invalid
	// 
	// Notes:	
	//=========================================================================
	METHOD(Document, createElement, Element *)
		METHOD_PARM1(const unicode * tagName)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	createDocumentFragment
	// Desc:	Creates an empty DocumentFragment object.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Document, createDocumentFragment, DocumentFragment *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	createTextNode
	// Desc:	Creates a Text node given the specified string.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Document, createTextNode, Text *)
		METHOD_PARM1(const unicode * data)
		METHOD_PARM(int offset)
		METHOD_PARM(int length)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	createComment
	// Desc:	Creates a Comment node given the specified string.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Document, createComment, Comment *)
		METHOD_PARM1(const unicode * data)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	createCDATASection
	// Desc:	Creates a CDATASection node whose value is the specified string.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Document, createCDATASection, CDATASection *)
		METHOD_PARM1(const unicode * data)
		METHOD_PARM(int offset)
		METHOD_PARM(int length)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	createProcessingInstruction
	// Desc:	Creates a ProcessingInstruction node given the specified
	//			target and data strings.
	//
	//			DOMException registered or thrown:
	//				INVALID_CHARACTER_ERROR: If target or data is invalid
	// 
	// Notes:	
	//=========================================================================
	METHOD(Document, createProcessingInstruction, ProcessingInstruction *)
		METHOD_PARM1(const unicode * target)
		METHOD_PARM(const unicode * data)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	createAttribute
	// Desc:	Creates an Attr of the given name. Note that the Attr instance
	//			can then be set on an Element using the setAttribute() method.
	//
	//			DOMException registered or thrown:
	//				INVALID_CHARACTER_ERROR: If name is invalid
	// 
	// Notes:	
	//=========================================================================
	METHOD(Document, createAttribute, Attr *)
		METHOD_PARM1(const unicode * name)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	createEntityReference
	// Desc:	Creates an EntityReference object.
	//
	//			DOMException registered or thrown:
	//				INVALID_CHARACTER_ERROR: If name is invalid
	// 
	// Notes:	
	//=========================================================================
	METHOD(Document, createEntityReference, EntityReference *)
		METHOD_PARM1(const unicode * name)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getElementsByTagName
	// Desc:	Returns a NodeList of all the Elements with a given tag name
	//			in the order in which they would be encountered in a preorder
	//			traversal of the Document tree.
	// 
	// Notes:	The returned NodeList must be deleted when it is no longer needed.
	//			This must be done by calling the NodeList::destroy() method.
	//
	//			If this Node is deleted before the NodeList is destroyed, the
	//			NodeList becomes invalid and results are unpredictable.
	//=========================================================================
	METHOD(Document, getElementsByTagName, NodeList *)
		METHOD_PARM1(const unicode * tagName)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	enableExceptions
	// Desc:	Enable the throwing of C++ exceptions for all nodes created
	//			by this Document and for all nodes in this Document object's
	//			tree.
	// 
	// Notes:	C++ Extension to DOM level 1
	//			A C implementation of the DOM will not support this and will
	//			always return 0, indicating exceptions are not supported.
	//=========================================================================
	METHOD(Node, enableExceptions, int)
		METHOD_PARM1(int enable)
	END_METHOD
	//--

END_INTERFACE(Document)

//++
/*****************************************************************
Interface: DocumentFragment
Desc:
	From the DOM spec:

	<code>DocumentFragment</code> is a "lightweight" or "minimal" 
	<code>Document</code> object. It is very common to want to be able to 
	extract a portion of a document's tree or to create a new fragment of a 
	document. Imagine implementing a user command like cut or rearranging a 
	document by moving fragments around. It is desirable to have an object 
	which can hold such fragments and it is quite natural to use a Node for 
	this purpose. While it is true that a <code>Document</code> object could 
	fulfil this role,  a <code>Document</code> object can potentially be a 
	heavyweight  object, depending on the underlying implementation. What is 
	really needed for this is a very lightweight object.  
	<code>DocumentFragment</code> is such an object.
	<p>Furthermore, various operations -- such as inserting nodes as children 
	of another <code>Node</code> -- may take <code>DocumentFragment</code> 
	objects as arguments;  this results in all the child nodes of the 
	<code>DocumentFragment</code>  being moved to the child list of this node.
	<p>The children of a <code>DocumentFragment</code> node are zero or more 
	nodes representing the tops of any sub-trees defining the structure of the 
	document. <code>DocumentFragment</code> nodes do not need to be 
	well-formed XML documents (although they do need to follow the rules 
	imposed upon well-formed XML parsed entities, which can have multiple top 
	nodes).  For example, a <code>DocumentFragment</code> might have only one 
	child and that child node could be a <code>Text</code> node. Such a 
	structure model  represents neither an HTML document nor a well-formed XML 
	document.  
	<p>When a <code>DocumentFragment</code> is inserted into a  
	<code>Document</code> (or indeed any other <code>Node</code> that may take 
	children) the children of the <code>DocumentFragment</code> and not the 
	<code>DocumentFragment</code>  itself are inserted into the 
	<code>Node</code>. This makes the <code>DocumentFragment</code> very 
	useful when the user wishes to create nodes that are siblings; the 
	<code>DocumentFragment</code> acts as the parent of these nodes so that the
	 user can use the standard methods from the <code>Node</code>  interface, 
	such as <code>insertBefore()</code> and  <code>appendChild()</code>.  
Notes: 
*****************************************************************/
DERIVED_INTERFACE(DocumentFragment, Node)
//--
	//no additional methods

END_INTERFACE(DocumentFragment)

//++
/*****************************************************************
Interface: Element
Desc:
	From the DOM spec:

	By far the vast majority of objects (apart from text) that authors 
	encounter when traversing a document are <code>Element</code> nodes.  
	Assume the following XML document:&lt;elementExample id="demo"&gt; 
	&lt;subelement1/&gt; 
	&lt;subelement2&gt;&lt;subsubelement/&gt;&lt;/subelement2&gt;
	&lt;/elementExample&gt;  
	<p>When represented using DOM, the top node is an <code>Element</code> node 
	for "elementExample", which contains two child <code>Element</code> nodes, 
	one for "subelement1" and one for "subelement2". "subelement1" contains no 
	child nodes.
	<p>Elements may have attributes associated with them; since the 
	<code>Element</code> interface inherits from <code>Node</code>, the generic
	 <code>Node</code> interface method <code>getAttributes</code> may be used 
	to retrieve the set of all attributes for an element.  There are methods on
	 the <code>Element</code> interface to retrieve either an <code>Attr</code>
	 object by name or an attribute value by name. In XML, where an attribute 
	value may contain entity references, an <code>Attr</code> object should be 
	retrieved to examine the possibly fairly complex sub-tree representing the 
	attribute value. On the other hand, in HTML, where all attributes have 
	simple string values, methods to directly access an attribute value can 
	safely be used as a convenience.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(Element,Node)
//--

	//++
	//=========================================================================
	// Method:	getTagName
	// Desc:	Return the name of the element.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Element, getTagName, const unicode *)	//return tag name of element
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getAttribute
	// Desc:	Return an attribute value by name. If the element does not
	//			have an attribute with the passed name null is returned.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Element, getAttribute, const unicode *)	//return value of attribute
		METHOD_PARM1(const unicode * name)			//the name of the attribute
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	setAttribute
	// Desc:	Adds a new attribute. If an attribute with that name is already
	//			present in the element, its value is changed to be that of the
	//			value parameter. This value is a simple string. it is not parsed
	//			as it is being set. So any markup (such as syntax to be 
	//			recognized an an entity reference) is treated as literal text
	//			and needs to be appropriately escaped by the implementation
	//			when it is written out. In order to assign an attribute value
	//			that contains entity references, the user must create an Attr
	//			node plus any Text and EntityReference nodes, build the
	//			appropriate sub tree, and use setAttributeNode() to assign it
	//			as the value of an attribute.
	//
	//			DOMException returned or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	//				INVALID_CHARACTER_ERR if name is invalid
	// 
	// Notes:	
	//=========================================================================
	METHOD(Element, setAttribute, DOMException *)
		METHOD_PARM1(const unicode * name)		//the name of the attribute
		METHOD_PARM(const unicode * value)		//the value of the attribute
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	removeAttribute
	// Desc:	Removes an attribute by name. If the removed attribute has a
	//			default value it is immediately replaced.
	//
	//			DOMException returned or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	// 
	// Notes:	
	//=========================================================================
	METHOD(Element, removeAttribute, DOMException *)
		METHOD_PARM1(const unicode * name)		//the name of the attribute
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getAttributeNode
	// Desc:	Retrieves an Attr node by name. This returns null is there is
	//			no attribute by the passed name.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Element, getAttributeNode, Attr *)
		METHOD_PARM1(const unicode * name)		//the name of the attribute
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	setAttributeNode
	// Desc:	Adds a new attribute. If an attribute with that name is already
	//			present in the element, it is replaced by the new one.
	//
	//			DOMException registered or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	//				WRONG_DOCUMENT_ERR if this Node is of a type that does not
	//				allow children of the newChild type, or if newChild is one
	//				of this Node's ancestors
	//				INUSE_ATTRIBUTE_ERR if newAttr is an Attr that already belongs
	//				to another Element.
	// 
	// Notes:	If the new Attr replaces an existing Attr node the previously
	//			existing Attr node is returned (and must be destroyed), otherwise
	//			null is returned.
	//=========================================================================
	METHOD(Element, setAttributeNode, Attr *)	
		METHOD_PARM1(Attr * newAttr)			//the new Attr node
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	removeAttributeNode
	// Desc:	Removes the specified attribute.
	//
	//			DOMException registered or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	//				NOT_FOUND_ERR if oldAttr is not an attribute of this Node
	// 
	// Notes:	
	//=========================================================================
	METHOD(Element, removeAttributeNode, Attr *)
		METHOD_PARM1(Attr * oldAttr)			//the Attr node to remove
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getElementsByTagName
	// Desc:	Returns a NodeList of all descendant elements with a given
	//			tag name, in the order in which they would be encountered in a
	//			preorder traversal of the Element tree.
	// 
	// Notes:	The special tag name value "*" matches all tags.
	//			The returned NodeList must be deleted when it is no longer needed.
	//			This must be done by calling the NodeList::destroy() method.
	//
	//			If this Node is deleted before the NodeList is destroyed, the
	//			NodeList becomes invalid and results are unpredictable.
//=========================================================================
	METHOD(Document, getElementsByTagName, NodeList *)
		METHOD_PARM1(const unicode * tagName)	//the name of the tag to match on.
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	normalize
	// Desc:	Puts all Text nodes in the full depth of the sub-tree underneath
	//			this Element into a "normal" form where only markup (e.g., tags,
	//			comments, processing instructions, CDATA sections, and entity
	//			references separates Text nodes, i.e., there are no adjacent
	//			Text nodes. This can be used to ensure that the DOM view of a
	//			document is the same if it were saved and re-loaded, and is 
	//			useful when operations (such as XPointer lookups) that depend on
	//			a particular document tree structure are to be used.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Element, normalize, DOMException *)
	END_METHOD
	//--

END_INTERFACE(Element)

//++
/*****************************************************************
Interface: Attr
Desc:
	From the DOM spec:

	The <code>Attr</code> interface represents an attribute in an 
	<code>Element</code> object.Typically the allowable values for the 
	attribute are defined in a documenttype definition.
	<p><code>Attr</code> objects inherit the <code>Node</code>  interface, but 
	since they are not actually child nodes of the element  they describe, the 
	DOM does not consider them part of the document  tree.  Thus, the 
	<code>Node</code> attributes <code>parentNode</code>,  
	<code>previousSibling</code>, and <code>nextSibling</code> have a  null 
	value for <code>Attr</code> objects. The DOM takes the  view that 
	attributes are properties of elements rather than having a  separate 
	identity from the elements they are associated with;  this should make it 
	more efficient to implement such features as default attributes associated 
	with all elements of a  given type.  Furthermore, <code>Attr</code> nodes 
	may not be immediate children of a <code>DocumentFragment</code>. However, 
	they can be associated with <code>Element</code> nodes contained within a 
	<code>DocumentFragment</code>. In short, users and implementors of the DOM 
	need to be aware that  <code>Attr</code> nodes have some things in  common 
	with other objects inheriting the <code>Node</code> interface,  but they 
	also are quite distinct.
	<p> The attribute's effective value is determined as follows: if this  
	attribute has been explicitly assigned any value, that value is the  
	attribute's effective value; otherwise, if there is a declaration for  
	this attribute, and that declaration includes a default value, then  that 
	default value is the attribute's effective value; otherwise, the  
	attribute does not exist on this element in the structure model until  it 
	has been explicitly added.  Note that the <code>nodeValue</code>  
	attribute on the <code>Attr</code> instance can also be used to retrieve 
	the string version of the attribute's value(s). 
	<p>In XML, where the value of an attribute can contain entity references, 
	the child nodes of the <code>Attr</code> node provide a representation in 
	which entity references are not expanded. These child nodes may be either 
	<code>Text</code> or <code>EntityReference</code> nodes. Because the 
	attribute type may be unknown, there are no tokenized attribute values.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(Attr,Node)
//--
	
	//++
	//=========================================================================
	// Method:	getName
	// Desc:	Returns the name of this attribute.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Attr, getName, const unicode *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getSpecified
	// Desc:	If this attribute was explicitly given a value in the original
	//			document this is TRUE; otherwise it is FALSE. Note that the
	//			implementation is in charge of this attribute, not the user. If
	//			the user changes the value of the attribute (even if it ends up
	//			having the same value as the default value) then the specified
	//			flag is automatically flipped to TRUE. To re-specify the 
	//			attribute as the default value from the DTD, the user must delete
	//			the attribute. The implementation will then make a new attribute
	//			available with the specified flag set to false and the default
	//			value (if one exists).
	// 
	// Notes:	
	//=========================================================================
	METHOD(Attr, getSpecified, int)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getValue
	// Desc:	On retrieval, the value of the attribute is returned as a 
	//			string. Character and general entity references are replaced
	//			with their values.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Attr, getValue, const unicode *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	setValue
	// Desc:	On setting, this creates a Text node with the unparsed contents
	//			of the string.
	//
	//			DOMException returned or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	// 
	// Notes:	
	//=========================================================================
	METHOD(Attr, setValue, DOMException *)
		METHOD_PARM1(const unicode * value)		//the new attribute value
	END_METHOD
	//--

END_INTERFACE(Attr)

//++
/*****************************************************************
Interface: CharacterData
Desc:
	From the DOM spec:

	The <code>CharacterData</code> interface extends Node with a set  of 
	attributes and methods for accessing character data in the DOM.  For 
	clarity this set is defined here rather than on each object that uses 
	these attributes and methods. No DOM objects correspond directly to 
	<code>CharacterData</code>, though <code>Text</code> and others do inherit 
	the interface from it. All <code>offset</code>s in this interface start 
	from 0.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(CharacterData,Node)
//--

	//++
	//=========================================================================
	// Method:	getData
	// Desc:	Return the character data of the node that implements this
	//			interface. The DOM implementation may not put arbitrary limits
	//			on the amount of data that may be stored in a CharacterData
	//			node. However, implementation limits may mean that the 
	//			entirety of a node's data may not fit into a single string. In
	//			such cases the user may call substringData to retrieve the data
	//			in appropriately sized pieces.
	// 
	// Notes:	
	//=========================================================================
	METHOD(CharacterData, getData, const unicode *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	setData
	// Desc:	Set the character data of the node that implements this interface.
	//
	//			DOMException returned or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	// 
	// Notes:	
	//=========================================================================
	METHOD(CharacterData, setData, DOMException *)
		METHOD_PARM1(const unicode * data)		//The character data for this node
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getLength
	// Desc:	The number of characters that are available through getData()
	//			and the substringData() method below. This may have the value
	//			zero, i.e., CharacterData nodes may be empty
	// 
	// Notes:	
	//=========================================================================
	METHOD(CharacterData, getLength, int)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	substringData
	// Desc:	Extracts a range of data from the node.
	//
	//			DOMException registered or thrown:
	//				INDEX_SIZE_ERR if the specified offset is negative or greater 
	//				than the number of characters in data, or if the 
	//				specified count is negative.
	// 
	// Notes:	
	//=========================================================================
	METHOD(CharacterData, substringData, const unicode *)
		METHOD_PARM1(int offset)		//Start offset of substring to extract
		METHOD_PARM(int count)			//The number of characters to extract
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	appendData
	// Desc:	Append the string to the end of the character data of the node.
	//
	//			DOMException returned or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	// 
	// Notes:	
	//=========================================================================
	METHOD(CharacterData, appendData, DOMException *)
		METHOD_PARM1(const unicode * arg)	//The string to append
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	insertData
	// Desc:	Insert a string at the specified character offset.
	//
	//			DOMException returned or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	//				INDEX_SIZE_ERR if the specified offset is negative or greater 
	//				than the number of characters in data
	// 
	// Notes:	
	//=========================================================================
	METHOD(CharacterData, insertData, DOMException *)
		METHOD_PARM1(int offset)			//The character offset at which to insert
		METHOD_PARM(const unicode * arg)	//The string to insert
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	deleteData
	// Desc:	Remove a range of characters from the node.
	//
	//			DOMException returned or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	//				INDEX_SIZE_ERR if the specified offset is negative or greater 
	//				than the number of characters in data, or if the 
	//				specified count is negative.
	// 
	// Notes:	
	//=========================================================================
	METHOD(CharacterData, deleteData, DOMException *)
		METHOD_PARM1(int offset)			//The offset from which to remove characters
		METHOD_PARM(int count)				//The number of characters to delete.
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	replaceData
	// Desc:	Replace the characters starting at the specified character
	//			offset within the specified string.
	//
	//			DOMException returned or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	//				INDEX_SIZE_ERR if the specified offset is negative or greater 
	//				than the number of characters in data, or if the 
	//				specified count is negative.
	// 
	// Notes:	
	//=========================================================================
	METHOD(CharacterData, replaceData, DOMException *)
		METHOD_PARM1(int offset)			//The offset from which to start replacing
		METHOD_PARM(int count)				//The number of characters to replace
		METHOD_PARM(const unicode * arg)	//The string with which the range must be replaced
	END_METHOD
	//--

END_INTERFACE(CharacterData)

//++
/*****************************************************************
Interface: Text
Desc:
	From the DOM spec:

	The <code>Text</code> interface represents the textual content (termed 
	character  data in XML) of an <code>Element</code> or <code>Attr</code>.  
	If there is no markup inside an element's content, the text is contained 
	in a single object implementing the <code>Text</code> interface that is 
	the only child of the element. If there is markup, it is parsed into a 
	list of elements and <code>Text</code> nodes that form the list of 
	children of the element.
	<p>When a document is first made available via the DOM, there is  only one 
	<code>Text</code> node for each block of text. Users may create  adjacent 
	<code>Text</code> nodes that represent the  contents of a given element 
	without any intervening markup, but should be aware that there is no way 
	to represent the separations between these nodes in XML or HTML, so they 
	will not (in general) persist between DOM editing sessions. The 
	<code>normalize()</code> method on <code>Element</code> merges any such 
	adjacent <code>Text</code> objects into a single node for each block of 
	text; this is  recommended before employing operations that depend on a 
	particular document structure, such as navigation with 
	<code>XPointers.</code> 
Notes: 
*****************************************************************/
DERIVED_INTERFACE(Text,CharacterData)
//--

	//++
	//=========================================================================
	// Method:	splitText
	// Desc:	Breaks this Text node into two Text nodes at the specified
	//			offset, keeping both in the tree as siblings. This node then
	//			only contains all the content up to the offset point. And a new
	//			Text node, which is inserted as the next sibling of this node,
	//			contains all the content at and after the offset point.
	//
	//			DOMException registered or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	//				INDEX_SIZE_ERR if the specified offset is negative or greater 
	//				than the number of characters in data
	//
	// Notes:	
	//=========================================================================
	METHOD(Text, splitText, Text *)
		METHOD_PARM1(int offset)		//The offset at which to split, starting from 0
	END_METHOD
	//--

END_INTERFACE(Text)

//++
/*****************************************************************
Interface: CDATASection
Desc:
	From the DOM spec:

	CDATA sections are used to escape blocks of text containing  characters 
	that would otherwise be regarded as markup. The only  delimiter that is 
	recognized in a CDATA section is the "]]&gt;" string  that ends the CDATA 
	section. CDATA sections can not be  nested. The primary purpose is for 
	including material such as XML fragments, without needing to escape all 
	the delimiters.
	<p>The <code>DOMString</code> attribute of the <code>Text</code> node holds 
	the text that is contained by the CDATA section. Note that this may 
	contain characters that need to be escaped outside of CDATA sections and 
	that, depending on the character encoding ("charset") chosen for 
	serialization, it may be impossible to write out some characters as part 
	of a CDATA section.
	<p> The <code>CDATASection</code> interface inherits the 
	<code>CharacterData</code> interface through the <code>Text</code> 
	interface. Adjacent <code>CDATASections</code> nodes are not merged by use 
	of the Element.normalize() method.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(CDATASection, Text)
//--
	//no additional methods
END_INTERFACE(CDATASection)

//++
/*****************************************************************
Interface: Comment
Desc:
	From the DOM spec:

	This represents the content of a comment, i.e., all the characters between 
	the starting '<code>&lt;!--</code>' and ending '<code>--&gt;</code>'. Note 
	that this is the definition of a comment in XML, and, in practice, HTML, 
	although some HTML tools may implement the full SGML comment structure.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(Comment, CharacterData)
//--
	//no additional methods

END_INTERFACE(Comment)

//++
/*****************************************************************
Interface: ProcessingInstruction
Desc:
	From the DOM spec:

	The <code>ProcessingInstruction</code> interface represents a  "processing 
	instruction", used in XML as a way to keep processor-specific information 
	in the text of the document.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(ProcessingInstruction, Node)
//--
	
	//++
	//=========================================================================
	// Method:	getTarget
	// Desc:	Return the target of this ProcessingInstruction node. XML
	//			defines this as being the first token following the markup
	//			that begins the processing instruction.
	// 
	// Notes:	
	//=========================================================================
	METHOD(ProcessingInstruction, getTarget, const unicode *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getData
	// Desc:	Return the content of this ProcessingInstruction node. This is
	//			from the first non white space character after the target to
	//			the character immediately preceding the ?>.
	// 
	// Notes:	
	//=========================================================================
	METHOD(ProcessingInstruction, getData, const unicode *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	setData
	// Desc:	Set the content of this ProcessingInstruction node. This is
	//			from the first non white space character after the target to
	//			the character immediately preceding the ?>.
	//
	//			DOMException returned or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if Node is read only
	// 
	// Notes:	
	//=========================================================================
	METHOD(ProcessingInstruction, setData, DOMException *)
		METHOD_PARM1(const unicode * data)
	END_METHOD
	//--

END_INTERFACE(ProcessingInstruction)

//++
/*****************************************************************
Interface: DocumentType
Desc:
	From the DOM spec:

	Each <code>Document</code> has a <code>doctype</code> attribute whose value 
	is either <code>null</code> or a <code>DocumentType</code> object. The 
	<code>DocumentType</code> interface in the DOM Level 1 Core provides an 
	interface to the list of entities that are defined for the document, and 
	little else because the effect of namespaces and the various XML scheme 
	efforts on DTD representation are not clearly understood as of this 
	writing.
	<p>The DOM Level 1 doesn't support editing <code>DocumentType</code> nodes.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(DocumentType, Node)
//--

	//++
	//=========================================================================
	// Method:	getName
	// Desc:	The name of the DTD; i.e., the name immediately following the
	//			DOCTYPE keyword.
	// 
	// Notes:	
	//=========================================================================
	METHOD(DocumentType, getName, const unicode *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getEntities
	// Desc:	Return a NamedNodeMap containing the general entities, both
	//			external and internal, declared in the DTD. Duplicates are
	//			discarded. Every Node in this map also implements the Entity
	//			interface.
	// 
	// Notes:	Note that the returned NamedNodeMap belongs to this node. No
	//			attempt must be made to delete it.
	//=========================================================================
	METHOD(DocumentType, getEntities, NamedNodeMap *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getNotations
	// Desc:	Return a NamedNodeMap containing the notations declared in the
	//			DTD. Duplicates are discarded. Every Node in this map also 
	//			implements the Notation interface.
	// 
	// Notes:	Note that the returned NamedNodeMap belongs to this node. No
	//			attempt must be made to delete it.
	//=========================================================================
	METHOD(DocumentType, getNotations, NamedNodeMap *)
	END_METHOD
	//--

END_INTERFACE(DocumentType)

//++
/*****************************************************************
Interface: DOMImplementation
Desc:
	From the DOM spec:

	The <code>DOMImplementation</code> interface provides a number of methods 
	for performing operations that are independent of any particular instance 
	of the document object model. 
	<p>The DOM Level 1 does not specify a way of creating a document instance, 
	and hence document creation is an operation specific to an implementation. 
	Future Levels of the DOM specification are expected to provide methods for 
	creating documents directly.
Notes: 
*****************************************************************/
INTERFACE(DOMImplementation)
//--

	//++
	//=========================================================================
	// Method:	hasFeature
	// Desc:	Test if the DOM implementation implements a specific feature.
	// 
	// Notes:	In Level 1, the legal value for feature are "HTML" and "XML"
	//			(case insensitive). In Level 1 this is the string "1.0". If the
	//			version is not specified, supporting any version of the feature
	//			will cause the method to return non-zero (TRUE).
	//=========================================================================
	METHOD(DOMImplementation, hasFeature, int)	//return non-zero if feature supported
		METHOD_PARM1(const unicode * feature)	//The package name of the feature to test
		METHOD_PARM(const unicode * version)	//The version number of the package name to test
	END_METHOD
	//--

END_INTERFACE(DOMImplementation)

//++
/*****************************************************************
Interface: Entity
Desc:
	From the DOM spec:

	This interface represents an entity, either parsed or unparsed, in an XML 
	document. Note that this models the entity itself not the entity 
	declaration. <code>Entity</code> declaration modeling has been left for a 
	later Level of the DOM specification.
	<p>The <code>nodeName</code> attribute that is inherited from 
	<code>Node</code> contains the name of the entity.
	<p>An XML processor may choose to completely expand entities before  the 
	structure model is passed to the DOM; in this case there will be no 
	<code>EntityReference</code> nodes in the document tree.
	<p>XML does not mandate that a non-validating XML processor read and 
	process entity declarations  made in the external subset or declared in 
	external parameter entities. This means that parsed entities declared in 
	the external subset need not be expanded by some classes of applications, 
	and that the replacement value of the entity may not be available. When the
	 replacement value is available, the corresponding  <code>Entity</code> 
	node's child list represents the structure of that replacement text. 
	Otherwise, the child list is empty.
	<p>The resolution of the children of the <code>Entity</code> (the  
	replacement value) may be lazily evaluated; actions by the user (such as 
	calling the <code>childNodes</code> method on the <code>Entity</code> 
	Node) are assumed to trigger the evaluation.  
	<p>The DOM Level 1 does not support editing <code>Entity</code> nodes; if a 
	user wants to make changes to the contents of an <code>Entity</code>, 
	every related <code>EntityReference</code> node has to be replaced in the 
	structure model by a clone of the <code>Entity</code>'s contents, and then 
	the desired changes must be made to each of those clones instead. All the 
	descendants of an <code>Entity</code> node are readonly.
	<p>An <code>Entity</code> node does not have any parent.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(Entity,Node)
//--

	//++
	//=========================================================================
	// Method:	getPublicId
	// Desc:	Return the public identifier associated with the entity, if
	//			specified. If the public identifier was not specified, this is 
	//			null.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Entity, getPublicId, const unicode *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getSystemId
	// Desc:	Return the system identifier associated with the entity, if
	//			specified. If the system identifier was not specified, this is 
	//			null.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Entity, getSystemId, const unicode *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getNotationName
	// Desc:	For unparsed entities, the name of the notation for the entity.
	//			For parsed entities, this is null.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Entity, getNotationName, const unicode *)
	END_METHOD
	//--

END_INTERFACE(Entity)

//++
/*****************************************************************
Interface: EntityReference
Desc:
	From the DOM spec:

	 <code>EntityReference</code> objects may be inserted into the structure 
	model when an entity reference is in the source document, or when the user 
	wishes to insert an entity reference. Note that  character references and 
	references to predefined entities are considered to be expanded by the 
	HTML or XML processor so that characters are represented by their Unicode 
	equivalent rather than by an entity reference. Moreover, the XML  
	processor may completely expand references to entities while building the 
	structure model, instead of providing <code>EntityReference</code> 
	objects. If it does provide such objects, then for a given 
	<code>EntityReference</code> node, it may be that there is no 
	<code>Entity</code> node representing the referenced entity; but if such 
	an <code>Entity</code> exists, then the child list of the 
	<code>EntityReference</code> node is the same as that of the 
	<code>Entity</code> node. As with the <code>Entity</code> node, all 
	descendants of the <code>EntityReference</code> are readonly.
	<p>The resolution of the children of the <code>EntityReference</code> (the  
	replacement value of the referenced <code>Entity</code>) may be lazily  
	evaluated; actions by the user (such as calling the  
	<code>childNodes</code> method on the <code>EntityReference</code> node)  
	are assumed to trigger the evaluation.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(EntityReference,Node)
//--
	//no additional methods
END_INTERFACE(EntityReference)

//++
/*****************************************************************
Interface: Notation
Desc:
	From the DOM spec:

	This interface represents a notation declared in the DTD. A notation either 
	declares, by name, the format of an unparsed entity (see section 4.7 of 
	the XML 1.0 specification), or is used for formal declaration of 
	Processing Instruction targets (see section 2.6 of the XML 1.0 
	specification). The <code>nodeName</code> attribute inherited from 
	<code>Node</code> is set to the declared name of the notation.
	<p>The DOM Level 1 does not support editing <code>Notation</code> nodes; 
	they are therefore readonly.
	<p>A <code>Notation</code> node does not have any parent.
Notes: 
*****************************************************************/
DERIVED_INTERFACE(Notation,Node)
//--

	//++
	//=========================================================================
	// Method:	getPublicId
	// Desc:	The public identifier of this notation. If the public identifier
	//			was not specified, this is null.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Entity, getPublicId, const unicode *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getSystemId
	// Desc:	The system identifier of this notation. If the system identifier
	//			was not specified, this is null.
	// 
	// Notes:	
	//=========================================================================
	METHOD(Entity, getSystemId, const unicode *)
	END_METHOD
	//--

END_INTERFACE(Notation)

//++
/*****************************************************************
Interface: NamedNodeMap
Desc:
	From the DOM spec:

	Objects implementing the <code>NamedNodeMap</code> interface are used to 
	represent collections of nodes that can be accessed by name. Note that 
	<code>NamedNodeMap</code> does not inherit from <code>NodeList</code>; 
	<code>NamedNodeMap</code>s are not maintained in any particular order. 
	Objects contained in an object implementing <code>NamedNodeMap</code> may 
	also be accessed by an ordinal index, but this is simply to allow 
	convenient enumeration of the contents of a <code>NamedNodeMap</code>, and 
	does not imply that the DOM specifies an order to these Nodes. 
Notes: 
*****************************************************************/
INTERFACE(NamedNodeMap)
//--

	//++
	//=========================================================================
	// Method:	getNamedItem
	// Desc:	Retrieves a node specified by name.
	// 
	// Notes:	
	//=========================================================================
	METHOD(NamedNodeMap, getNamedItem, Node *)
		METHOD_PARM1(const unicode * name)		//Name of a node to retrieve
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	setNamedItem
	// Desc:	Adds a node using its nodeName attribute.
	//
	//			DOMException registered or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if NamedNodeMap is read only
	//				WRONG_DOCUMENT_ERR if arg was created with a different
	//				Document than the NamedNodeMap belongs to.
	//				INUSE_ATTRIBUTE_ERR if arg is an Attr that already belongs
	//				to another Element.
	// 
	// Notes:	As the node name attribute is used to derive the name which
	//			the node must be stored under, multiple nodes of certain types
	//			(those that have a "special" string value) cannot be stored as
	//			the names would clash.
	//=========================================================================
	METHOD(NamedNodeMap, setNamedItem, Node *)
		METHOD_PARM1(Node * arg)				//A node to store in the map
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	removedNamedItem
	// Desc:	Removes a node specified by name. If the removed node is an 
	//			Attr with a default value it is immediately replaced.
	//
	//			DOMException registered or thrown:
	//				NO_MODIFICATION_ALLOWED_ERR if NamedNodeMap is read only
	//				NOT_FOUND_ERR if no Node with name is in map
	// 
	// Notes:	
	//=========================================================================
	METHOD(NamedNodeMap, removeNamedItem, Node *)
		METHOD_PARM1(const unicode * name)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	item
	// Desc:	Returns the "index"th item in the map. If the index is greater
	//			than or equal to the number of nodes in the map, this returns
	//			null.
	// 
	// Notes:	
	//=========================================================================
	METHOD(NamedNodeMap, item, Node *)
		METHOD_PARM1(int index)					//Index into the map
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getLength
	// Desc:	Then number of nodes in the map. The range of valid child node
	//			indices is 0 to length-1, inclusive.
	// 
	// Notes:	
	//=========================================================================
	METHOD(NamedNodeMap, getLength, int)
	END_METHOD
	//--

END_INTERFACE(NamedNodeMap)

//++
/*****************************************************************
Interface: NodeList
Desc:
	From the DOM spec:

	The <code>NodeList</code> interface provides the abstraction of an ordered 
	collection of nodes, without defining or constraining how this collection 
	is implemented.
	<p>The items in the <code>NodeList</code> are accessible via an integral 
	index, starting from 0. 
Notes: 
	Non-DOM method: destroy()
*****************************************************************/
INTERFACE(NodeList)
//--

	//++
	//=========================================================================
	// Method:	item
	// Desc:	Returns the "index"th item in the list. If the index is greater
	//			than or equal to the number of nodes in the list, this returns
	//			null.
	// 
	// Notes:	
	//=========================================================================
	METHOD(NodeList, item, Node *)
		METHOD_PARM1(int index)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getLength
	// Desc:	Then number of nodes in the list. The range of valid child node
	//			indices is 0 to length-1, inclusive.
	// 
	// Notes:	
	//=========================================================================
	METHOD(NodeList, getLength, int)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	destroy
	// Desc:	Delete this NodeList. This must be called when the NodeList
	//			is no longer needed.
	// 
	// Notes:	C/C++ extension to DOM level 1
	//=========================================================================
	METHOD(Node, destroy, void)
	END_METHOD
	//--

END_INTERFACE(NodeList)

//++
/*****************************************************************
Interface: DOMException
Desc:
	From the DOM spec:

	DOM operations only raise exceptions in "exceptional" circumstances, i.e., 
	when an operation is impossible to perform (either for logical reasons, 
	because data is lost, or  because the implementation has become unstable). 
	In general, DOM methods return specific error values in ordinary 
	processing situation, such as out-of-bound errors when using 
	<code>NodeList</code>.  
	<p>Implementations may raise other exceptions under other circumstances. 
	For example, implementations may raise an implementation-dependent  
	exception if a <code>null</code> argument is passed. 
	<p>Some languages and object systems do not support the concept of 
	exceptions. For such systems, error conditions may be indicated using 
	native error reporting mechanisms. For some bindings, for example, methods 
	may return error codes similar to those listed in the corresponding method 
	descriptions.
Notes: 
*****************************************************************/
INTERFACE(DOMException)
//--
	
	//++
	//=========================================================================
	// Method:	getMessage
	// Desc:	Return the text message associated with the exception, if any.
	//			This may return null if no message was associated with the
	//			exception.
	// 
	// Notes:	This is an extension to DOM Level 1.
	//=========================================================================
	METHOD(DOMException, getMessage, const unicode *)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	getCode
	// Desc:	Get the DOM-defined exception code associated with this
	//			exception.
	// 
	// Notes:	
	//=========================================================================
	METHOD(DOMException, getCode, int)
	END_METHOD
	//--

	//++
	//=========================================================================
	// Method:	destroy
	// Desc:	Delete this exception.
	// 
	// Notes:	C/C++ extension to DOM level 1
	//=========================================================================
	METHOD(DOMException, destroy, void)
	END_METHOD
	//--

#if defined(__cplusplus)
	//These are the possible DOM error codes 
	enum DOMErrors
	{
		//DOM-extension:
		NO_DOM_ERROR				= 0,
		//standard DOM:
		INDEX_SIZE_ERR				= 1,
		DOMSTRING_SIZE_ERR			= 2,
		HIERARCHY_REQUEST_ERR		= 3,
		WRONG_DOCUMENT_ERR			= 4,
		INVALID_CHARACTER_ERR		= 5,
		NO_DATA_ALLOWED_ERR			= 6,
		NO_MODIFICATION_ALLOWED_ERR	= 7,
		NOT_FOUND_ERR				= 8,
		NOT_SUPPORTED_ERR			= 9,
		INUSE_ATTRIBUTE_ERR			= 10
	};
#endif

END_INTERFACE(DOMException)

#if !defined(__cplusplus)
	enum DOMErrors
	{
		//DOM-extension:
		NO_DOM_ERROR				= 0,
		//standard DOM:
		INDEX_SIZE_ERR				= 1,
		DOMSTRING_SIZE_ERR			= 2,
		HIERARCHY_REQUEST_ERR		= 3,
		WRONG_DOCUMENT_ERR			= 4,
		INVALID_CHARACTER_ERR		= 5,
		NO_DATA_ALLOWED_ERR			= 6,
		NO_MODIFICATION_ALLOWED_ERR	= 7,
		NOT_FOUND_ERR				= 8,
		NOT_SUPPORTED_ERR			= 9,
		INUSE_ATTRIBUTE_ERR			= 10,
	};
#endif	//#if !defined(__cplusplus)

#if defined(__cplusplus)
//close namespace DOM 
}
#endif

#if !defined(__cplusplus)
//C macros for calling methods
//Node
#define Node_getNodeName(_this)\
	(*_this->vTbl->getNodeName)(_this)

#define Node_getNodeValue(_this)\
	(*_this->vTbl->getNodeValue)(_this)

#define Node_setNodeValue(_this, value)\
	(*_this->vTbl->setNodeValue)(_this, value)

#define Node_getNodeType(_this)\
	(*_this->vTbl->getNodeType)(_this)

#define Node_getParentNode(_this)\
	(*_this->vTbl->getParentNode(_this)

#define Node_getChildNodes(_this)\
	(*_this->vTbl->getChildNodes)(_this)

#define Node_getFirstChild(_this)\
	(*_this->vTbl->getFirstChild)(_this)

#define Node_getPreviousSibling(_this)\
	(*_this->vTbl->getPreviousSibling)(_this)

#define Node_getNextSibling(_this)\
	(*_this->vTbl->getNextSibling)(_this)

#define Node_getAttributes(_this)\
	(*_this->vTbl->getAttributes)(_this)

#define Node_getOwnerDocument(_this)\
	(*_this->vTbl->getOwnerDocument)(_this)

#define Node_insertBefore(_this,newChild,refChild)\
	(*_this->vTbl->insertBefore)(_this,newChild,refChild)

#define Node_replaceChild(_this,newChild,oldChild)\
	(*_this->vTbl->replaceChild)(_this,newChild,oldChild)

#define Node_removeChild(_this,oldChild)\
	(*_this->vTbl->removeChild)(_this,oldChild)

#define Node_appendChild(_this,newChild)\
	(*_this->vTbl->appendChild)(_this,newChild)

#define Node_hasChildNodes(_this)\
	(*_this->vTbl->hasChildNodes)(_this)

#define Node_cloneNode(_this,deep)\
	(*_this->vTbl->cloneNode)(_this,deep)

#define Node_getLastError(_this)\
	(*_this->vTbl->getLastError)(_this)

#define Node_destroy(_this)\
	(*_this->vTbl->destroy)(_this)

//Document
//	Node-defined methods
#define Document_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define Document_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define Document_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define Document_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define Document_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define Document_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define Document_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define Document_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define Document_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define Document_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define Document_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define Document_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define Document_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define Document_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define Document_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define Document_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define Document_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define Document_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define Document_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//	Document-defined methods
#define Document_getDocType(_this)\
	(*_this->vTbl->getDocType)(_this)

#define Document_getImplementation(_this)\
	(*_this->vTbl->getImplementation)(_this)

#define Document_getDocumentElement(_this)\
	(*_this->vTbl->getDocumentElement)(_this)

#define Document_createElement(_this,tagName)\
	(*_this->vTbl->createElement)(_this,tagName)

#define Document_createDocumentFragment(_this)\
	(*_this->vTbl->createDocumentFragment)(_this)

#define Document_createTextNode(_this,data)\
	(*_this->vTbl->createTextNode)(_this,data)

#define Document_createComment(_this,data)\
	(*_this->vTbl->createComment)(_this,data)

#define Document_createCDATASection(_this,data)\
	(*_this->vTbl->createCDATASection)(_this,data)

#define Document_createProcessingInstruction(_this,target,data)\
	(*_this->vTbl->createProcessingInstruction)(_this,target,data)

#define Document_createAttribute(_this,name)\
	(*_this->vTbl->createAttribute)(_this,name)

#define Document_createEntityReference(_this,name)\
	(*_this->vTbl->createEntityReference)(_this,name)

#define Document_getElementsByTagName(_this,tagName)\
	(*_this->vTbl->getElementsByTagName)(_this,tagName)

#define Document_enableExceptions(_this,enable)\
	(*_this->vTbl->enableExceptions)(_this,enable)

//DocumentFragment
//	Node-defined methods
#define DocumentFragment_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define DocumentFragment_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define DocumentFragment_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define DocumentFragment_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define DocumentFragment_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define DocumentFragment_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define DocumentFragment_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define DocumentFragment_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define DocumentFragment_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define DocumentFragment_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define DocumentFragment_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define DocumentFragment_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define DocumentFragment_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define DocumentFragment_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define DocumentFragment_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define DocumentFragment_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define DocumentFragment_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define DocumentFragment_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define DocumentFragment_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//Element
//	Node-defined methods
#define Element_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define Element_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define Element_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define Element_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define Element_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define Element_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define Element_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define Element_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define Element_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define Element_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define Element_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define Element_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define Element_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define Element_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define Element_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define Element_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define Element_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define Element_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define Element_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//	Element-defined methods
#define Element_getTagName(_this)\
	(*_this->vTbl->getTagName)(_this)

#define Element_getAttribute(_this,name)\
	(*_this->vTbl->getAttribute)(_this,name)

#define Element_setAttribute(_this,name,value)\
	(*_this->vTbl->setAttribute)(_this,name,value)

#define Element_removeAttribute(_this,name)\
	(*_this->vTbl->removeAttribute)(_this,name)

#define Element_getAttributeNode(_this,name)\
	(*_this->vTbl->getAttributeNode)(_this,name)

#define Element_setAttributeNode(_this,newAttr)\
	(*_this->vTbl->setAttributeNode)(_this,newAttr)

#define Element_removeAttributeNode(_this,oldAttr)\
	(*_this->vTbl->removeAttributeNode)(_this,oldAttr)

#define Element_getElementsByTagName(_this,tagName)\
	(*_this->vTbl->getElementByTagName)(_this,tagName)

#define Element_normalize(_this)\
	(*_this->vTbl->normalize)(_this)

//Attr
//	Node-defined methods
#define Attr_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define Attr_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define Attr_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define Attr_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define Attr_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define Attr_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define Attr_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define Attr_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define Attr_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define Attr_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define Attr_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define Attr_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define Attr_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define Attr_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define Attr_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define Attr_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define Attr_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define Attr_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define Attr_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//	Attr-defined methods
#define	Attr_getName(_this)\
	(*_this->vTbl->getName)(_this)

#define	Attr_getSpecified(_this)\
	(*_this->vTbl->getSpecified)(_this)

#define	Attr_getValue(_this)\
	(*_this->vTbl->getValue)(_this)

#define	Attr_setValue(_this,value)\
	(*_this->vTbl->setValue)(_this,value)

//CharacterData
//	Node-defined methods
#define CharacterData_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define CharacterData_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define CharacterData_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define CharacterData_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define CharacterData_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define CharacterData_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define CharacterData_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define CharacterData_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define CharacterData_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define CharacterData_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define CharacterData_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define CharacterData_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define CharacterData_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define CharacterData_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define CharacterData_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define CharacterData_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define CharacterData_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define CharacterData_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define CharacterData_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//	CharacterData-defined methods
#define CharacterData_getData(_this)\
	(*_this->vTbl->getData)(_this)

#define CharacterData_setData(_this,data)\
	(*_this->vTbl->setData)(_this,data)

#define CharacterData_getLength(_this)\
	(*_this->vTbl->getLength)(_this)

#define CharacterData_substringData(_this,offset,count)\
	(*_this->vTbl->substringData)(_this,offset,count)

#define CharacterData_appendData(_this,arg)\
	(*_this->vTbl->appendData)(_this,arg)

#define CharacterData_insertData(_this,offset,arg)\
	(*_this->vTbl->insertData)(_this,offset,arg)

#define CharacterData_deleteData(_this,offset,count)\
	(*_this->vTbl->deleteData)(_this,offset,count)

#define CharacterData_replaceData(_this,offset,count,arg)\
	(*_this->vTbl->replaceData)(_this,offset,count,arg)

//Text
//	Node-defined methods
#define Text_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define Text_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define Text_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define Text_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define Text_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define Text_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define Text_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define Text_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define Text_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define Text_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define Text_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define Text_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define Text_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define Text_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define Text_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define Text_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define Text_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define Text_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define Text_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//	CharacterData-defined methods
#define Text_getData(_this)\
	(*((CharacterData *)_this)->vTbl->getData)(_this)

#define Text_setData(_this,data)\
	(*((CharacterData *)_this)->vTbl->setData)(_this,data)

#define Text_getLength(_this)\
	(*((CharacterData *)_this)->vTbl->getLength)(_this)

#define Text_substringData(_this,offset,count)\
	(*((CharacterData *)_this)->vTbl->substringData)(_this,offset,count)

#define Text_appendData(_this,arg)\
	(*((CharacterData *)_this)->vTbl->appendData)(_this,arg)

#define Text_insertData(_this,offset,arg)\
	(*((CharacterData *)_this)->vTbl->insertData)(_this,offset,arg)

#define Text_deleteData(_this,offset,count)\
	(*((CharacterData *)_this)->vTbl->deleteData)(_this,offset,count)

#define Text_replaceData(_this,offset,count,arg)\
	(*((CharacterData *)_this)->vTbl->replaceData)(_this,offset,count,arg)

//	Text-defined methods
#define Text_splitText(_this,offset)\
	(*_this->vTbl->splitText)(offset)

//CDATASection
//	Node-defined methods
#define CDATASection_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define CDATASection_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define CDATASection_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define CDATASection_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define CDATASection_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define CDATASection_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define CDATASection_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define CDATASection_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define CDATASection_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define CDATASection_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define CDATASection_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define CDATASection_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define CDATASection_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define CDATASection_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define CDATASection_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define CDATASection_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define CDATASection_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define CDATASection_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define CDATASection_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//	CharacterData-defined methods
#define CDATASection_getData(_this)\
	(*((CharacterData *)_this)->vTbl->getData)(_this)

#define CDATASection_setData(_this,data)\
	(*((CharacterData *)_this)->vTbl->setData)(_this,data)

#define CDATASection_getLength(_this)\
	(*((CharacterData *)_this)->vTbl->getLength)(_this)

#define CDATASection_substringData(_this,offset,count)\
	(*((CharacterData *)_this)->vTbl->substringData)(_this,offset,count)

#define CDATASection_appendData(_this,arg)\
	(*((CharacterData *)_this)->vTbl->appendData)(_this,arg)

#define CDATASection_insertData(_this,offset,arg)\
	(*((CharacterData *)_this)->vTbl->insertData)(_this,offset,arg)

#define CDATASection_deleteData(_this,offset,count)\
	(*((CharacterData *)_this)->vTbl->deleteData)(_this,offset,count)

#define CDATASection_replaceData(_this,offset,count,arg)\
	(*((CharacterData *)_this)->vTbl->replaceData)(_this,offset,count,arg)

//	Text-defined methods
#define CDATASection_splitText(_this,offset)\
	(*((Text *)_this)->vTbl->splitText)(offset)

//Comment
//	Node-defined methods
#define Comment_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define Comment_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define Comment_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define Comment_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define Comment_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define Comment_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define Comment_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define Comment_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define Comment_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define Comment_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define Comment_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define Comment_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define Comment_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define Comment_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define Comment_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define Comment_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define Comment_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define Comment_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define Comment_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//	CharacterData-defined methods
#define Comment_getData(_this)\
	(*((CharacterData *)_this)->vTbl->getData)(_this)

#define Comment_setData(_this,data)\
	(*((CharacterData *)_this)->vTbl->setData)(_this,data)

#define Comment_getLength(_this)\
	(*((CharacterData *)_this)->vTbl->getLength)(_this)

#define Comment_substringData(_this,offset,count)\
	(*((CharacterData *)_this)->vTbl->substringData)(_this,offset,count)

#define Comment_appendData(_this,arg)\
	(*((CharacterData *)_this)->vTbl->appendData)(_this,arg)

#define Comment_insertData(_this,offset,arg)\
	(*((CharacterData *)_this)->vTbl->insertData)(_this,offset,arg)

#define Comment_deleteData(_this,offset,count)\
	(*((CharacterData *)_this)->vTbl->deleteData)(_this,offset,count)

#define Comment_replaceData(_this,offset,count,arg)\
	(*((CharacterData *)_this)->vTbl->replaceData)(_this,offset,count,arg)

//ProcessingInstruction
//	Node-defined methods
#define ProcessingInstruction_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define ProcessingInstruction_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define ProcessingInstruction_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define ProcessingInstruction_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define ProcessingInstruction_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define ProcessingInstruction_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define ProcessingInstruction_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define ProcessingInstruction_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define ProcessingInstruction_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define ProcessingInstruction_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define ProcessingInstruction_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define ProcessingInstruction_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define ProcessingInstruction_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define ProcessingInstruction_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define ProcessingInstruction_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define ProcessingInstruction_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define ProcessingInstruction_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define ProcessingInstruction_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define ProcessingInstruction_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//	ProcessingInstruction-defined methods
#define ProcessingInstruction_getTarget(_this)\
	(*_this->vTbl->getTarget)(_this)

#define ProcessingInstruction_getData(_this)\
	(*_this->vTbl->getData)(_this)

#define ProcessingInstruction_setData(_this,data)\
	(*_this->vTbl->setData)(_this,data)

//DocumentType
//	Node-defined methods
#define DocumentType_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define DocumentType_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define DocumentType_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define DocumentType_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define DocumentType_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define DocumentType_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define DocumentType_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define DocumentType_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define DocumentType_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define DocumentType_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define DocumentType_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define DocumentType_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define DocumentType_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define DocumentType_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define DocumentType_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define DocumentType_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define DocumentType_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define DocumentType_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define DocumentType_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//	DocumentType-defined methods
#define	DocumentType_getName(_this)\
	(*_this->vTbl->getName(_this)

#define	DocumentType_getEntities(_this)\
	(*_this->vTbl->getEntities(_this)

#define	DocumentType_getNotations(_this)\
	(*_this->vTbl->getNotations(_this)

//DOMImplementation
#define DOMImplementation_hasFeature(_this,feature,version)\
	(*_this->vTbl->hasFeature)(_this,feature,version)

//Entity
//	Node-defined methods
#define Entity_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define Entity_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define Entity_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define Entity_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define Entity_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define Entity_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define Entity_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define Entity_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define Entity_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define Entity_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define Entity_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define Entity_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define Entity_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define Entity_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define Entity_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define Entity_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define Entity_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define Entity_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define Entity_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//	Entity-defined methods
#define Entity_getPublicId(_this)\
	(*_this->vTbl->getPublicId)(_this)

#define Entity_getSystemId(_this)\
	(*_this->vTbl->getSystemId)(_this)

#define Entity_getNotationName(_this)\
	(*_this->vTbl->getNotationName)(_this)

//EntityReference
//	Node-defined methods
#define EntityReference_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define EntityReference_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define EntityReference_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define EntityReference_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define EntityReference_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define EntityReference_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define EntityReference_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define EntityReference_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define EntityReference_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define EntityReference_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define EntityReference_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define EntityReference_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define EntityReference_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define EntityReference_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define EntityReference_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define EntityReference_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define EntityReference_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define EntityReference_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define EntityReference_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//Notation
//	Node-defined methods
#define Notation_getNodeName(_this)\
	(*((Node *)_this)->vTbl->getNodeName)(_this)

#define Notation_getNodeValue(_this)\
	(*((Node *)_this)->vTbl->getNodeValue)(_this)

#define Notation_setNodeValue(_this, value)\
	(*((Node *)_this)->vTbl->setNodeValue)(_this, value)

#define Notation_getNodeType(_this)\
	(*((Node *)_this)->vTbl->getNodeType)(_this)

#define Notation_getParentNode(_this)\
	(*((Node *)_this)->vTbl->getParentNode(_this)

#define Notation_getChildNodes(_this)\
	(*((Node *)_this)->vTbl->getChildNodes)(_this)

#define Notation_getFirstChild(_this)\
	(*((Node *)_this)->vTbl->getFirstChild)(_this)

#define Notation_getPreviousSibling(_this)\
	(*((Node *)_this)->vTbl->getPreviousSibling)(_this)

#define Notation_getNextSibling(_this)\
	(*((Node *)_this)->vTbl->getNextSibling)(_this)

#define Notation_getAttributes(_this)\
	(*((Node *)_this)->vTbl->getAttributes)(_this)

#define Notation_getOwnerDocument(_this)\
	(*((Node *)_this)->vTbl->getOwnerDocument)(_this)

#define Notation_insertBefore(_this,newChild,refChild)\
	(*((Node *)_this)->vTbl->insertBefore)(_this,newChild,refChild)

#define Notation_replaceChild(_this,newChild,oldChild)\
	(*((Node *)_this)->vTbl->replaceChild)(_this,newChild,oldChild)

#define Notation_removeChild(_this,oldChild)\
	(*((Node *)_this)->vTbl->removeChild)(_this,oldChild)

#define Notation_appendChild(_this,newChild)\
	(*((Node *)_this)->vTbl->appendChild)(_this,newChild)

#define Notation_hasChildNodes(_this)\
	(*((Node *)_this)->vTbl->hasChildNodes)(_this)

#define Notation_cloneNode(_this,deep)\
	(*((Node *)_this)->vTbl->cloneNode)(_this,deep)

#define Notation_getLastError(_this)\
	(*((Node *)_this)->vTbl->getLastError)(_this)

#define Notation_destroy(_this)\
	(*((Node *)_this)->vTbl->destroy)(_this)

//	Notation-defined methods
#define Notation_getPublicId(_this)\
	(*_this->vTbl->getPublicId)(_this)

#define Notation_getSystemId(_this)\
	(*_this->vTbl->getSystemId)(_this)

//NamedNodeMap
#define NamedNodeMap_getNamedItem(_this,name)\
	(*_this->vTbl->getNamedItem)(_this,name)

#define NamedNodeMap_setNamedItem(_this,arg)\
	(*_this->vTbl->setNamedItem)(_this,arg)

#define NamedNodeMap_removeNamedItem(_this,name)\
	(*_this->vTbl->removeNamedItem)(_this,name)

#define NamedNodeMap_item(_this,index)\
	(*_this->vTbl->item)(_this,index)

#define NamedNodeMap_getLength(_this)\
	(*_this->vTbl->getLength)(_this)

//NodeList
#define NodeList_item(_this,index)\
	(*_this->vTbl->item)(_this,index)

#define NodeList_getLength(_this)\
	(*_this->vTbl->getLength)(_this)

#endif //if !defined(__cplusplus)

#endif	//#ifndef _DOM_H_
