/**
 * Copyright (c) Unpublished Work of Novell, Inc. All Rights Reserved.
 * THIS WORK IS AN UNPUBLISHED WORK AND CONTAINS CONFIDENTIAL,
 * PROPRIETARY AND TRADE SECRET INFORMATION OF NOVELL, INC. ACCESS TO
 * THIS WORK IS RESTRICTED TO (I) NOVELL, INC. EMPLOYEES WHO HAVE A
 * NEED TO KNOW HOW TO PERFORM TASKS WITHIN THE SCOPE OF THEIR
 * ASSIGNMENTS AND (II) ENTITIES OTHER THAN NOVELL, INC. WHO HAVE
 * ENTERED INTO APPROPRIATE LICENSE AGREEMENTS. NO PART OF THIS WORK
 * MAY BE USED, PRACTICED, PERFORMED, COPIED, DISTRIBUTED, REVISED,
 * MODIFIED, TRANSLATED, ABRIDGED, CONDENSED, EXPANDED, COLLECTED,
 * COMPILED, LINKED, RECAST, TRANSFORMED OR ADAPTED WITHOUT THE PRIOR
 * WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION OF THIS
 * WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
 * CRIMINAL AND CIVIL LIABILITY.
 *
 * %version:  %
 * %created_by:  %
 * %date_modified: %
 */

package com.novell.nds.dirxml.driver.xds.skeleton;



//DriverShim, PublicationShim, and SubscriptionShim are the interfaces
//  a driver needs to implement to interface with the DirXML engine
import com.novell.nds.dirxml.driver.DriverShim;
import com.novell.nds.dirxml.driver.PublicationShim;
import com.novell.nds.dirxml.driver.SubscriptionShim;
import com.novell.nds.dirxml.driver.XmlDocument;
import com.novell.nds.dirxml.driver.xds.*;
import com.novell.nds.dirxml.driver.xds.util.StatusAttributes;
import com.novell.nds.dirxml.driver.xds.util.XDSUtil;

import java.util.HashMap;


/**
 * A basic skeleton for implementing a <code>DriverShim</code>.
 *
 * <p>
 * <code>com.novell.nds.dirxml.driver.DriverShim</code> defines the interface
 * used by the DirXML engine to start stop an application shim.
 * </p>
 *
 * <p>
 * A <code>DriverShim</code> must implement a constructor which takes no
 * parameters in order for the DirXML engine to instantiate it.
 * </p>
 *
 * <p>
 * NOTE:  the Skeleton Driver makes no use of .ini files or any information
 * outside of that supplied through the initialization data supplied to the
 * shim <code>init()</code> methods; in general, it is inappropriate for a
 * driver to store configuration information outside of eDirectory where it
 * cannot be remotely configured using ConsoleOne, iManager, or other remote
 * configuration utilities
 * </p>
 *
 * <p>
 * NOTE:  it is unwise to have static mutable class data, since that prevents
 * multiple instances of the driver from functioning independently
 * </p>
 */
public class SkeletonDriverShim
    extends CommonImpl
    implements DriverShim
{
    //~ Static fields/initializers ---------------------------------------------

    /*
       <driver-options>
           <option-1 display-name="Sample String option">any string will do</option-1>
           <option-2 display-name="Sample int option (enter an integer)">10</option-2>
           <option-3 display-name="Sample boolean option (enter a boolean value)">true</option-3>
           <option-4 display-name="Sample enumerated option (1, 2 or 3)">2</option-4>
           <option-5 display-name="Sample regular expression option (must match patterh a*b)">aab</option-5>
           <option-6 display-name="Sample required option (enter some value)">not null</option-6>
       </driver-options>
     */

    //MODIFY:  put the number of driver parameters your driver uses here

    /** The number of driver parameters. */
    private static final int NO_OF_PARAMS = 9;

    //MODIFY:  put the tag names of your parameters here

    /** The first driver parameter tag name. */
    private static final String TAG_OPTION_1 = "option-1";

    /** The second driver parameter tag name. */
    private static final String TAG_OPTION_2 = "option-2";

    /** The third driver parameter tag name. */
    private static final String TAG_OPTION_3 = "option-3";

    /** The fourth driver parameter tag name. */
    private static final String TAG_OPTION_4 = "option-4";

    /** The fifth driver parameter tag name. */
    private static final String TAG_OPTION_5 = "option-5";

    /** The sixth driver parameter tag name. */
    private static final String TAG_OPTION_6 = "option-6";

    //MODIFY:  put the default values for your parameters here

    /** The default value for the first driver parameter. */
    private static final String DEFAULT_OPTION_1 = "any string will do";

    /** The default value for the second driver parameter. */
    private static final String DEFAULT_OPTION_2 = "10";

    /** The default value for the third driver parameter. */
    private static final String DEFAULT_OPTION_3 = "true";

    /** The default value for the fourth driver parameter. */
    private static final String DEFAULT_OPTION_4 = "1";

    /** The default value for the fifth driver parameter. */
    private static final String DEFAULT_OPTION_5 = null;

    /** The default value for the sixth driver parameter. */
    private static final String DEFAULT_OPTION_6 = null;

    /** The default value for the server parameter. */
    private static final String DEFAULT_SERVER = null;//7

    /** The default value for the user parameter. */
    private static final String DEFAULT_USER = null;//8

    /** The default value for the password parameter. */
    private static final String DEFAULT_PASSWORD = null;//9

    //MODIFY:  put the name of your application here

    /** The name of the target application. */
    private static final String APPLICATION_NAME = "Skeleton Application";

    //~ Instance fields --------------------------------------------------------

    /**
     * Object implementing
     * <code>com.novell.nds.dirxml.driver.SubscriptionShim</code>, which is
     * supplied to the DirXML engine via the
     * <code>getSubscriptionShim()</code> method.
     *
     * <p></p>
     *
     * @see #getSubscriptionShim()
     */
    private SkeletonSubscriptionShim subscriptionShim;

    /**
     * Object implementing
     * <code>com.novell.nds.dirxml.driver.PublicationShim</code>, which is
     * supplied to the DirXML engine via the <code>getPublicationShim()</code>
     * method.
     *
     * <p></p>
     *
     * @see #getPublicationShim()
     */
    private SkeletonPublicationShim publicationShim;

    //~ Constructors -----------------------------------------------------------

    /**
     * A Java driver shim <code>must</code> have a constructor which takes no
     * parameters.  The DirXML engine uses this to construct the driver
     * object.
     */
    public SkeletonDriverShim()
    {
        //create temporary trace object until we know what the driver's
        //  RDN is
        setDriverRDN(TRACE_ID);
        setTrace();
        setDriverParams();

        subscriptionShim     = null;
        publicationShim      = null;
    }//SkeletonDriverShim()

    //~ Methods ----------------------------------------------------------------

    /**
     * A non-interface method that describes the parameters this DriverShim is
     * expecting.
     *
     * <p></p>
     *
     * @see SkeletonDriverShim#init(XmlDocument)
     */
    private void setDriverParams()
    {
        //MODIFY:  construct parameter descriptors here for your driver
        //  parameters
        //The XDS.jar library automatically checks parameter data
        //types for you.  When a RequiredConstraint is added to a parameter,
        //the library will check init documents to ensure the parameter is
        //present and has a value.  When you add RangeConstraints or
        //EnumConstraints to a parameter, the library will check parameter
        //values to see if they satisfy these constraints.

        /*
           <driver-options>
               <option-1 display-name="Sample String option">any string will do</option-1>
               <option-2 display-name="Sample int option (enter an integer)">10</option-2>
               <option-3 display-name="Sample boolean option (enter a boolean value)">true</option-3>
               <option-4 display-name="Sample enumerated option (1, 2 or 3)">2</option-4>
               <option-5 display-name="Sample regular expression option (must match patterh a*b)">aab</option-5>
               <option-6 display-name="Sample required option (enter some value)">not null</option-6>
           </driver-options>
         */
        Parameter       param;
        EnumConstraint  enum;
        RegExConstraint regex;

        driverParams     = new HashMap(NO_OF_PARAMS);

        //non-authentication params
        param = new Parameter(
                TAG_OPTION_1,//tag name
                DEFAULT_OPTION_1,//default value
                DataType.STRING);//data type
        driverParams.put(
            param.tagName(),
            param);

        param = new Parameter(
                TAG_OPTION_2,
                DEFAULT_OPTION_2,
                DataType.INT);
        driverParams.put(
            param.tagName(),
            param);

        param = new Parameter(
                TAG_OPTION_3,
                DEFAULT_OPTION_3,
                DataType.BOOLEAN);
        driverParams.put(
            param.tagName(),
            param);

        param     = new Parameter(
                TAG_OPTION_4,
                DEFAULT_OPTION_4,
                DataType.INT);
        enum = new EnumConstraint();
        enum.addLiterals(new String[]
            {
                "1",
                "2",
                "3"
            });
        param.add(enum);
        driverParams.put(
            param.tagName(),
            param);

        param     = new Parameter(
                TAG_OPTION_5,
                DEFAULT_OPTION_5,
                DataType.STRING);

        // NOTE:  To use a regular expression constraint, your driver MUST target a
        //   1.4 JVM or later.
        regex = new RegExConstraint("a*b");
        param.add(regex);
        driverParams.put(
            param.tagName(),
            param);

        param = new Parameter(
                TAG_OPTION_6,
                DEFAULT_OPTION_6,
                DataType.STRING);
        param.add(RequiredConstraint.REQUIRED);
        driverParams.put(
            param.tagName(),
            param);

        //authentication parameters
        param = new Parameter(
                DTD.TAG_SERVER,
                DEFAULT_SERVER,
                DataType.STRING);
        driverParams.put(
            param.tagName(),
            param);

        param = new Parameter(
                DTD.TAG_USER,
                DEFAULT_USER,
                DataType.STRING);
        driverParams.put(
            param.tagName(),
            param);

        param = new Parameter(
                DTD.TAG_PASSWORD,
                DEFAULT_PASSWORD,
                DataType.STRING);
        driverParams.put(
            param.tagName(),
            param);

        /*
           //additional examples:
           //non-negative (0...Integer.MAX_VALUE); required
           RangeConstraint range;
           range = new RangeConstraint(0, EqualityOp.GTE);
           param = new Parameter("non-negative",
                                 "default value",
                                 DataType.INT);
           param.add(RequiredConstraint.REQUIRED);
           param.add(range);
           //enumeration of legal values; optional
           EnumConstraint enum;
           enum = new EnumConstraint();
           enum.addLiteral("first");
           enum.addLiteral("second");
           enum.addLiteral("third");
           param = new Parameter("enumeration",
                                 "first",
                                 DataType.STRING);
           param.add(enum);
           //complex param; (1...4) or (6...10)
           ConstraintGroup complex;
           complex = new ConstraintGroup();
           range = new RangeConstraint(1, 4);
           complex.add(range);
           range = new RangeConstraint(6, 10);
           complex.add(range);
           param = new Parameter("complex",
                                 "1",
                                 DataType.INT);
           param.add(RequiredConstraint.REQUIRED);
           param.add(complex);
         */
    }//setDriverParams():void

    /**
     * <code>init</code> will be called after a driver is instantiated to allow
     * it to perform any necessary initialization before event processing
     * begins. Typically it will perform any setup that is common to both the
     * subscriber and publisher channels.
     *
     * <p></p>
     *
     * @param initXML XML document that contains the driver's initialization
     *        parameters
     *
     * @return an XML document containing status messages for this operation
     */
    public XmlDocument init(XmlDocument initXML)
    {
        //MODIFY:  initialize your driver here
        //example initialization document:

        /*
           <nds dtdversion="1.1" ndsversion="8.6">
               <source>
                   <product version="1.1a">DirXML</product>
                   <contact>Novell, Inc.</contact>
               </source>
               <input>
                   <init-params src-dn="\NEW_DELL_TREE\NOVELL\Driver Set\Skeleton Driver (Java, XDS)">
                       <authentication-info>
                           <server>server.app:400</server>
                           <user>User1</user>
                       </authentication-info>
                       <driver-options>
                   <option-1 display-name="Sample String option">This is a string</option-1>
                   <option-2 display-name="Sample int option (enter an integer)">10</option-2>
                   <option-3 display-name="Sample boolean option (enter a boolean value)">true</option-3>
                   <option-4 display-name="Sample required option (enter some value)">not null</option-4>
               </driver-options>
                   </init-params>
               </input>
           </nds>
         */

        //example result document:

        /*
           <nds dtdversion="1.1" ndsversion="8.6">
               <source>
                   <product build="20021214_0304" instance="Skeleton Driver (Java, XDS)" version="1.1">DirXML Skeleton Driver (Java, XDS)</product>
                   <contact>My Company Name</contact>
               </source>
               <output>
                   <status level="success" type="driver-status">
                       <parameters>
                           <option-1 display-name="Sample String option">This is a string</option-1>
                           <option-2 display-name="Sample int option (enter an integer)">10</option-2>
                           <option-3 display-name="Sample boolean option (enter a boolean value)">true</option-3>
                           <option-4 display-name="Sample required option (enter some value)">not null</option-4>
                           <password display-name="password"><!-- content suppressed -->
                           </password>
                           <server display-name="server">server.app:400</server>
                           <user display-name="user">User1</user>
                       </parameters>
                   </status>
               </output>
           </nds>
         */
        trace.trace(
            "init",
            1);

        XDSResultDocument result;
        StatusAttributes  attrs;

        //create result document for reporting status to DirXML engine
        //can't add source info yet since we don't have the driver's RDN
        result = new XDSResultDocument();

        try
        {
            XDSInitDocument  init;
            XDSStatusElement status;

            //parse initialization document
            init = new XDSInitDocument(initXML);

            setDriverRDN(init.rdn());

            //create new trace object that uses driver's RDN
            setTrace();

            //append a <source> element now that we know the driver's RDN
            appendSourceInfo(result);

            //get driver parameters from the initialization doc
            init.parameters(driverParams);

            //if we don't have any authentication parameters, the skeleton
            //  driver won't run; your driver may or may not need
            //  information in the authentication parameters
            if((driverParams.get(DTD.TAG_SERVER).toString() == null) &&
                (driverParams.get(DTD.TAG_USER).toString() == null) &&
                (driverParams.get(DTD.TAG_PASSWORD).toString() == null))
            {
                attrs = StatusAttributes.factory(
                        StatusLevel.FATAL,
                        StatusType.DRIVER_GENERAL,
                        null);//event-id
                XDSUtil.appendStatus(
                    result,//doc to append to
                    attrs,//status attribute values
                    Errors.NO_AUTH);//description
            }//if
            else
            {
                //create a publisher and subscriber instance
                subscriptionShim     = new SkeletonSubscriptionShim(this);
                publicationShim      = new SkeletonPublicationShim(this);

                //perform any other initialization that might be required
                //append a successful <status> element to the result doc
                attrs     = StatusAttributes.factory(
                        StatusLevel.SUCCESS,
                        StatusType.DRIVER_STATUS,
                        null);//event-id
                status = XDSUtil.appendStatus(
                        result,//doc to append to
                        attrs,//status attribute values
                        null);//description

                //append the parameter values the driver is actually using
                status.parametersAppend(driverParams);
            }//else
        }//try
        catch(Exception e)//don't want to catch Error class with Throwable
        {
            //e instance of XDSException:
            //
            //  init document is malformed or invalid -- or --
            //  it is missing required parameters or contains
            //  illegal parameter values
            //e instance of RuntimeException:
            //  e.g., NullPointerException
            attrs = StatusAttributes.factory(
                    StatusLevel.FATAL,
                    StatusType.DRIVER_STATUS,
                    null);//event-id
            XDSUtil.appendStatus(
                result,//doc to append to
                attrs,//status attribute values
                null,//description
                e,//exception
                XDSUtil.appendStackTrace(e),//append stack trace?
                initXML);//xml to append
        }//catch

        //return result doc w/ status to DirXML engine
        return result.toXML();
    }//init(XmlDocument):XmlDocument

    /**
     * <code>shutdown</code> indicates to the DriverShim that the driver is
     * being terminated.
     *
     * <p></p>
     *
     * @param reasonXML unused
     *
     * @return an XML document containing status messages for this operation
     */
    public XmlDocument shutdown(XmlDocument reasonXML)
    {
        //MODIFY:  put your shutdown code here
        //example result document:

        /*
           <nds dtdversion="1.1" ndsversion="8.6">
               <source>
                   <product build="20021214_0304" instance="Skeleton Driver (Java, XDS)" version="1.1">DirXML Skeleton Driver (Java, XDS)</product>
                   <contact>My Company Name</contact>
               </source>
               <output>
                   <status level="success" type="driver-status"/>
               </output>
           </nds>
         */
        trace.trace(
            "shutdown",
            1);

        XDSResultDocument result;
        StatusAttributes  attrs;

        result = newResultDoc();

        try
        {
            //shutdown whatever needs shutting down
            //  which includes telling the publication shim to
            //  return control back to the DirXML engine and
            //  telling the subscription shim to free resources
            if(publicationShim != null)
            {
                publicationShim.shutdown();
            }

            if(subscriptionShim != null)
            {
                subscriptionShim.shutdown();
            }

            //append a successful <status> element to the result doc
            attrs = StatusAttributes.factory(
                    StatusLevel.SUCCESS,
                    StatusType.DRIVER_STATUS,
                    null);//event-id
            XDSUtil.appendStatus(
                result,//doc to append to
                attrs,//status attribute values
                null);//description
        }//try
        catch(Exception e)//don't want to catch Error class with Throwable
        {
            //something bad happened...
            attrs = StatusAttributes.factory(
                    StatusLevel.FATAL,
                    StatusType.DRIVER_STATUS,
                    null);//event-id
            XDSUtil.appendStatus(
                result,//do to append to
                attrs,//status attribute values
                null,//description
                e,//exception
                true,//append stack trace?
                null);//xml to append
        }//catch

        //return the result doc w/ status to DirXML engine
        return result.toXML();
    }//shutdown(XmlDocument):XmlDocument

    /**
     * <code>getSubscriptionShim</code> gets the implementation of
     * SubscriptionShim that will be used to process commands on behalf of the
     * DirXML engine.
     *
     * <p>
     * NOTE: the returned instance will be initialized by the DirXML engine
     * </p>
     *
     * <p></p>
     *
     * @return an instance of  SubscriptionShim that will be used to process
     *         commands on behalf of the DirXML engine
     *
     * @see SkeletonSubscriptionShim#init(XmlDocument)
     */
    public SubscriptionShim getSubscriptionShim()
    {
        trace.trace(
            "getSubscriptionShim",
            1);

        return subscriptionShim;
    }//getSubscriptionShim():SubscriptionShim

    /**
     * <code>getPublicationShim</code> gets the implementation of
     * PublicationShim that will be invoked by the DirXML engine.
     *
     * <p>
     * NOTE: the returned instance will be initialized by the DirXML engine
     * </p>
     *
     * <p></p>
     *
     * @return an instance of PublicationShim that will be invoked by the
     *         DirXML engine
     *
     * @see SkeletonPublicationShim#init(XmlDocument)
     */
    public PublicationShim getPublicationShim()
    {
        trace.trace(
            "getPublicationShim",
            1);

        return publicationShim;
    }//getPublicationShim():PublicationShim

    /**
     * <code>getSchema</code> returns the application schema encoded in XDS
     * XML.
     *
     * <p>
     * This will be called only when the driver is not running. In other words,
     * if this method is called init()/shutdown() will not be called for the
     * current instance of the DriverShim.
     * </p>
     *
     * <p></p>
     *
     * @param initXML XML document containing the driver shim initialization
     *        parameters as well as the subscription shim and publication shim
     *        initialization parameters.
     *
     * @return XML document containing the application schema or an error
     *         status
     */
    public XmlDocument getSchema(XmlDocument initXML)
    {
        //MODIFY:  get your application schema here
        //example initialization document:

        /*
           <nds dtdversion="1.1" ndsversion="8.6">
               <source>
                   <product version="1.1a">DirXML</product>
                   <contact>Novell, Inc.</contact>
               </source>
               <input>
                   <init-params src-dn="\NEW_DELL_TREE\NOVELL\Driver Set\Skeleton Driver (Java, XDS)">
                       <authentication-info>
                           <server>server.app:400</server>
                           <user>User1</user>
                       </authentication-info>
                       <driver-options>
                   <option-1 display-name="Sample String option">This is a string</option-1>
                   <option-2 display-name="Sample int option (enter an integer)">10</option-2>
                   <option-3 display-name="Sample boolean option (enter a boolean value)">true</option-3>
                   <option-4 display-name="Sample required option (enter some value)">not null</option-4>
               </driver-options>
                       <subscriber-options>
                   <sub-1 display-name="Sample Subscriber option">String for Subscriber</sub-1>
               </subscriber-options>
                       <publisher-options>
                   <pub-1 display-name="Sample Publisher option">String for Publisher</pub-1>
                   <polling-interval display-name="Polling interval in seconds">10</polling-interval>
               </publisher-options>
                   </init-params>
               </input>
           </nds>
         */

        //example result document:

        /*
           <nds dtdversion="1.1" ndsversion="8.6">
               <source>
                   <product build="20021214_0304" instance="Skeleton Driver (Java, XDS)" version="1.1">DirXML Skeleton Driver (Java, XDS)</product>
                   <contact>My Company Name</contact>
               </source>
               <output>
                   <schema-def application-name="Skeleton Application" hierarchical="true">
                       <class-def class-name="fake-class-1" container="true">
                           <attr-def attr-name="fake-attr-1" case-sensitive="true" multi-valued="true" naming="true" read-only="true" required="true" type="string"/>
                       </class-def>
                       <class-def class-name="fake-class-2" container="false">
                           <attr-def attr-name="fake-attr-2" case-sensitive="false" multi-valued="false" naming="false" read-only="false" required="false" type="int"/>
                       </class-def>
                   </schema-def>
                   <status level="warning" type="driver-general">
                       <description>Get schema not implemented.</description>
                   </status>
               </output>
           </nds>
         */
        trace.trace(
            "getSchema",
            1);

        XDSSchemaResultDocument result;
        StatusAttributes        attrs;

        //create result document for reporting status to DirXML engine
        //can't add source info yet since we don't have the driver's instance name
        result = new XDSSchemaResultDocument();

        try
        {
            XDSInitDocument init;

            //parse initialization document
            init = new XDSInitDocument(initXML);

            //append a <source> element to the result document
            setDriverRDN(init.rdn());
            setTrace();
            appendSourceInfo(result);

            //get driver parameters from the initialization doc
            init.parameters(driverParams);

            //MODIFY:  put your application's schema in the result document
            XDSSchemaDefElement schemaDef;//the <schema-def> element from the result doc
            XDSClassDefElement  classDef;//a <class-def> element from the result doc
            XDSAttrDefElement   attrDef;//an <attr-def> element from the result doc

            //a driver that supports a real application would do whatever is
            //necessary to create an XDS schema description document and
            //return it; for example, it might create an instance of the
            //Subscriber object and query the application schema; if the
            //application has an invariant schema, the driver might just
            //have an embedded XML document describing the schema in XDS
            //terms; this embedded document could then be returned to DirXML
            //wrapped by an XmlDocument object
            //
            //however, since we are just a skeleton, return a fake schema with
            //a warning that this method isn't really implemented
            schemaDef = result.appendSchemaDefElement();
            schemaDef.setApplicationName(APPLICATION_NAME);
            schemaDef.setHierarchical(true);
            classDef = schemaDef.appendClassDefElement();
            classDef.setClassName("fake-class-1");
            classDef.setContainer(true);
            attrDef = classDef.appendAttrDefElement();
            attrDef.setAttrName("fake-attr-1");
            attrDef.setCaseSensitive(true);
            attrDef.setMultiValued(true);
            attrDef.setNaming(true);
            attrDef.setReadOnly(true);
            attrDef.setRequired(true);
            attrDef.setType(ValueType.STRING);
            classDef = schemaDef.appendClassDefElement();
            classDef.setClassName("fake-class-2");
            classDef.setContainer(false);
            attrDef = classDef.appendAttrDefElement();
            attrDef.setAttrName("fake-attr-2");
            attrDef.setCaseSensitive(false);
            attrDef.setMultiValued(false);
            attrDef.setNaming(false);
            attrDef.setReadOnly(false);
            attrDef.setRequired(false);
            attrDef.setType(ValueType.INT);

            attrs = StatusAttributes.factory(
                    StatusLevel.WARNING,
                    StatusType.DRIVER_GENERAL,
                    null);//event-id
            XDSUtil.appendStatus(
                result,//doc to append to
                attrs,//status attribute values
                Errors.NO_SCHEMA);//description
        }//try
        catch(Exception e)//don't want to catch Error class with Throwable
        {
            //e instance of XDSException:
            //
            //  init document is malformed or invalid -- or --
            //  it is missing required parameters or contains
            //  illegal parameter values
            //e instance of RuntimeException:
            //  e.g., NullPointerException
            //clean out partial schema
            result.empty();
            attrs = StatusAttributes.factory(
                    StatusLevel.FATAL,
                    StatusType.DRIVER_STATUS,
                    null);//event-id
            XDSUtil.appendStatus(
                result,//doc to append to
                attrs,//status attribute values
                null,//description
                e,//exception
                XDSUtil.appendStackTrace(e),//append stack trace?
                initXML);//xml to append
        }//catch

        //return result doc w/ schema and status to DirXML engine
        return result.toXML();
    }//getSchema(XmlDocument):XmlDocument
}//class SeletonDriverShim
