/*******************************************************************************

 * $Novell: OperationalAttrs.java,v 1.8 2003/08/21 11:36:59 $

 * Copyright (c) 2001 Novell, Inc. All Rights Reserved.

 *

 * THIS WORK IS SUBJECT TO U.S. AND INTERNATIONAL COPYRIGHT LAWS AND

 * TREATIES. USE AND REDISTRIBUTION OF THIS WORK IS SUBJECT TO THE LICENSE

 * AGREEMENT ACCOMPANYING THE SOFTWARE DEVELOPMENT KIT (SDK) THAT CONTAINS

 * THIS WORK. PURSUANT TO THE SDK LICENSE AGREEMENT, NOVELL HEREBY GRANTS TO

 * DEVELOPER A ROYALTY-FREE, NON-EXCLUSIVE LICENSE TO INCLUDE NOVELL'S SAMPLE

 * CODE IN ITS PRODUCT. NOVELL GRANTS DEVELOPER WORLDWIDE DISTRIBUTION RIGHTS

 * TO MARKET, DISTRIBUTE, OR SELL NOVELL'S SAMPLE CODE AS A COMPONENT OF

 * DEVELOPER'S PRODUCTS. NOVELL SHALL HAVE NO OBLIGATIONS TO DEVELOPER OR

 * DEVELOPER'S CUSTOMERS WITH RESPECT TO THIS CODE.

 *

 * $name:         OperationalAttrs.java

 * $description:  OperationalAttrs.java shows how to get LDAP operational

 *                attributes of a entry.

 *

 *                LDAP operational attributes are not returned in search

 *                or read results unless explicitly requested by name.

 ******************************************************************************/

import com.novell.ldap.LDAPAttribute;

import com.novell.ldap.LDAPAttributeSet;

import com.novell.ldap.LDAPConnection;

import com.novell.ldap.LDAPEntry;

import com.novell.ldap.LDAPException;

import com.novell.ldap.LDAPDSConstants;

import java.util.Date;

import java.util.Enumeration;

import java.util.Iterator;

import java.util.TimeZone;

import java.text.DateFormat;

import java.text.SimpleDateFormat;

import java.text.ParseException;

import java.io.UnsupportedEncodingException;



public class OperationalAttrs implements LDAPDSConstants {

    public static void main( String[] args )

    {

        if ( args.length != 5 ) {

            System.err.println(

                "Usage:   java OperationalAttrs <host Name> <port number>"

                                                + " <login dn> <password>"

                                             + "\n              <entryDN>");

            System.err.println( "Example: java OperationalAttrs Acme.com 389 "

                                              + "\"cn=Admin,o=Acme\" secret\n"

                                      + "              \"cn=James,o=Sales\"");

            System.exit(1);

        }



        int ldapVersion   = LDAPConnection.LDAP_V3;

        int ldapPort      = Integer.parseInt(args[1]);

        String ldapHost   = args[0];

        String loginDN    = args[2];

        String password   = args[3];

        String entryDN    = args[4];

        LDAPConnection lc = new LDAPConnection();



        try {

            // connect to the server

            lc.connect( ldapHost, ldapPort );

            // bind to the server

            lc.bind( ldapVersion, loginDN, password.getBytes("UTF8") );



            // get the LDAP operational attributes of the test entry

            printOperationalAttrs( lc, entryDN );



            lc.disconnect();

        }

        catch( LDAPException e ) {

            System.out.println( "\nError: " + e.toString() );

            System.exit(1);

        }

        catch( UnsupportedEncodingException e ) {

            System.out.println( "Error: " + e.toString() );

        }

        System.exit(0);

    }



    // printOperationalAttrs() searches the directory to get all the LDAP

    // operational attributes  of the entry specified by dn, and then

    // prints them out

    public static void printOperationalAttrs( LDAPConnection lc, String dn)

        throws LDAPException

    {

        int flags;

        Date date = null;

        String attributeName, attrValue;

        Iterator allAttributes;

        Enumeration allValues;

        LDAPAttribute attribute;

        LDAPAttributeSet attributeSet;

        // explicitely specify the LDAP operational attribute names

        String returnAttrs[] = { "createTimeStamp",

                                 "creatorsName",

                                 "entryFlags",

                                 "federationBoundary",

                                 "localEntryID",

                                 "modifiersName",

                                 "modifyTimeStamp",

                                 "structuralObjectClass",

                                 "subordinateCount",

                                 "subschemaSubentry"

                                 };



        try {

            // read the entry

            LDAPEntry entry = lc.read( dn, returnAttrs );



            // printout the entry's LDAP operational attributes and values

            System.out.println( "    Entry: " + dn );

            System.out.println( "    LDAP Operational Attributes: ");

            attributeSet = entry.getAttributeSet();

            allAttributes = attributeSet.iterator();



            while(allAttributes.hasNext()) {

                attribute = (LDAPAttribute)allAttributes.next();

                attributeName = attribute.getName();

                if ( (allValues = attribute.getStringValues()) != null

                  && (attrValue = (String) allValues.nextElement()) != null) {

                    if ( attributeName.equalsIgnoreCase( "modifyTimeStamp" )) {

                        System.out.println("        modifyTimeStamp: "

                                                   + attrValue + " (UTC)");

                        PrintTime( attrValue );

                    }

                    else if (attributeName.equalsIgnoreCase("createTimeStamp")){

                        System.out.println("        createTimeStamp: "

                                                   + attrValue + " (UTC)");

                        PrintTime( attrValue );

                    }

                    else if ( attributeName.equalsIgnoreCase( "entryFlags" )) {

                        if ( (flags = Integer.parseInt( attrValue )) != 0 )

                            EntryFlags( flags );

                    }

                    else

                        System.out.println("        " + attributeName +

                                                              ": " + attrValue);

                }

            }

        }

        catch( LDAPException e ) {

            System.err.println( "\nOperationalAttrs() failed.");

            System.err.println( "\nError: " + e.toString() );

            System.exit(1);

        }

    }



    // PrintTime() prints utc in text format

    public static void PrintTime( String utc )

    {

        Date date = null;

        // setup x.208 generalized time formatter

        DateFormat formatter = new SimpleDateFormat("yyyyMMddHHmmss'Z'");



        try {

            // parse utc into Date

            date = formatter.parse( utc );

        }

        catch(ParseException pe) {

            System.out.println( "Error: " + pe.toString() );

        }



        System.out.println("                     " + date + " (UTC)");

    }



    // EntryFlags() decodes and displays entry flags

    public static void EntryFlags( int flags ) {



        StringBuffer entryFlags = new StringBuffer();



        if ( (flags & LDAP_DS_ALIAS_ENTRY) != 0 )

            entryFlags.append("AliasEntry ");

        if (( flags & LDAP_DS_PARTITION_ROOT) != 0 )

            entryFlags.append("PartionRoot ");

        if (( flags & LDAP_DS_CONTAINER_ENTRY) != 0 )

            entryFlags.append("ContainerEntry ");

        if (( flags & LDAP_DS_CONTAINER_ALIAS) != 0 )

            entryFlags.append("ContainerAlias ");

        if (( flags & LDAP_DS_MATCHES_LIST_FILTER) != 0 )

            entryFlags.append("MatchesListFilter ");

        if (( flags & LDAP_DS_REFERENCE_ENTRY) != 0 )

            entryFlags.append("ReferenceEntry ");

        if (( flags & LDAP_DS_40X_REFERENCE_ENTRY) != 0 )

            entryFlags.append("40XReferenceEntry ");

        if (( flags & LDAP_DS_BACKLINKED) != 0 )

            entryFlags.append("Backlinked ");

        if (( flags & LDAP_DS_NEW_ENTRY) != 0 )

            entryFlags.append("NewEntry ");

        if (( flags & LDAP_DS_AUDITED) != 0 )

            entryFlags.append("TemporaryEntry ");

        if (( flags & LDAP_DS_AUDITED) != 0 )

            entryFlags.append("Audited ");

        if (( flags & LDAP_DS_ENTRY_NOT_PRESENT) != 0 )

            entryFlags.append("EntryNotPresent ");

        if (( flags & LDAP_DS_ENTRY_VERIFY_CTS) != 0 )

            entryFlags.append("EntryVerifyCTS ");

        if (( flags & LDAP_DS_ENTRY_DAMAGED) != 0 )

            entryFlags.append("EntryDamaged");



        System.out.println("    entryFlags: " + entryFlags);

    }

}

