/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/java/io/spi/DataAccessor.java $
  $Revision: 5 $
  $Modtime: 1/14/00 2:58p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.java.io.spi;

import java.io.IOException;

/**
 * Provides an interface that allows read and/or write access to a given
 * resource.  
 * 
 * <p>This interface is not needed or used by the user application layer,  
 * because it is an SPI interface, which provides for plugging a device
 * into the com.novell.java.io Streams world.
 *
 * <p>The DataAccessor interface contains all the methods needed to 
 * implement InputStream, OutputStream, and RandomAccessFile. Any 
 * class that implements this interface can be accessed as a stream
 * or a random access device if supported. It requires the construction 
 * of three fully instantiatable classes that operate on a DataAccessor: 
 * NInputStream, NOutputStream, and RandomAccess.
 *
 * <p>NInputStream, NOutputStream, and RandomAccess are the super 
 * classes of any corresponding classes that might extend them by
 * supplying any custom data constructors. For example, NFileInputStream
 * extends NInputStream by providing support for mode and data selector
 * parameters that are custom to a file access.
 *
 * <p>For a device accessor to be able to operate, the device must be
 * open. To facilitate the opening of a device, the DataAccessable 
 * interface is implemented.
 */

public interface DataAccessor
{
   /**
    * Read/Input Stream Type.
    *
    * <p>(READ = 1)</p>
    */
   public static final int READ = 1;

   /**
    * Write/Output Stream Type.
    *
    * <p>(WRITE = 2)</p>
    */
   public static final int WRITE = 2;

   /**
    * RANDOM READ/WRITE Stream Type.
    *
    * <p>(RANDOM = 3)</p>
    */
   public static final int RANDOM = 3;


   /**
    * Returns the number of bytes left from the current stream
    * position.
    *
    * @return The number of bytes available to read (without blocking)
    *         from the current position in the stream.
    *
    * @exception  IOException  When there is an error getting the current
    *                          stream position.
    */
   public int available() throws IOException;

   /**
    * Closes the stream.
    *
    * <p>Any data that has been written to the stream but has not 
    * yet been written to actual resource is flushed before the
    * stream is closed.</p>
    *
    * @exception  IOException When there is an error flushing data or
    *                         closing the stream.
    */
   public void close() throws IOException;

   /**
    * Flushes all unwritten data.
    *
    * @exception IOException  When there is an error writing the data.
    */
   public void flush() throws IOException;

   /**
    * Determines whether the mark() and reset() methods are supported.
    *
    * <p>If this method returns FALSE, reset() will throw an exception
    *  if called.</p>
    * 
    * @return A boolean set to TRUE if mark() and reset() are supported;
    *         otherwise FALSE is returned. 
    */
   public boolean markSupported();

   /**
    * Marks the current stream position, which can be restored by
    * calling reset().</p>
    *
    * @param limit The number of bytes that can be read before
    *              the mark is invalidated.
    */
   public void mark (int limit);

   /**
    * Reads and returns the next byte in the stream.
    *
    * @return The next byte in the stream. Returns -1\
    *         if end-of-file was reached.
    *
    * @exception IOException When an error occurs reading the next
    *                        byte.
    */
   public int read() throws IOException;

   /**
    * Reads bytes from the stream and fills a byte array with the
    * data read.
    *
    * <p>'b.length' bytes are read and stored in 'b'. If less than 
    * 'b.length' bytes are read, the the end-of-file was reached.
    * If no bytes are read and end-of-file was reached, then -1 is
    * returned.</p>
    *
    * @param  b The byte array to which to write.
    * 
    * @return The number of bytes actually read, or -1 if
    *         end-of-file was reached.
    *
    * @exception IOException When an error occurs reading bytes from 
    *                        the stream.
    */
   public int read(byte[] b) throws IOException;

   /**
    * Reads bytes from the stream and places the bytes at an offset
    * within a byte array. 
    *
    * <p>'len' bytes are read into 'b' starting at position off. If less than
    * 'len' bytes are read, then the end-of-file was reached. If no bytes
    * are read and end-of-file was reached, -1 is returned.</p>
    *
    * @param  b   The byte array to which to write.
    * @param  off The array offset where writing should start.
    * @param  len The number of bytes to read.
    *
    * @return The number of bytes actually read, or -1 if
    *         end-of-file was reached.
    *
    * @exception IOException When an error occurs reading bytes from 
    *                        the stream.
    */
   public int read(byte[] b, int off, int len) throws IOException;

   /**
    * Writes one byte to the stream.
    *
    * @param b The byte to write.
    *
    * @exception IOException When an error occurs writing the byte.
    */
   public void write(int b) throws IOException;

   /**
    * Writes an array of bytes to the stream.
    *
    * <p>'b.length' bytes are written.</p>
    *
    * @param b The byte array to write.
    *
    * @exception IOException  When an error occurs writing the
    *                         bytes.
    */
   public void write(byte[] b) throws IOException;

   /**
    * Writes a specified number of bytes from a byte array to the
    * stream, starting at a specified offset in the array.
    *
    * @param b   The source array.
    * @param off The array offset from which to start writing.
    * @param len The number of bytes to write.
    *
    * @exception IOException When an error occurs writing the bytes.
    */
     
   public void write(byte[] b, int off, int len) throws IOException;

   /**
    * Resets the stream position to the location saved by mark().
    *
    * <p>If mark() and reset() are not supported by the stream,
    * an exception is always thrown when reset() is called.</p>
    *
    * @exception IOException When an error occurs moving the current
    *                        stream position to the saved location.
    */
   public void reset()
      throws IOException;

   /**
    * Moves the stream position forward a specified number of bytes.
    *
    * @param  n The number of bytes to skip.
    *
    * @return The number of bytes skipped.
    *
    * @exception IOException When an error occurs in changing
    *                        the stream position.
    */
   public long skip(long n) throws IOException;

   /**
    * Returns the total number of bytes in the stream.
    *
    * @return The total number of bytes contained in the stream.
    *
    * @exception IOException When an error occurs getting the stream's
    *                        length.
    */
   public long length() throws IOException;

   /**
    * Moves the stream position to an absolute position in the stream.
    *
    * @param  pos The absolute position to which to move.
    *
    * @return The absolute position moved to.
    *
    * @exception IOException When an error occurs setting stream
    *                            position.
    */
   public long seek(long pos) throws IOException;

   /**
    * Changes the stream length.
    *
    * <p>If the stream is smaller than the desired length, the stream
    * is extended to the desired length using a byte value 0. If the
    * stream is larger than the desired length, the stream is truncated
    * at the desired length.</p>
    *
    * @param len The desired absolute length of the stream.
    *
    * @exception IOException When an error occurs setting the length of
    *                        the stream.
    */
   public void setLength(long len) throws IOException;
}
