/* ***************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/bindery/BinderyUtil.java $
  $Revision: 6 $
  $Modtime: 10/18/99 10:35a $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

 ****************************************************************************/

package com.novell.service.bindery;

/**
 * Defines a number of constants for use with the Bindery. Also, 
 * provides one method, byteswapID(int), for swapping the byte ordering
 * for making native calls.
 */

public final class BinderyUtil
{
   /**
    * Maximum name size an object in the bindery can have (in ASCII
    * chars, including the null terminator).
    *
    * <p>(NW_MAX_OBJECT_NAME_LEN = 48)
    * </p>
    */
   public final static int  NW_MAX_OBJECT_NAME_LEN    = 48; // includes null

   // Bindery Object types come in two forms, wire-format and human-readable

   /**
    * Bindery type used as a wildcard in searching.
    *
    * <p>(NW_ANY_TYPE = -1)
    * </p>
    */
   public final static int NW_ANY_TYPE = -1;

   /**@internal
    */
   public final static int NW_PRINT_QUEUE_TYPE      = 0x0300;

   /**@internal
    */
   public final static int NW_SERVER_TYPE           = 0x0400;

   /**@internal
    */
   public final static int NW_ARCHIVE_QUEUE_TYPE    = 0x0800;

   /**@internal
    */
   public final static int NW_JOB_QUEUE_TYPE        = 0x0a00;

   /**@internal
    */
   public final static int NDS_TREE_TYPE            = 0x7802;

   /**@internal
    */
   public final static int NDS_TREE                 = 0x6b02;

   /**
    * Bindery object type for a print queue object.
    *
    * <p>(NW_PRINT_QUEUE_TYPE_HR = 0x0003)
    * </p>
    */
   public final static int NW_PRINT_QUEUE_TYPE_HR   = 0x0003;

   /**
    * Bindery object type for a NetWare server object.
    *
    * <p>(NW_SERVER_TYPE_HR = 0x0004)
    * </p>
    */
   public final static int NW_SERVER_TYPE_HR        = 0x0004;

   /**
    * Bindery object type for an archive queue object.
    *
    * <p>(NW_ARCHIVE_QUEUE_TYPE_HR = 0x0008)
    * </p>
    */
   public final static int NW_ARCHIVE_QUEUE_TYPE_HR = 0x0008;

   /**
    * Bindery object type for a job queue object.
    *
    * <p>(NW_JOB_QUEUE_TYPE_HR = 0x000a)
    * </p>
    */
   public final static int NW_JOB_QUEUE_TYPE_HR     = 0x000a;

   /**
    * Bindery object type for an NDS tree object.
    *
    * <p>(NDS_TREE_TYPE_HR = 0x0278)
    * </p>
    */
   public final static int NDS_TREE_TYPE_HR         = 0x0278;

   // Need your ID set to -1 on first call to NWScanObject()

   /**@internal
    * Object ID used for first call to search for bindery objects.
    */
   public final static int NW_FIRST_SCAN_ID  = -1;

   // Bits for bindery object and property flags.
   // Bindery objects and properties can be static or dynamic.

   /**
    * Flag identifying a bindery object or bindery object property as
    * static. A static object or property is saved to disk and is
    * restored when the server is restarted.
    *
    * <p>(BF_STATIC = 0x00)
    * </p>
    * @see #BF_DYNAMIC
    */
   public final static int BF_STATIC        = 0x00;

   /**
    * Flag identifying a bindery object or bindery object property as
    * dynamic. A dynamic object or property is only kept in memory and
    * is not restored when a server is restarted.
    *
    * <p>(BF_DYNAMIC = 0x01)
    * </p>
    * @see #BF_STATIC
    */
   public final static int BF_DYNAMIC       = 0x01;

   // Bindery object properties can have one item or a set of items.

   /**
    * Flag identifying a bindery property as containing a single item.
    *
    * <p>(BF_ITEM = 0x00)
    * </p>
    * @see #BF_SET
    */
   public final static int BF_ITEM          = 0x00;

   /**
    * Flag identifying a bindery property as containing a set of items.
    *
    * <p>(BF_SET = 0x02)
    * </p>
    * @see #BF_ITEM
    */
   public final static int BF_SET           = 0x02;

   // Security access levels

   /**
    * Bindery security flag allowing any attached connection to have
    * read privileges
    *
    * <p>(BS_ANY_READ = 0x00)
    * </p>
    * @see #BS_ANY_WRITE
    * @see #BS_LOGGED_READ
    * @see #BS_OBJECT_READ
    * @see #BS_SUPER_READ
    * @see #BS_BINDERY_READ
    */
   public final static int BS_ANY_READ      = 0x00;  // anyone can read this

   /**
    * Bindery security flag allowing a logged in object to have
    * read privileges
    *
    * <p>(BS_LOGGED_READ = 0x01)
    * </p>
    * @see #BS_ANY_READ
    * @see #BS_LOGGED_WRITE
    * @see #BS_OBJECT_READ
    * @see #BS_SUPER_READ
    * @see #BS_BINDERY_READ
    */
   public final static int BS_LOGGED_READ   = 0x01;  // only logged in users

   /**
    * Bindery security flag allowing the owner to have object read
    * privileges
    *
    * <p>(BS_OBJECT_READ = 0x02)
    * </p>
    * @see #BS_ANY_READ
    * @see #BS_LOGGED_READ
    * @see #BS_OBJECT_WRITE
    * @see #BS_SUPER_READ
    * @see #BS_BINDERY_READ
    */
   public final static int BS_OBJECT_READ   = 0x02;  // by same obj or super

   /**
    * Bindery security flag allowing the supervisor to have read
    * privileges
    *
    * <p>(BS_SUPER_READ = 0x03)
    * </p>
    * @see #BS_ANY_READ
    * @see #BS_LOGGED_READ
    * @see #BS_OBJECT_READ
    * @see #BS_SUPER_WRITE
    * @see #BS_BINDERY_READ
    */
   public final static int BS_SUPER_READ    = 0x03;  // by supervisor only

   /**
    * Bindery security flag allowing the server (bindery) to have
    * read privileges
    *
    * <p>(BS_BINDERY_READ = 0x04)
    * </p>
    * @see #BS_ANY_READ
    * @see #BS_LOGGED_READ
    * @see #BS_OBJECT_READ
    * @see #BS_SUPER_READ
    * @see #BS_BINDERY_WRITE
    */
   public final static int BS_BINDERY_READ  = 0x04;  // by the bindery only

   /**
    * Bindery security flag allowing any attached connection to have
    * write privileges
    *
    * <p>(BS_ANY_WRITE = 0x00)
    * </p>
    * @see #BS_ANY_READ
    * @see #BS_LOGGED_WRITE
    * @see #BS_OBJECT_WRITE
    * @see #BS_SUPER_WRITE
    * @see #BS_BINDERY_WRITE
    */
   public final static int BS_ANY_WRITE     = 0x00;  // anyone can WRITE this

   /**
    * Bindery security flag allowing a logged in object to have write
    * privileges
    *
    * <p>(BS_LOGGED_WRITE = 0x10)
    * </p>
    * @see #BS_ANY_WRITE
    * @see #BS_LOGGED_READ
    * @see #BS_OBJECT_WRITE
    * @see #BS_SUPER_WRITE
    * @see #BS_BINDERY_WRITE
    */
   public final static int BS_LOGGED_WRITE  = 0x10;  // only logged in users

   /**
    * Bindery security flag allowing the owner object to have write
    * privileges
    *
    * <p>(BS_OBJECT_WRITE = 0x20)
    * </p>
    * @see #BS_ANY_WRITE
    * @see #BS_LOGGED_WRITE
    * @see #BS_OBJECT_READ
    * @see #BS_SUPER_WRITE
    * @see #BS_BINDERY_WRITE
    */
   public final static int BS_OBJECT_WRITE  = 0x20;  // by same obj or super

   /**
    * Bindery security flag allowing the supervisor to have write
    * privileges
    *
    * <p>(BS_SUPER_WRITE = 0x30)
    * </p>
    * @see #BS_ANY_WRITE
    * @see #BS_LOGGED_WRITE
    * @see #BS_OBJECT_WRITE
    * @see #BS_SUPER_READ
    * @see #BS_BINDERY_WRITE
    */
   public final static int BS_SUPER_WRITE   = 0x30;  // by supervisor only

   /**
    * Bindery security flag allowing the server (bindery) to have
    * write privileges
    *
    * <p>(BS_BINDERY_WRITE = 0x40)
    * </p>
    * @see #BS_ANY_WRITE
    * @see #BS_LOGGED_WRITE
    * @see #BS_OBJECT_WRITE
    * @see #BS_SUPER_WRITE
    * @see #BS_BINDERY_READ
    */
   public final static int BS_BINDERY_WRITE = 0x40;  // by the bindery only

   /**
    * Private default constructor to insure this is a static, singleton class.
    */
   private BinderyUtil () {};
   
   /**
    * Swaps the byte ordering on a 16-bit integer for making native calls.
    * This must be done when sending bindery object types over the wire,
    * because the wire format for type 0x1 is 0x100. Only the lower
    * 16 bits are swapped, but the parameter and return values are in
    * 32-bit Java int format.
    *
    * @param orig The original 32-bit Java int.
    *
    * @return The new 32-bit int with the lower 16 bits swapped.
    */
   public static int byteswapID (
         int orig)
   {
      int byte1, byte2, swapped;
      
      byte1   = (orig & 0x000000ff) << 8;
      byte2   = (orig & 0x0000ff00) >> 8;
      swapped = (orig & 0xffff0000) | byte1 | byte2;

      return swapped;
   } // byteswapID ()
   
} // class BinderyUtil
