/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/file/nw/NFileInputStream.java $
  $Revision: 4 $
  $Modtime: 12/17/99 10:44a $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.file.nw;

import java.io.IOException;
import java.io.FileNotFoundException;

import com.novell.java.io.NInputStream;
import com.novell.java.io.DataAccessable;
import com.novell.java.io.SubordinateAccessOnlyException;
import com.novell.java.io.NoSubordinateAccessException;
import com.novell.java.io.InputStreamNotSupportedException;

import com.novell.java.io.spi.DataAccessor;

/**
 * Opens an input stream on a NetWare file or Extended Attribute.
 *
 * <p>NFileInputStream extends NInputStream by adding convenience 
 * constructors for the various custom parameters.
 *
 * <p>The DataAccessableParameters class can also be used by the user 
 * application layer to obtain constant flags for the various constructors 
 * of this class, or can instantiate a DataAccessableParameters object and
 * pass it into the NInputStream constructor.</p>
 *
 * @see NInputStream
 * @see DataAccessableParameters
 */

public class NFileInputStream extends NInputStream
{
/* **************************************************************************
   non-subordinate stream
****************************************************************************/

   /**
    * Constructs an NFileInputStream object using default parameters.
    *
    * <p>Opens an input stream through the specified DataAccessable
    * object, using default parameters.<p/>
    *
    * @param accessor The accessor object that will be called
    *                 to open the stream.
    *
    * @exception IOException When an I/O error occurs.
    *
    * @exception SubordinateAccessOnlyException If the accessor does not
    *            support a stream to itself, for example a directory.
    *
    * @exception InputStreamNotSupportedException If the accessor does
    *            not support an input stream.
    */

   public NFileInputStream(DataAccessable accessor)
      throws IOException
   {
      super(accessor);
   }

   /**
    * Constructs an NFileInputStream object with openFlags parameter.
    *
    * <p>Given an openFlags object, open an input stream through the 
    * specified DataAccessable object.<p/>
    *
    * @param accessor  The accessor object that will be called
    *                  to open the stream.
    *
    * @param openFlags The flags that are used in opening the
    *                  stream in different modes.
    *
    *
    * @exception IOException When an I/O error occurs.
    *
    * @exception SubordinateAccessOnlyException If the accessor does not
    *            support a stream to itself, for example a directory.
    *
    * @exception InputStreamNotSupportedException If the accessor does not
    *            support an input stream.
    */

   public NFileInputStream(DataAccessable accessor, int openFlags)
      throws IOException
   {
      this(
         accessor, 
         openFlags, 
         DataAccessableParameters.DEFAULT_DATASELECTOR);
   }

   /**
    * Constructs an NFileInputStream object with openFlags and
    * dataSelector parameters.
    *
    * <p>Given an openFlags and dataSelector, opens an input 
    * stream through the specified DataAccessable object.<p/>
    *
    * @param accessor The DataAccessable object that will be 
    *                 called to open the stream.
    *
    * @param openFlags The flags that are used in opening the
    *                  stream in different modes.
    *
    * @param dataSelector An integral value that allows the
    *                     individual data stream on a file to be
    *                     selected.
    *
    * @exception IOException When an I/O error occurs.
    *
    * @exception SubordinateAccessOnlyException If the accessor does not
    *            support a stream to itself, for example a directory.
    *
    * @exception InputStreamNotSupportedException If the accessor does not
    *            support an input stream.
    */

   public NFileInputStream(
      DataAccessable accessor, 
      int openFlags, 
      int dataSelector)
      throws IOException
   {
      DataAccessableParameters custom = new 
         DataAccessableParameters(openFlags, dataSelector);
      
      setDataAccessor(accessor.openStream(DataAccessor.READ, custom));
   }

/* **************************************************************************
   subordinate stream
****************************************************************************/

   /**
    * Constructs an NFileInputStream object in the subordinate input
    * stream with default parameters.
    *
    * <p>Opens a subordinate input stream through the specified 
    * DataAccessable object default parameters.<p/>
    *
    * @param name The atomic subordinate name to open.
    *
    * @param accessor The accessor object that will be called
    *                 to open the stream.
    *
    * @exception  IOException When an I/O error occurs.
    *
    * @exception NoSubordinateAccessException If the accessor does not
    *            support opening a subordinate access, for example, a file.
    *
    * @exception InputStreamNotSupportedException If the accessor does not
    *            support an input stream.
    */

   public NFileInputStream(
      String name,
      DataAccessable accessor)
      throws IOException
   {
      super(name, accessor);
   }

   /**
    * Constructs an NFileInputStream object in the subordinate input
    * stream with the openFlags parameter.
    *
    * <p>Given an openFlags, opens a subordinate input stream through 
    * the specified DataAccessable object.<p/>
    *
    * @param name The atomic subordinate name to open.
    *
    * @param accessor The accessor object that will be called
    *                 to open the stream.
    *
    * @param openFlags The flags that are used in opening the
    *                  stream in different modes.
    *
    *
    * @exception IOException When an I/O error occurs.
    *
    * @exception NoSubordinateAccessException If the accessor does not
    *            support opening a subordinate access, for example, a file.
    *
    * @exception InputStreamNotSupportedException If the accessor does not
    *            support an input stream.
    */

   public NFileInputStream(
      String name,
      DataAccessable accessor, 
      int openFlags)
      throws IOException
   {
      this(
         name,
         accessor, 
         openFlags, 
         DataAccessableParameters.DEFAULT_DATASELECTOR);
   }

   /**
    * Constructs an NFileInputStream object in the subordinate input
    * stream with the openFlags and dataSelector paramters.
    *
    * <p>Given an openFlags and dataSelector, opens a subordinate input 
    * stream through the specified DataAccessable object.<p/>
    *
    * @param name The atomic subordinate name to open.
    *
    * @param accessor The DataAccessable object that will be 
    *                 called to open the stream.
    *
    * @param openFlags The flags that are used in opening the
    *                  stream in different modes.
    *
    * @param dataSelector An integral value that allows the
    *                     individual data stream on a file to be
    *                     selected.
    *
    * @exception IOException When an I/O error occurs.
    *
    * @exception NoSubordinateAccessException If the accessor does not
    *            support opening a subordinate access, for example, a file.
    *
    * @exception InputStreamNotSupportedException If the accessor does not
    *            support an input stream.
    */

   public NFileInputStream(
      String name,
      DataAccessable accessor, 
      int openFlags, 
      int dataSelector)
      throws IOException
   {
      DataAccessableParameters custom = new 
         DataAccessableParameters(openFlags, dataSelector);
      
      setDataAccessor(accessor.openStream(name, DataAccessor.READ, custom));
   }
}
