
/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/nds/NdsBoolean.java $
  $Revision: 10 $
  $Modtime: 10/15/99 3:26p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import com.novell.service.schema.SchemaBoolean;

/**
 * Provides access to attribute values of the Boolean syntax.
 * The Boolean value represents either TRUE (1) or FALSE (0).
 * Two Boolean attributes match for equality if they are both
 * TRUE or both FALSE. Any attribute defined using this syntax
 * is single-valued.
 *
 * <p>Matching Rules: Equality</p>
 *
 * @see NdsSyntaxId
 */
public class NdsBoolean
   implements NdsAttributeValue, SchemaBoolean, Cloneable, Serializable
{
   private static final int matchingRules = equality;

  /**
   * @internal
   */
   protected boolean value;
   private String name;

  /**
   * Constructs an NdsBoolean object based on the specified value
   * parameter.
   * 
   * @param value A Boolean value (TRUE or FALSE) to be stored.
   */
   public NdsBoolean (
         boolean value)
   {
      this ("", value);

   }

  /**
   * @internal
   */
   protected NdsBoolean ()
   {
      value = false;
      name = null;
   }

   /*
    * Private constructor; not documented!
    */
   private NdsBoolean (
         String name,
         boolean value)
   {
      this.value = value;
      this.name = name;

   }

  /**
   * @internal
   */
   protected NdsBoolean (
         NdsBoolean aBoolean)
   {
      this.value = aBoolean.value;
      this.name = aBoolean.name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically.
   *
   * The compareTo method does not apply to the NdsBoolean syntax
   * because this syntax does not support the ordering and substrings
   * matching rules.
   */
   public int compareTo (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals (
         Object anObject)
   {
      if ((anObject != null) && (anObject instanceof NdsBoolean))
      {
         NdsBoolean anotherBoolean = (NdsBoolean) anObject;

         return (value == anotherBoolean.booleanValue () ? true : false);
      }
      return (false);
   }

  /**
   * @internal
   */
   public String getName ()
   {
      return (name);
   }

  /**
   * @internal
   */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.BOOLEAN);
   }


   // ******************** SchemaBoolean Interface ********************

   /**
   * Returns a Boolean object with the value of TRUE or FALSE.
   *
   * @return The Boolean value of this object.
   */
   public boolean booleanValue ()
   {
      return (value);
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Compares two Objects using the approximate matching rule.
   * 
   * The approximate method does not apply to the NdsBoolean syntax
   * because this syntax does not support the approximate equals
   * matching rule.
   */
   public boolean approximate (
         Object anObject)
      throws Exception
   {
     throw (new Exception ());
   }

  /**
   * Returns the int that represents the syntax ID.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.BOOLEAN_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules. The NdsBoolean syntax supports only the
   * equality matching rule.
   *
   * @param matchingRules Set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
         return (true);
      else
         return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone ()
   {
      try
      {
         NdsBoolean aBoolean = (NdsBoolean) super.clone ();

         aBoolean.value = this.value;
         aBoolean.name = this.name;

         return (aBoolean);
      }
      catch (CloneNotSupportedException e)
      {
         // this shouldn't happen, since we are Cloneable
         throw (new InternalError ());
      }

   } /* clone () */

  /**
   * Generates and returns a string representation of the object. In
   * general, the toString method returns a string that textually 
   * represent this object. The result should be a concise but
   * informative representation that is easy to read.
   *
   * @return A String representation of the object.
   */
   public String toString ()
   {
      return (value ? "true" : "false");
   }


   // ******************** NdsBoolean Class ********************

  /**
   * Returns the Boolean value stored in the object data member.
   * 
   * @return The stored Boolean value.
   */
   public boolean getBoolean ()
   {
      return (value);
   }

} /* NdsBoolean */


