
/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/nds/NdsIteratorControls.java $
  $Revision: 17 $
  $Modtime: 10/15/99 3:33p $

  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import javax.naming.directory.SearchControls;


/**
 * Encapsulates factors that determine the scope of iterators and 
 * what gets returned as a result of the iterator. The sort index is an 
 * array of strings. The first string is the primary index, the second 
 * string is the secondary index, and so forth. The default sort indexes 
 * are BaseClass and Name.
 *
 * <p>An NdsIteratorControls instance is not synchronized against 
 * concurrent multithreaded access. Multiple threads trying to access
 * and modify a single NdsIteratorControls instance should synchronize
 * access to the object.
 *
 * <p>The serialized form of an NdsIteratorControls object consists of 
 * the sort indexes (an array of strings), and the serialized form of 
 * search controls.
 */
public class NdsIteratorControls
   extends SearchControls implements Serializable
{
   private String[] indexSelect;
   private String[] sortKey;
   private int scalability;

  /**
   * Specifies that a scalable server is preferred.
   *
   * <p>(DS_ITR_PREFER_SCALABLE = 0)</p>
   */
   public final static int DS_ITR_PREFER_SCALABLE           = 0;

  /**
   * Specifies that a scalable server is required.
   *
   * <p>(DS_ITR_REQUIRE_SCALABLE = 1)</p>
   */
   public final static int DS_ITR_REQUIRE_SCALABLE          = 1;

  /**
   * Specifies emulation mode even if a scalable server is available.
   *
   * <p>(DS_ITR_FORCE_EMULATION = 2)</p>
   */
   public final static int DS_ITR_FORCE_EMULATION           = 2;

  /**
   * Constructs an NdsIteratorControls instance using the default
   * sort index CN. The indexSelect and sortKey are set to NULL,
   * and scalability is set to preferred.
   */
   public NdsIteratorControls ()
   {
      indexSelect = null;
      sortKey = null;
      scalability = DS_ITR_PREFER_SCALABLE;
   }

  /**
   * Constructs an NdsIteratorControls instance using 9 parameters.
   *
   * @param scope     The scope of the search (SearchControls.OBJECT_SCOPE,
   *                  SearchControls.ONELEVEL_SCOPE, SearchControls.SUBTREE_SCOPE).
   * @param maxCount  The maximum number of entries to return. If 0, return
   *                  all entries that satisfy the filter.
   * @param timeout   The number of seconds to wait before returning.
   *                  If 0, wait indefinitely.
   * @param attrs     The identifiers of the attributes to return along
   *                  with the entry. If NULL, return all attributes.
   *                  If empty, return no attributes.
   * @param retobj    A boolean set to TRUE if the object bound to the name
   *                  of the entry is to be returned; otherwise, set to FALSE
   *                  if no object is to be returned.
   * @param deref     A boolean set to TRUE if de-reference of links is to
   *                  be done during the search; otherwise, set to FALSE.
   * @param indexSelect An optional string selecting the index to be used
   *                    in creating the iterator. If set to NULL or empty,
   *                    the index is selected by the server based on the
   *                    search filter. Applies only to NDS v8 iterators.
   * @param sortKey   List of attributes specifying the sort keys to use
   *                  when sorting the objects. Up to three attributes may
   *                  be specified. Applies only to non-NDS v8 iterators.
   * @param scalability  One of the following variable values:
   *                     DS_ITR_REQUIRE_SCALABLE - Return error if unable
   *                     to connect to a NDS v8 server.
   *                     DS_ITR_PREFER_SCALABLE - First tries to connect to
   *                     a NDS v8 server. If unable to do so, the iterator
   *                     functionality will be emulated by the client
   *                     libraries.
   *                     DS_ITR_FORCE_EMULATION - Forces emulation mode even
   *                     if a connection to a NDS v8 server is possible.
   */
   public NdsIteratorControls (
         int scope,
         long maxCount, 
         int timeout, 
         String[] attrs,
         boolean retobj, 
         boolean deref,
         String[] indexSelect,
         String[] sortKey,
         int scalability)
   {
      super (scope, maxCount, timeout * 1000, attrs, retobj, deref);

      this.indexSelect = indexSelect;
      this.sortKey = sortKey;
      this.scalability = scalability;
   }

  /**
   * Returns the sort key array of the NdsIteratorControls instance.
   *
   * @return The sort key array of this NdsIteratorControls instance.
   */
   public String[] getSortKey ()
   {
      return (sortKey);
   }

  /**
   * Sets the sort key array of the NdsIteratorControls instance.
   *
   * @param sortKey The sort key array of this NdsIteratorControls
   *              instance.
   */
   public void setSortKey (
         String[] sortKey)
   {
      this.sortKey = sortKey;
   }

  /**
   * Returns the array of selected indexes for this
   * NdsIteratorControls instance.
   *
   * @return The array of selected indexes.
   *
   * @see #setIndexSelect
   */
   public String[] getIndexSelect ()
   {
      return (indexSelect);
   }

  /**
   * Sets the indexes for this NdsIteratorControls instance
   * based on the passed in array of indexes.
   *
   * @param indexSelect The array of indexes to use in setting the
   *                    NdsIteratorControls instance indexes.
   */
   public void setIndexSelect (
         String[] indexSelect)
   {
      this.indexSelect = indexSelect;
   }

  /**
   * Returns the scalability status for this NdsIteratorControls
   * instance.
   *
   * @return The scalability status as an int.
   *
   * @see #setScalability
   */
   public int getScalability ()
   {
      return (scalability);
   }

  /**
   * Sets the scalability status for this NdsIteratorControls instance.
   *
   * <p>The scalability status is set to one of the following values:
   * <ul><li>DS_ITR_REQUIRE_SCALABLE - Return an error if unable to
   * connect to a NDS v8 server.
   * <li>DS_ITR_PREFER_SCALABLE - First tries to connect to a NDS v8 server.
   * If unable to do so, the iterator functionality will be emulated by
   * the client libraries.
   * <li>DS_ITR_FORCE_EMULATION - Forces emulation mode even if a connection
   * to a NDS v8 server is possible.</ul></p>
   *
   * @param scalability The scalability status as an int.
   */
   public void setScalability (int scalability)
   {
      this.scalability = scalability;
   }

} /* NdsIteratorControls */


