
/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/nds/NdsObject.java $
  $Revision: 18 $
  $Modtime: 1/28/00 12:45p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.util.Date;

import javax.naming.Referenceable;
import javax.naming.directory.DirContext;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;

import com.novell.service.jncp.NSIException;

import com.novell.service.nds.naming.Partitionable;


/**
 * Provides methods for operations on NDS Objects.
 *
 * <p>The NdsObject interface provides a window into functionality
 * not provided by JNDI. It defines methods dealing with
 * rights as well as methods related to operational attributes.
 */
public interface NdsObject
   extends DirContext, NdsIteratorFactory, Referenceable, Partitionable
{
   /**
    * Returns the Distinguished Name of the NDS Object.
    *
    * <p>The largest supported Distinguished Name is 256 Unicode
    * characters, not including the terminating NULL character.
    * Distinguished Names are not case sensitive, even if one of
    * the naming attributes is case-sensitive.
    *
    * @return A String containing the Distinguished Name.
    *
    * @exception NSIException Standard NJCL exception for working on
    *                         top of XPlat libraries.
    *                         
    */
   public String getDistinguishedName ()
      throws NSIException;

   /**
    * Returns the DSI Entry Flag ID for the object. These flags
    * return information about an entry's state. The 12 flags
    * are defined within this NDK under NDS Libraries for C
    * documentation (NDS > NDS:Values > DSI ENTRY FLAGS Values).
    *
    * @return The object's DSI Entry Flag ID as an int.
    *
    * @exception NSIException Standard NJCL exception for working on
    *                         top of XPlat libraries.
    */
   public int getObjectFlags ()
      throws NSIException;

   /**
    * Returns a count of the subordinates.
    *
    * @return A count of the subordinates as an int.
    *
    * @exception NSIException Standard NJCL exception for working on
    *                         top of XPlat libraries.
    */
   public int getSubordinateCount ()
      throws NSIException;

  /**
   * Returns the object modification time as a Date type.
   *
   * @return The Date object as the modification time.
   *
   * @exception NSIException Standard NJCL exception for working on
   *                         top of XPlat libraries.
   */
   public Date getModificationTime ()
      throws NSIException;

  /**
   * Returns the base class (immediate parent class) of the object.
   *
   * @return A String containing the base class of the object.
   *
   * @exception NSIException Standard NJCL exception for working on
   *                         top of XPlat libraries.
   */
   public String getBaseClass ()
      throws NSIException;

   /**
    * Returns the name of the NDS tree where the object
    * resides.
    *
    * @return A String containing the name of the NDS tree in which
    *         the object resides.
    *
    * @exception NSIException Standard NJCL exception for working on
    *                         top of XPlat libraries.
    */
    public String getTreeName ()
      throws NSIException;

  /**
   * Returns the effective rights of the specified trustee object.
   * Determines the rights of trustee (subject) for the current
   * NDS object.
   *
   * @param trustee NDS Object for which access privileges will be
   *                provided.
   *
   * @return The NdsObjectRights object containing the effective
   *         rights of the specified trustee object.
   *
   * @exception NSIException Standard NJCL exception for working on
   *                         top of XPlat libraries.
   * @see NdsObjectRights
   */
   public NdsObjectRights getObjectEffectiveRights (
         NdsObject trustee)
      throws NSIException;

  /**
   * Returns the effective rights of the specified trustee (subject)
   * String object. The String represents the current NDS object for
   * which access rights will be provided.
   *
   * <p>The trustee can be one of the following:
   * <ul><li>[Public]
   * <li>[Root]
   * <li>[Creator]
   * <li>[Self]
   * <li>DN of any NDS Object
   * </ul></p>
   *
   * @param trustee NDS String object for which access privileges
   *                will be provided.
   *
   * @return The NdsObjectRights object containing the effective
   *         rights of the specified trustee String object.
   *
   * @exception  NSIException Standard NJCL exception for working on
   *                         top of XPlat libraries.
   * @see NdsObjectRights
   */
   public NdsObjectRights getObjectEffectiveRights (
         String trustee)
      throws NSIException;


  /**
   * Returns the effective rights of the specified trustee NDS object.
   * Determines the rights of the trustee (subject) for the current
   * NDS object.
   *
   * @param trustee NDS object for which access privileges will be
   *                provided.
   *
   * @return The NdsAttributeRights object containing the effective
   *         rights of the specified trustee NDS object.
   *
   * @exception  NSIException Standard NJCL exception for working on
   *                         top of XPlat libraries.
   */
   public NdsAttributeRights getAttributeEffectiveRights (
         NdsObject trustee)
      throws NSIException;

  /**
   * Returns the effective rights of the specified trustee String
   * object. Determines the rights of the trustee (subject) for
   * the current NDS object.
   *
   * <p>The trustee can be one of the following:
   * <ul><li>[Public]
   * <li>[Root]
   * <li>[Creator]
   * <li>[Self]
   * <li>DN of any NDS Object</ul></p>
   * 
   * @param trustee Name for which access privileges will be
   *                provided.
   *
   * @return The NdsObjectRights object containing the effective
   *         rights of the specified trustee String object.
   *
   * @exception  NSIException Standard NJCL exception for working on
   *                         top of XPlat libraries.
   */
   public NdsAttributeRights getAttributeEffectiveRights (
         String trustee)
      throws NSIException;

  /**
   * Returns the effective rights of the specified trustee NDS object
   * for the specified attribute ID.
   *
   * @param trustee  NDS object for which access privileges will be
   *                 provided.
   * @param attrName Specific attribute name.
   *                 
   * @return The NdsAttributeRights object containing the effective
   *         rights of the specified trustee NDS object and attribute.
   *         ID
   * @exception  NSIException Standard NJCL exception for working on
   *                         top of XPlat libraries.
   */
   public NdsAttributeRights getAttributeEffectiveRights (
         NdsObject trustee,
         String attrName)
      throws NSIException;

  /**
   * Returns the effective rights of the specified trustee String
   * object for the attribute ID. Determines the rights of the
   * trustee (subject) for the specified attribute ID.
   *
   * <p>The trustee can be one of the following:
   * <ul><li>[Public]
   * <li>[Root]
   * <li>[Creator]
   * <li>[Self]
   * <li>DN of any NDS Object</ul></p>
   * 
   * @param trustee Name for which access privileges will be
   *                provided.
   *
   * @param attrName Specific attribute name.
   *
   * @return The NdsAttributeRights object containing the effective
   *         rights of the specified trustee and attribute name.
   *
   * @exception NSIException Standard NJCL exception for working on
   *                         top of XPlat libraries.
   */
   public NdsAttributeRights getAttributeEffectiveRights (
         String trustee,
         String attrName)
      throws NSIException;

  /**
   * Returns the effective rights of the specified trustee NDS object
   * and array of attribute IDs.
   *
   * @param trustee NDS object for which access privileges will be
   *                provided.
   * @param attrIds Array of attribute IDs for which to return
   *                  effective rights.
   *
   * @return The NamingEnumeration object containing the effective
   *         rights of the specified trustee NDS object and array
   *         of attribute IDs.
   *
   * @exception NSIException Standard NJCL exception for working on
   *                         top of XPlat libraries.
   */
   public NamingEnumeration getAttributesEffectiveRights (
         NdsObject trustee,
         String[] attrIds)
      throws NSIException;

  /**
   * Returns the effective rights of the specified trustee String
   * object and array of attribute IDs. Determines the rights of
   * the trustee (subject) for the specified array of attribute IDs.
   *
   * <p>The trustee can be one of the following:
   * <ul><li>[Public]
   * <li>[Root]
   * <li>[Creator]
   * <li>[Self]
   * <li>DN of any NDS Object</ul></p>
   * 
   * @param trustee Name for which access privileges will be
   *                provided.
   *
   * @param attrIds Array of attribute IDs for which to return
   *                effective rights.
   *
   * @return The NamingEnumeration object containing the effective
   *         rights of the specified trustee String and String array
   *         of attribute IDs.
   *
   * @exception NSIException Standard NJCL exception for working on
   *                         top of XPlat libraries.
   */
   public NamingEnumeration getAttributesEffectiveRights (
         String trustee,
         String[] attrIds)
      throws NSIException;

   /**
    * Returns a NamingEnumeration of NameClassPair objects that may be
    * container objects only, or all objects subordinate to the specified
    * name.
    *
    * @param name Container name for which subordinate objects will be
    *             returned.
    *
    * @param containers If true, specifies that all containers subordinate to
    *                   name will be returned. If false, specifies that all
    *                   objects subordinate to name will be returned.
    *
    * @return A NamingEnumeration of NameClassPair objects.
    *
    */
   public NamingEnumeration list (
         String name,
         boolean containers)
      throws NamingException;

   /**
    * Returns a NamingEnumeration of NameClassPair objects that satisfy
    * the specified filter criteria in the className and subordinateName
    * parameters.
    *
    * @param name Container name for which NameClassPair objects will be
    *             returned for matching subordinates.
    *
    * @param className Specifies a class name to be used as a filter (can
    *                  contain wildcards).
    *
    * @param subordinateName Specifies an object name to be used as a filter
    *                        (can contain wildcards).
    *
    * @return A NamingEnumeration of NameClassPair objects matching the
    *         criteria specified in the className, subordinateName and
    *         containers parameters.
    *
    */
   public NamingEnumeration list (
         String name,
         String className,
         String subordinateName)
      throws NamingException;

   /**
    * Returns a NamingEnumeration of Binding objects that may be container
    * objects only, or all objects subordinate to the specified name.
    *
    * @param name Container name for which subordinate objects will be
    *             returned.
    *
    * @param containers If true, specifies that all containers subordinate to
    *                   name will be returned. If false, specifies that all
    *                   objects subordinate to name will be returned.
    *
    * @return A NamingEnumeration of Binding objects.
    *
    */
   public NamingEnumeration listBindings (
         String name,
         boolean containers)
      throws NamingException;

   /**
    * Returns a NamingEnumeration of Binding objects that satisfy
    * the specified filter criteria in the className and subordinateName
    * parameters.
    *
    * @param name Container name for which Binding objects will be
    *             returned for matching subordinates.
    *
    * @param className Specifies a class name to be used as a filter (can
    *                  contain wildcards).
    *
    * @param subordinateName Specifies an object name to be used as a filter
    *                        (can contain wildcards).
    *
    * @return A NamingEnumeration of Binding objects matching the
    *         criteria specified in the className, subordinateName and
    *         containers parameters.
    *
    */
   public NamingEnumeration listBindings (
         String name,
         String className,
         String subordinateName)
      throws NamingException;

} /* NdsObject */
