
/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/nds/NdsOctetString.java $
  $Revision: 8 $
  $Modtime: 10/15/99 3:26p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import java.util.Enumeration;
import java.util.Vector;

import com.novell.service.schema.SchemaSequence;


/**
 * Provides access to values of the OctetString attribute.
 * It is used for attributes whose values are byte strings that
 * are not interpreted by NDS.
 *
 * <p>For two octet strings to match, they must be the same
 * length and the corresponding bit sequence (octets) must be
 * identical. When comparing two octet strings, the first pair
 * of octets that do not match are used to determine the order
 * of the strings. Octet strings are not Unicode strings.
 * 
 * <p>Matching Rules: Equality and Ordering</p>
 * 
 * @see NdsSyntaxId
 */
public class NdsOctetString
   implements NdsAttributeValue, SchemaSequence, Cloneable, Serializable
{
   private static final int matchingRules = equality|ordering;

  /**
   * @internal
   */
   protected byte[] value;

   private String name;


  /**
   * Constructs an NdsOctetString object based on the specified
   * value variable.
   * 
   * @param value  The octet string to be stored as a byte array.
   */
   public NdsOctetString (
         byte[] value)
   {
      this ("", value);
   }

  /**
   * @internal
   */
   protected NdsOctetString ()
   {

   }

   private NdsOctetString (
         String name,
         byte[] value)
   {
      this.value = (byte[]) value.clone();
      this.name = name;
   }

  /**
   * @internal
   */
   protected NdsOctetString (
         NdsOctetString anOctetString)
   {
      this.value = anOctetString.value;
      this.name = anOctetString.name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically. For the Octet String syntax, the
   * compareTo method compares this NDS Object with a
   * reference NDS Object contained in the anObject parameter.
   * The comparison is done according to the Ordering matching
   * rule for this syntax.
   * 
   * @param anObject The reference NDS Object with which
   *                 to compare.
   *
   * @return An int set to 0 if the reference Object is equal
   *         to this Object; an int <0 if the reference Object
   *         is less than this Object; an int >0 if the
   *         reference Object is greater than this Object.
   *
   * @exception IllegalArgumentException The passed-in object is of
   *            a different type than this NDS Object.
   */
   public int compareTo (
         Object anObject)
      throws Exception
   {
      if (anObject instanceof NdsOctetString)
      {
         int cmpLength = ((NdsOctetString)anObject).getLength ();

         byte[] cmpValue = ((NdsOctetString)anObject).getOctetString ();
         for (int i = 0; i < this.value.length; i++)
         {
            if (cmpValue[i] != this.value[i])
            {
               return (this.value[i] - cmpValue[i]);
            }
         }
         return (0);
      }
      throw (new Exception ());
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals (
         Object anObject)
   {
      if (anObject instanceof NdsOctetString)
      {
         if (((NdsOctetString)anObject).getLength () == this.value.length)
         {
            try
            {
               return (0 == this.compareTo (anObject) ? true : false);
            }
            catch (Exception e)
            {
               // Ignore and return false
            }
         }
      }
      return (false);
   }

  /**
   * @internal
   */
   public String getName ()
   {
      return (name);
   }

  /**
   * @internal
   */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.OCTET_STRING);
   }


   // ******************** SchemaSequence Interface ********************

   /**
    * Returns the number of elements in this sequence.
    *
    * @return The number of elements in this sequence as an int.
    */
   public int count ()
   {
      return (value.length);
   }

   /**
    * Returns an enumeration of this sequence's values.
    *
    * @return An enumeration of value elements.
    */
   public Enumeration getValues ()
   {
      Vector elements = new Vector ();

      for (int i = 0; i < value.length; i++)
      {
         elements.addElement (new Byte (value[i]));
      }
      return (elements.elements ());
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Compares two Objects using the approximate matching rule.
   * 
   * The approximate method does not apply to the Octet String syntax
   * because this syntax does not support the approximate equals
   * matching rule.
   */
   public boolean approximate (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Returns the int that represents this NDS syntax ID.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.OCTET_STRING_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules. The OctetString syntax supports both
   * the Equality and Ordering matching rules.
   *
   * @param matchingRules The set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
         return (true);
      else
         return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone ()
   {
      try
      {
         NdsOctetString string = (NdsOctetString) super.clone ();

         string.value = (byte[]) this.value.clone ();

         string.name = this.name;

         return (string);
      }
      catch (CloneNotSupportedException e)
      {
         // this shouldn't happen, since we are Cloneable
         throw (new InternalError ());
      }

   } /* clone () */

  /**
   * Generates a string representation of the object. It
   * returns a string that textually represents the object.
   * The result should be a concise but informative
   * representation that is easy to read.
   *
   * @return The String representation of the object.
   */
   public String toString ()
   {
      return (new String (value));
   }


   // ******************** NdsCaseIgnoreList Class ********************

  /**
   * Returns the length of the OctetString.
   *
   * @return The length of the OctetString as an int.
   */
   public int getLength ()
   {
      return (value.length);
   }

  /**
   * Returns the byte array of values stored in the NDS
   * attribute.
   *
   * @return The byte array of values.
   */
   public byte[] getOctetString ()
   {
      byte[] value = new byte[this.value.length];

      System.arraycopy (this.value, 0, value, 0, value.length);
      return (value);
   }

} /* NdsOctetString */

