
/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/nds/NdsPath.java $
  $Revision: 8 $
  $Modtime: 10/15/99 3:26p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import java.util.Enumeration;
import java.util.Vector;

import com.novell.service.schema.SchemaValue;
import com.novell.service.schema.SchemaComposite;


/**
 * Provides access to values of the Path attribute. It is used
 * for attributes that represent a file system path.
 *
 * <p>Two paths match for equality when their lengths and corresponding
 * characters, including case, are identical. In matching attributes
 * that conform to this syntax, NDS omits those spaces that are not
 * significant (leading spaces, trailing spaces, and multiple
 * consecutive internal spaces). NDS stores insignificant spaces
 * with the attribute value.
 * 
 * <p>Matching Rules: Equality</p>
 * 
 * @see NdsSyntaxId
 */
public class NdsPath
   implements NdsAttributeValue, SchemaComposite, Cloneable, Serializable
{
   private static final int matchingRules = equality;

   private static String[] compositeIds;

   static
   {
      compositeIds = new String[3];
      compositeIds[0] = "nameSpaceType";
      compositeIds[1] = "volumeName";
      compositeIds[2] = "path";
   }

  /**
   * @internal
   */
   protected int nameSpaceType;

  /**
   * @internal
   */
   protected String volumeName;

  /**
   * @internal
   */
   protected String path;

   private String name;


  /**
   * Constructs an NdsPath object based on the specified
   * nameSpaceType, volumeName and path parameters.
   * 
   * @param nameSpaceType The type of the namespace stored as a long.
   * @param volumeName    The name of the volume object stored as a String.
   * @param path          The file system path stored as a String.
   */
   public NdsPath (
         long nameSpaceType,
         String volumeName,
         String path)
   {
      this ("", nameSpaceType, volumeName, path);
   }

  /**
   * @internal
   */
   protected NdsPath ()
   {
      this.nameSpaceType = -1;
   }

   private NdsPath (
         String name,
         long nameSpaceType,
         String volumeName,
         String path)
   {
      this.nameSpaceType = (int) (nameSpaceType & INT_MASK);
      this.volumeName = volumeName;
      this.path = path;
      this.name = name;
   }

  /**
   * @internal
   */
   protected NdsPath (
         NdsPath aPath)
   {
      this.nameSpaceType = aPath.nameSpaceType;
      this.volumeName = aPath.volumeName;
      this.path = aPath.path;
      this.name = aPath.name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically.
   *
   * The compareTo method does not apply to the NdsPath syntax
   * because this syntax does not support the ordering and substrings
   * matching rules.
   */
   public int compareTo (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals (
         Object anObject)
   {
      if ((anObject != null) && (anObject instanceof NdsPath))
      {
         NdsPath anotherPath = (NdsPath) anObject;

         if (nameSpaceType != anotherPath.nameSpaceType)
         {
            return (false);
         }
         if (false == volumeName.equals (anotherPath.volumeName))
         {
            return (false);
         }
         if (false == path.equals (anotherPath.path))
         {
            return (false);
         }
         return (true);
      }
      return (false);
   }

  /**
   * @internal
   */
   public String getName ()
   {
      return (name);
   }

  /**
   * @internal
   */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.PATH);
   }


   // ******************** SchemaComposite Interface ********************

   /**
    * Returns the number of elements in this sequence.
    *
    * @return The number of elements in this sequence as an int.
    */
   public int count ()
   {
      return (compositeIds.length);
   }

   /**
   * Returns a SchemaValue object based on the given passed-in
   * string containing the element value. The SchemaValue object
   * is a Java language data type.
   *
   * @param compositeId The passed-in string containing the element
   *                    value.
   *
   * @return The SchemaValue object containing the nameSpaceType,
   *         volumeName and path, or NULL.
   */
   public SchemaValue getValue (
         String compositeId)
   {
      if (compositeId.equalsIgnoreCase (compositeIds[0]))
      {
         // getNameSpaceType ();
         return (new NdsInteger (compositeIds[0], nameSpaceType));
      }
      else if (compositeId.equalsIgnoreCase (compositeIds[1]))
      {
         // getVolumeName ();
         return (new NdsString (compositeIds[1], volumeName));
      }
      else if (compositeId.equalsIgnoreCase (compositeIds[2]))
      {
         // getPath ();
         return (new NdsString (compositeIds[2], path));
      }
      return (null);
   }

  /**
   * Returns an enumeration of element IDs in this composite.
   *
   * @return An enumeration of element IDs in this composite.
   */
   public Enumeration getValueIds ()
   {
      Vector vector = new Vector ();

      for (int i = 0; i < compositeIds.length; i++)
         vector.addElement (compositeIds[i]);
      return (vector.elements ());
   }

   /**
    * Returns an enumeration of this sequence's values.
    *
    * @return An enumeration of this sequence's values.
    */
   public Enumeration getValues ()
   {
      Vector values = new Vector ();

      values.addElement (new NdsInteger (compositeIds[0], nameSpaceType));
      values.addElement (new NdsString (compositeIds[1], volumeName));
      values.addElement (new NdsString (compositeIds[2], path));
      return (values.elements ());
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Compares two Objects using the approximate matching rule.
   * 
   * The approximate method does not apply to the NdsPath syntax
   * because this syntax does not support the approximate equals
   * matching rule.
   */
   public boolean approximate (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Returns the int that represents this NDS syntax ID.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.PATH_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules. The NdsPath syntax supports only the
   * equality matching rule.
   *
   * @param matchingRules The set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
      {
         return (true);
      }
      return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone ()
   {
      try
      {
         NdsPath path = (NdsPath) super.clone ();

         path.nameSpaceType = this.nameSpaceType;
         path.volumeName = this.volumeName;
         path.path = this.path;

         path.name = this.name;

         return (path);
      }
      catch (CloneNotSupportedException e)
      {
         // this shouldn't happen, since we are Cloneable
         throw (new InternalError ());
      }

   } /* clone () */

  /**
   * Generates a string representation of the object. It
   * returns a string that textually represents the object.
   * The result should be a concise but informative
   * representation that is easy to read.
   *
   * @return The String representation of the object.
   */
   public String toString ()
   {
      return (
         String.valueOf (nameSpaceType) + ";" +
         volumeName + ";" +
         path);
   }


   // ******************** NdsPath Class ********************

  /**
   * Returns the long value of the namespace type stored in
   * the NDS attribute.
   *
   * @return The namespace type stored as a long.
   */
   public long getNameSpaceType ()
   {
      return (nameSpaceType & INT_MASK);
   }

  /**
   * Returns the volume name stored as a String in
   * the NDS attribute.
   *
   * @return The volume name stored as a string.
   */
   public String getVolumeName ()
   {
      return (volumeName);
   }

  /**
   * Returns the file system path stored as a String in
   * the NDS attribute.
   *
   * @return The file system path stored as a string.
   */
   public String getPath ()
   {
      return (path);
   }

} /* NdsPath */


