
/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/nds/NdsPrintableString.java $
  $Revision: 7 $
  $Modtime: 10/15/99 3:36p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;


/**
 * Provides access to values of the Printable String attribute.
 * It is used for attributes whose values are printable strings.
 * Printable strings are defined in CCITT X.208.
 *
 * <p>Two printable strings match for equality when they are the
 * same length and their corresponding characters are identical.
 * Case (upper or lower) is significant when comparing printable
 * strings. In matching attributes that conform to this syntax,
 * NDS omits those spaces that are not significant (leading spaces,
 * trailing spaces, and multiple consecutive internal spaces). NDS
 * stores insignificant spaces with the attribute value.
 * 
 * <p>Matching Rules: Equality and Substrings</p>
 * 
 * @see NdsSyntaxId
 */
public class NdsPrintableString
   extends NdsString
   implements NdsAttributeValue, Cloneable, Serializable
{
   private static final int matchingRules = equality|substrings;

  /**
   * Constructs an NdsPrintableString object based on the specified
   * value parameter.
   * 
   * @param value The printable string to be stored as a String value.
   */
   public NdsPrintableString (
         String value)
   {
      this.value = value;
   }

   /**
    * @internal
    */
   protected NdsPrintableString ()
   {

   }

   private NdsPrintableString (
         String name,
         String value)
   {
      super (name, value);
   }

   /**
    * @internal
    */
   protected NdsPrintableString (
         NdsPrintableString aPrintableString)
   {
      this.value = aPrintableString.value;
      this.name = aPrintableString.name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically. For the Printable String syntax, the
   * compareTo method compares this NDS Object (string) with a
   * reference NDS Object (string) contained in the anObject
   * parameter. The comparison is done according to the
   * substrings matching rule for this syntax.
   * 
   * @param anObject The reference NDS Object (string) with which
   *                 to compare.
   *
   * @return An int set to 0 if the reference Object (string) is equal
   *         to this Object (string); an int <0 if the reference
   *         Object (string) is less than this Object (string); an
   *         int >0 if the reference Object (string) is greater than
   *         this Object (string).
   *
   * @exception IllegalArgumentException The passed-in object is of
   *            a different type than this NDS Object.
   */
   public int compareTo (
         Object anObject)
      throws Exception
   {
      NdsPrintableString anotherString;

      anotherString = (NdsPrintableString) anObject;
      return (value.compareTo (anotherString.value));
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals (
         Object anObject)
   {
      if ((anObject != null) && (anObject instanceof NdsPrintableString))
      {
         NdsPrintableString anotherString = (NdsPrintableString) anObject;

         if (false == value.equals (anotherString.value))
         {
            return (false);
         }
         return (true);
      }
      return (false);
   }

   /**
    * @internal
    */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.PRINTABLE_STRING);
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Compares two Objects using the approximate matching rule.
   * 
   * The approximate method does not apply to the Printable String
   * syntax because this syntax does not support the approximate
   * equals matching rule.
   */
   public boolean approximate (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Returns the int that represents this NDS syntax ID.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.PRINTABLE_STRING_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules. The Printable String syntax supports
   * both the equality and substrings matching rules.
   *
   * @param matchingRules The set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
      {
         return (true);
      }
      return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * Them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone ()
   {
      NdsPrintableString string = (NdsPrintableString) super.clone ();

      return (string);

   } /* clone () */


   // ******************** NdsPrintableString Class ********************

  /**
   * Returns the String value of a printable string stored in
   * the NDS attribute.
   *
   * @return The String value of a printable string.
   */
   public String getPrintableString ()
   {
      return (value);
   }

} /* NdsPrintableString */


