/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/qms/QMSJob.java $
  $Revision: 11 $
  $Modtime: 1/28/00 12:46p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.qms;

import java.io.*;
import java.util.*;

import com.novell.service.jncp.NSIException;

/**
 * Represents a job in the queue that can represent almost any
 * application task. It is up to the developer to define the job
 * service protocol.
 *
 * <p>The attribute IDs of a QueueJob include the following:
 * <ul><li>ClientID
 * <li>ClientStation
 * <li>ClientTask
 * <li>ControlFlags
 * <li>Description
 * <li>EntryTime
 * <li>ID
 * <li>Position
 * <li>TargetExecutionTime
 * <li>TargetServerID
 * <li>Type</ul>
 *
 * <p>The following is an example of how a queue job is created and
 * submitted:
 * <pre><code>
 *    import java.util.Hashtable;
 *    import javax.naming.*;
 *    import javax.naming.directory.*;
 *    import com.novell.utility.naming.Environment;
 *    import com.novell.service.qms.*;
 *    ...
 *    public void createAJob (String serverName, String queueName)
 *    {
 *       DirContext dirCtx;
 *       Hashtable hash;
 *       QMSQueue queue;
 *       QMSJob job;
 *       QMSOutputStream out;
 *
 *       try
 *       {
 *          hash = new Hashtable ();
 *          hash.put (
 *                Context.INITIAL_CONTEXT_FACTORY,
 *                Environment.QMS_INITIAL_CONTEXT_FACTORY);
 *          hash.put (Context.PROVIDER_URL, serverName);
 *          hash.put (QMSEnvironment.QUEUE_NAME, queueName);
 *          hash.put (QMSEnvironment.QUEUE_TYPE, queueName);
 *          dirCtx = new InitialDirContext (hash);
 *          queue = (QMSQueue) dirCtx.lookup ("");
 *          job = queue.createJob ();
 *          job.setDescription ("Test job");
 *          out = job.submit ();
 *          out.close ();        // We don't put anything in the file
 *       } catch (Exception e)
 *       {
 *         System.out.println ("Exception occurred:" + e);
 *       }
 *    }
 * </code></pre>
 * </p>
 *
 * @see QMSQueue
 * @see QMSOutputStream
 */

public interface QMSJob
{
   /**
    * Queue control flag indicating that a job can be released for
    * servicing automatically, even if the connection is broken. If
    * not set, a job is automatically removed if the connection is
    * broken. This flag is part of the QueueJob ControlFlags attribute.
    *
    * <p>(QF_AUTO_START = 0x08)
    * </p>
    * @see #setControlFlags
    * @see #getControlFlags
    */
   public static final int QF_AUTO_START     = 0x08;

   /**
    * Queue control flag indicating that a job can be restarted in the
    * event of a service failure. The job will be attempted again when
    * service to the queue resumes. Clearing this bit allows the job
    * entry to be removed from the queue if it is not serviced
    * successfully. This flag is part of the QueueJob ControlFlags
    * attribute.
    *
    * <p>(QF_ENTRY_RESTART = 0x10)
    * </p>
    * @see #setControlFlags
    * @see #getControlFlags
    */
   public static final int QF_ENTRY_RESTART  = 0x10;

   /**
    * Queue control flag that is set by the queue and remains set as
    * long as the client has not released the queue job entry for
    * servicing. When the client has released the job, QMS clears the
    * bit. This flag is part of the QueueJob ControlFlags attribute.
    *
    * <p>(QF_ENTRY_OPEN = 0x20)
    * </p>
    * @see #setControlFlags
    * @see #getControlFlags
    */
   public static final int QF_ENTRY_OPEN     = 0x20;

   /**
    * Queue control flag indicating that a job is on hold. The job will
    * continue to advance in the queue, but will not be serviced until
    * this bit is cleared. This flag can be set by the object that
    * placed the job in the queue. This flag is part of the QueueJob
    * ControlFlags attribute.
    *
    * <p>(QF_USER_HOLD = 0x40)
    * </p>
    * @see #setControlFlags
    * @see #getControlFlags
    */
   public static final int QF_USER_HOLD      = 0x40;

   /**
    * Queue control flag indicating that a job is on hold. The job will
    * continue to advance in the queue, but will not be serviced until
    * this bit is cleared.  This flag can be set only by an object that
    * has queue operator privileges. This flag is part of the QueueJob
    * ControlFlags attribute.
    *
    * <p>(QF_OPERATOR_HOLD = 0x80)
    * </p>
    * @see #setControlFlags
    * @see #getControlFlags
    */
   public static final int QF_OPERATOR_HOLD  = 0x80;

   /**
    * Queue control flag indicating that a job is already open.
    *
    * <p>(JOB_ALREADY_OPEN = 0x882B)
    * </p>
    * @see #setControlFlags
    * @see #getControlFlags
    */
   public static final int JOB_ALREADY_OPEN  = 0x882B;

   // QMSJob methods =========================================================

   /**
    * Submits a job into the queue and gets an OutputStream back. It
    * does not support re-submitting an already submitted job.
    *
    * @return An output stream on which to write data.
    *
    * @exception IOException  An I/O error occurred.
    * @exception NSIException A network exception occurred.
    *
    * @see #cancel
    */
   public QMSOutputStream submit()
      throws IOException, NSIException;

   /**
    * Cancels the job, removes it from the queue, and aborts the
    * output stream if it is open.
    *
    * @exception IOException  An I/O error occurred.
    * @exception NSIException A network exception occurred.
    *
    * @see #submit
    */
   public void cancel()
      throws IOException, NSIException;

   // QMSJob public accessor methods =========================================

   /**
    * Returns the attribute ID of the client station that created the
    * queue job.
    *
    * @return  The originating client station attribute ID.
    */
   public int getClientStation ();

   /**
    * Returns the attribute ID of the client task of the originating
    * process that was active when the queue job was created.
    *
    * @return The originating client task attribute ID.
    */
   public int getClientTask ();

   /**
    * Returns the attribute ID of the client that created the queue job.
    *
    * @return  The originating client attribute ID.
    */
   public int getClientID ();

   /**
    * Returns the attribute ID of the target server.
    *
    * <p>A value of -1 (0xFFFFFFFF) indicates that any server can
    * service the job.
    * </p>
    *
    * @return  The target server attribute ID.
    *
    * @see #setTargetServerID
    */
   public int getTargetServerID ();

   /**
    * Returns the attribute ID of the target execution time.
    *
    * <p>If the target time is equal to the entry time then the job
    * may be serviced immediately.
    * </p>
    *
    * @return  The attribute ID of the target execution time.
    *
    * @see #setTargetExecutionTime
    */
   public Date getTargetExecutionTime ();

   /**
    * Returns the attribute ID of the job entry time.
    *
    * @return  The attribute ID of the job entry time.
    */
   public Date getEntryTime ();

   /**
    * Returns the attribute ID of the queue job.
    *
    * <p>The queue job ID is assigned by the queue when the job is
    * submitted.</p>
    *
    * @return  The attribute ID of the queue job.
    */
   public int getID ();

   /**
    * Returns the attribute ID of the queue job type.
    *
    * <p>The queue job type is defined by the job service protocol,
    * which is developer defined. If not used, the job type should
    * be set to 0. A -1 should not be used as a job type because
    * it is used by the job server to indicate that it will service any
    * job type.</p>
    *
    * @return  The attribute ID of the queue job type.
    *
    * @see #setType
    */
   public int getType ();

   /**
    * Returns the attribute ID of the job position in the queue.
    *
    * <p>The entry at the top of the queue is assigned a position of 1.
    * The next entry in the queue is position 2, and so forth. Position
    * 0 is returned if the job has not been submitted to the queue.
    * </p>
    *
    * @return  The attribute ID of the job's position in the queue.
    *
    * @exception NSIException  A network exception occurred.
    *
    * @see #setPosition
    */
   public int getPosition ()
      throws NSIException;

   /**
    * Returns the attribute ID of the job control flags.
    *
    * <p>The control flags indicate the current status of the job.
    * </p>
    *
    * @return  The attribute ID of the control flags for this job.
    *
    * @see #QF_AUTO_START
    * @see #QF_ENTRY_RESTART
    * @see #QF_ENTRY_OPEN
    * @see #QF_USER_HOLD
    * @see #QF_OPERATOR_HOLD
    * @see #setControlFlags
    */
   public int getControlFlags ();

   /**
    * Returns the queue job file name.
    *
    * <p>The queue associates a file with a queue job when it is
    * submitted. This file can be used for storing additional
    * information beyond that in the client record area for the job.
    *
    * @return The queue job file name.
    *
    * @see #submit
    */
   public byte[] getFileName ();

   /*
    * Returns the queue job file handle.
    *
    * <p>This call should not be used.
    * @see #submit
    * @see QMSOutputStream
    */
   //   public int getFileHandle ();

   /**
    * Returns the connection number of the server servicing this job.
    *
    * @return  The connection number of the server servicing this
    *          job.
    */
   public int getServicingServerStation ();
   // NOTE: This should ping the network and throw an NSIException

   /**
    * Returns the task number of the application servicing this job.
    *
    * @return  The task number of the application servicing this
    *          job.
    */
   public int getServicingServerTask ();
   // NOTE: This should ping the network and throw an NSIException

   /**
    * Returns the object ID of the server servicing this job.
    *
    * @return  The object ID of the server servicing this job.
    */
   public int getServicingServerID ();
   // NOTE: This should ping the network and throw an NSIException

   /**
    * Returns the attribute ID of the user defined description string
    * for this job.
    *
    * @return  The attribute ID of the user defined description String
    *          for this job.
    *
    * @see #setDescription
    */
   public String getDescription ();

   /**
    * Returns the user defined client record area for this job.
    *
    * <p>The format of this area is defined by the job service protocol.
    * It is limited to 152 bytes of data.</p>
    *
    * @return A Byte array containing the user defined client
    *         record area.
    *
    * @see #setClientRecordArea
    */
   public byte[] getClientRecordArea ();

   /**
    * Sets the attribute ID of the target server.
    *
    * <p>The targetServerID parameter is set to -1 if any server is
    * allowed to service the job. Otherwise, it is set to the object
    * ID of the job server.</p>
    *
    * @param targetServerID  The attribute ID of the target server.
    *
    * @exception NSIException  A network exception occurred.
    *
    * @see #getTargetServerID
    */
   public void setTargetServerID (int targetServerID)
      throws NSIException;

   /**
    * Sets the attribute ID of the target execution time.
    *
    * @param targetExecutionTime The date and time after which this
    *                            job can be serviced.
    *
    * @exception NSIException  A network exception occurred.
    *
    * @see #getTargetExecutionTime
    */
   public void setTargetExecutionTime (Date targetExecutionTime)
      throws NSIException;

   /**
    * Sets the attribute ID of the job type.
    *
    * <p>The job type can be used by the job service protocol to allow
    * multiple types of jobs to be serviced through a single queue. This
    * should be set to 0 if not used. A -1 should not be used because
    * that value is used by job servers to indicate that it will service
    * any job type.</p>
    *
    * @param type  The attribute ID of the job type.
    *
    * @exception NSIException  A network exception occurred.
    *
    * @see #getType
    */
   public void setType (int type)
      throws NSIException;

   /**
    * Sets the attribute ID of the position for the job within the queue.
    *
    * <p>The entry at the top of the queue is assigned a position of 1.
    * The next entry in the queue is position 2, and so forth.
    * Position 0 is returned if the job has not been submitted to the
    * queue.</p>
    *
    * @param newPosition  The desired position for this queue job.
    *
    * @exception NSIException  A network exception occurred.
    *
    * @see #getPosition
    */
   public void setPosition (int newPosition)
      throws NSIException;

   /**
    * Sets the attribute ID of the job control flags, which indicate the
    * current status of the job in the queue.
    *
    * @param controlFlags The attribute ID of the job control flags.
    *
    * @see #QF_AUTO_START
    * @see #QF_ENTRY_RESTART
    * @see #QF_USER_HOLD
    * @see #QF_OPERATOR_HOLD
    * @see #setControlFlags
    *
    * @exception NSIException  A network exception occurred.
    */
   public void setControlFlags (int controlFlags)
      throws NSIException;

   /**
    * Sets the user defined description string for this job.
    *
    * @param description The user defined description String for
    *                    this job.
    *
    * @exception NSIException  A network exception occurred.
    *
    * @see #getDescription
    */
   public void setDescription (String description)
      throws NSIException;

   /**
    * Sets the user defined client record area for this job.
    *
    * <p>The format of this area is defined by the job service protocol.
    * It is limited to 152 bytes of data.</p>
    *
    * @param clientRecordArea A byte array containing the user defined
    *                         client record area.
    *
    * @exception NSIException  A network exception occurred.
    *
    * @see #getClientRecordArea
    */
   public void setClientRecordArea (byte[] clientRecordArea)
      throws NSIException;

}