
/* **************************************************************************

  $Archive: /njcl/src/com/novell/utility/naming/directory/StaticAttributeValue.java $
  $Revision: 1 $
  $Modtime: 4/22/98 9:00a $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.utility.naming.directory;


import javax.naming.NamingException;
import javax.naming.directory.Attribute;

import com.novell.service.rfc1960.SearchStringComponent;


/**@internal
 * This interface is a moderator between static attribute value objects
 * and the JNDI attribute interface on DirContext.
 *
 * <p>A static attribute value can be any object that has an interface
 * to get at attribute information.  This interface is completely defined
 * by the object -- in other words, it is not restricted to the JNDI
 * attribute interface.  This allows an attribute value class
 * to be independant of JNDI, while implementing functionality (through
 * this interface) that allows it to be plugged in to JNDI.
 *
 * <p>This class provides a common interface for a DirContext to use static 
 * attribute value objects as values contained in a JNDI Attribute.  An
 * object that allows access to multiple attribute values (multi-valued
 * attribute) implements the 'hasMoreElements()' and 'nextElement()' methods.
 * to return all available values.  If the object is a single-valued
 * attribute, then the above methods are implemented to return one and only
 * one value.
 *
 * <p>There are some dynamic aspects of the static attribute values.  
 * examples of these are in the file system name space; Extended 
 * Attributes, Trustee lists, and volume restrictions.  All three of these
 * are dynamic lists of information that will be found in JNDI under a
 * single attribute ID.  This concept will be referred to as "dynamic
 * attribute values".  (This is not in opposition to the name
 * StaticAttributeValue).
 *
 * <p>This is an implementation level interface, never to be published
 * or used by a user application layer.
 * </p>
 */
public interface StaticAttributeValue
{
   /**
    * Builds and returns a JNDI Attribute from this static attribute
    * value.
    *
    * <p>On attribute values that support lazy evaluation, calling this
    * method causes this object's data state to be set.
    * </p>
    *
    * @return                    An Attribute object that reflects the state
    *                            of this object's member data.
    * @exception NamingException When an error occurs retrieving the data.
    */
   public Attribute buildAttribute() 
      throws NamingException;

   /**
    * Adds all values in the specified JNDI Attribute to this
    * object's values.
    *
    * <p>The addAttribute() method is implemented by static attribute
    * value classes that have dynamic attribute values.  All other classes
    * will throw an AttributeModificationException.
    * </p>
    *
    * @param      attr           The attribute containing the values
    *                            to be added.
    * @exception AttributeModificationException When the static attribute
    *                            value does not support dynamic attribute
    *                            values or when an error occurs when actually
    *                            adding the values as requested.
    * @exception NamingException When an error occurs in the process of
    *                            adding the values.
    */
   public void addAttribute(Attribute attr)
      throws NamingException;

   /**
    * Removes all values in the specified JNDI Attribute that are
    * equivalent with values of this object.
    *
    * <p>The delAttribute() method is implemented by static attribute value
    * classes that have dynamic attribute values.  All others classes
    * will throw an AttributeModificationException.
    * </p>
    *
    * @param      attr           The attribute to be removed.
    * @exception AttributeModificationException When the static attribute
    *                            value does not support dynamic attribute
    *                            values or when an error occurs when actually
    *                            removing the values as requested.
    * @exception NamingException When an error occurs in the process of
    *                            removing the values.
    */
   public void deleteAttribute(Attribute attr)
      throws NamingException;

   /**
    * Modifies the information in this object's values according to the
    * values in the specified attribute.
    *
    * <p>If this object has values that have more that one field, it is
    * possible for some of the fields to be read-only.  Any read-only
    * fields of the attribute value being passed in must match the current
    * state of this object or an AttributeModificationException will be
    * thrown.
    * </p>
    *
    * @param      attr           Attribute reflecting the desired state of 
    *                            the value.
    * @exception AttributModificationException When this object has a
    *                            read-only state that the user is trying to
    *                            modify or when an error occurs when
    *                            actually modifying the values.
    * @exception NamingException When an error occurs in the process of
    *                            modifying the values.
    */
   public void modifyAttribute(Attribute attr)
      throws NamingException;

   /**
    * Gets the attribute name.
    *
    * <p>Returns the Attribute name (attrId) of the implementation object.
    * This attribute ID is used when interfacing with JNDI.  Any JNDI
    * attribute that uses a static attribute value as a source for its
    * values must have the same attribute ID as returned from this method.
    * </p>
    *
    * @return                    The Attribute name (attrId).
    *
    * @see javax.naming.directory.DirContext#getAttributes
    */
   public String getID();

   /**
    * Performs the specified compare operation and returns the result.
    * 
    * <p>This method will compare the specified value object against this
    * attributes current value, in the manner specified in the
    * search string component, and return the result of this compare.
    * </p>
    *
    * @param      ssc            The search string component to use for the  
    *                            compare.
    * @return                    true if the operation compares true, false
    *                            otherwise.
    * @exception NamingException When an error occurs comparing the
    *                            specified values.
    */
   public boolean compare(SearchStringComponent ssc)
      throws NamingException;

   /**
    * Tests whether there is another attribute value available.
    * 
    * <p>This method works in conjuction with the nextElement() method to
    * provide the user with the attribute values as they ask for them.
    * Information is retrieved as neccesary (lazy evaluation), and this can
    * cause the wire to be hit when called.</p>
    *
    * @return                    true if there is an attribute value to
    *                            return, false if there are no more
    *                            attribute values to return.
    *
    * @see com.novell.service.file.naming.FSAttribute
    * @see #buildAttribute
    */
   public boolean hasMoreElements();

    
   /**
    * Returns the next available attribute value.
    * 
    * <p>This method works in conjuction with the hasMoreElements() method to
    * provide the user with the attribute values as they ask for them.
    * Information is retrieved as neccesary (lazy evaluation), and this can
    * cause the wire to be hit when called.</p>
    *
    * @return                    Object which is the next attribute value.
    * @exception NoSuchElementException When no attribute values are
    *                            available.
    *
    * @see com.novell.service.file.naming.FSAttribute
    * @see #buildAttribute
    */
   public Object nextElement();

   /**
    * Returns the static interface object.
    * 
    * <p>This method returns the appropriate object for the static interface
    * to this attribute value.
    * </p>
    *
    * @return                    Object which is the static interface.
    *
    */
   public Object getStaticInterface();
}
