/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/java/io/DataAccessable.java $
  $Revision: 5 $
  $Modtime: 1/18/00 8:57a $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.java.io;

import java.io.IOException;
import com.novell.java.io.spi.DataAccessor;

/**
 * Provides an object that may support input/output streams and random
 * access to data. 
 *
 * <p>The DataAccessable object may have container capabilities that
 * support subordinate access, or it may support access only to itself.
 * The various methods of this interface report if the device
 * supports access to itself or if the device supports subordinate access.
 *
 * <p>The constructors of the NInputStream, NOutputStream and RandomAccess
 * classes (or provider specific extensions of these classes) all take a
 * DataAccessable object as a parameter, and then implement themselves in
 * terms of this input parameter. The constructors that have a name parameter
 * (that must be atomic) will attempt to open a subordinate stream through
 * the given DataAccessable object. The constructors without a name parameter
 * attempt to open a stream on the given DataAcessable object itself.
 *
 * @see com.novell.java.io.NInputStream
 * @see com.novell.java.io.NOutputStream
 * @see com.novell.java.io.RandomAccess
 */

public interface DataAccessable
{

   /**
    * Reports if this accessor supports random access.
    * 
    * <p>If this DataAccessable object supports random access on itself
    * as the default action (no custom parameters), this method returns
    * TRUE; otherwise FALSE is returned.</p>
    *
    * @return  A boolean set to TRUE if random access is valid for this
    *          accessor; otherwise FALSE is returned.
    *
    * @see RandomAccessNotSupportedException
    * @see SubordinateAccessOnlyException
    */
    
   public boolean supportsRandomAccess();

   /**
    * Reports if this accessor supports input stream.
    * 
    * <p>If this DataAccessable object supports input stream on itself
    * as the default action (no custom parameters), this method returns
    * TRUE; otherwise FALSE is returned.</p>
    *
    * @return A boolean set to TRUE if input stream is valid for this
    *         accessor; otherwise FALSE is returned.
    *
    * @see InputStreamNotSupportedException
    * @see SubordinateAccessOnlyException
    */
    
   public boolean supportsInputStream();

   /**
    * Reports if this accessor supports output stream.
    * 
    * <p>If this DataAccessable object supports output stream on itself
    * as the default action (no custom parameters), this method returns
    * TRUE; otherwise FALSE is returned.</p>
    *
    * @return A boolean set to TRUE if input stream is valid for this
    *         accessor; otherwise FALSE is returned.
    *
    * @see OutputStreamNotSupportedException
    * @see SubordinateAccessOnlyException
    */
    
   public boolean supportsOutputStream();

   /**
    * Reports if this accessor supports subordinate random access.
    * 
    * <p>If this DataAccessable object supports subordinate random
    * access as the default action (no custom parameters), this method
    * returns TRUE; otherwise FALSE is returned.</p>
    *
    * @return A boolean set to TRUE if input stream is valid for this
    *         accessor; otherwise FALSE is returned.
    *
    * @see RandomAccessNotSupportedException
    * @see NoSubordinateAccessException
    */
    
   public boolean supportsSubordinateRandomAccess();

   /**
    * Reports if this accessor supports subordinate input streams.
    * 
    * <p>If this DataAccessable object supports subordinate input streams 
    * as the default action (no custom parameters), this method returns
    * TRUE; otherwise FALSE is returned.</p>
    *
    * @return A boolean set to TRUE if input stream is valid for this
    *         accessor; otherwise FALSE is returned.
    *
    * @see InputStreamNotSupportedException
    * @see NoSubordinateAccessException
    */
    
   public boolean supportsSubordinateInputStream();

   /**
    * Reports if this accessor supports subordinate output streams.
    * 
    * <p>If this DataAccessable object supports subordinate output
    * streams as the default action (no custom parameters), this method
    * returns TRUE; otherwise FALSE is returned.</p>
    *
    * @return A boolean set to TRUE if input stream is valid for this
    *         accessor; otherwise FALSE is returned.
    *
    * @see OutputStreamNotSupportedException
    * @see NoSubordinateAccessException
    */
    
   public boolean supportsSubordinateOutputStream();

/* *************************************************************************
* SPI openStream methods
***************************************************************************/

   /**
    * @internal
    *
    * SPI DEVELOPERS ONLY - Returns a DataAccessor given a type.
    *
    * <p>This method should not be used by the user application layer.
    *
    * <p>Returns a data accessor that allows platform independent I/O
    * access to a DataAccessable resource.
    *
    * <p>The NInputStream, NOutputStream and RandomAccess classes 
    * constructors call this method to open the data access.  Service 
    * providers that extend these classes to provide for the custom parameter
    * support, would also call this method in their appropriate constructors.
    *
    * <p>Implementers of this interface should select an appropriate
    * device accessor implementation class depending on the local client type, 
    * and initialize this device accessor.  Upon return from this method,
    * the device must be open and ready for I/O.
    * </p>
    *
    * @param type                The type of open to do. 
    *                            <ul>
    *                            <li>READ
    *                            <li>WRITE
    *                            <li>RANDOM
    *                            </ul>
    *
    * @exception IOException     Throws an IOException if the stream can not
    *                            be opened, and a DataAccessor returned.
    *
    * @exception InputStreamNotSupportedException     If this device does not 
    *                                                 support input streams.
    *
    * @exception OutputStreamNotSupportedException    If this device does not
    *                                                 support output streams.
    *
    * @exception RandomAccessNotSupportedException    If this device does not
    *                                                 support random access.
    *
    * @exception SubordinateAccessOnlyException       If this device only 
    *                                                 supports subordinate
    *                                                 access.
    *
    * @see com.novell.java.io.spi.DataAccessor#READ
    * @see com.novell.java.io.spi.DataAccessor#WRITE
    * @see com.novell.java.io.spi.DataAccessor#RANDOM
    */

   public DataAccessor openStream(int type)
      throws IOException;

   /**
    * @internal
    *
    * SPI DEVELOPERS ONLY - Returns a DataAccessor, given a type and a custom 
    * parameter.
    *
    * <p>This method should not be used by the user application layer.
    *
    * <p>Returns a data accessor that allows platform independent I/O
    * access to a DataAccessable resource.
    *
    * <p>The NInputStream, NOutputStream and RandomAccess classes 
    * constructors call this method to open the data access.  Service 
    * providers that extend these classes to provide for the custom parameter
    * support, would also call this method in their appropriate constructors.
    *
    * <p>Implementers of this interface should select an appropriate
    * device accessor implementation class depending on the local client type, 
    * and initialize this device accessor.  Upon return from this method,
    * the device must be open and ready for I/O.
    * </p>
    *
    * @param type                The type of open to do.
    *                            <ul>
    *                            <li>READ
    *                            <li>WRITE
    *                            <li>RANDOM
    *                            </ul>
    *
    * @param      custom         A provider based object that supplies custom
    *                            parameters which give the user application
    *                            control of implementation specific open
    *                            parameters (ie. openFlags, selector, size,
    *                            name space etc.)  If these types of 
    *                            parameters are required/available when 
    *                            opening the device, the implementor of this
    *                            interface should also supply the user 
    *                            application layer with this custom class.
    *                            For an example of this usage see;
    *                            com.novell.services.file.nw.NFileInputStream,
    *                            com.novell.services.file.nw.DataAccessableParameters.
    *
    * @exception IOException     Throws an IOException if the stream can not
    *                            be opened, and a DataAccessor returned.
    *
    * @exception InputStreamNotSupportedException     If this device does not 
    *                                                 support input streams.
    *
    * @exception OutputStreamNotSupportedException    If this device does not
    *                                                 support output streams.
    *
    * @exception RandomAccessNotSupportedException    If this device does not
    *                                                 support random access.
    *
    * @exception SubordinateAccessOnlyException       If this device only 
    *                                                 supports subordinate
    *                                                 access.
    *
    * @exception IllegalArgumentException    If the custom object is not of
    *                                        the expected type for the given
    *                                        provider.
    *
    * @see com.novell.java.io.spi.DataAccessor#READ
    * @see com.novell.java.io.spi.DataAccessor#WRITE
    * @see com.novell.java.io.spi.DataAccessor#RANDOM
    * @see com.novell.service.file.nw.NFileInputStream
    * @see com.novell.service.file.nw.NFileOutputStream
    * @see com.novell.service.file.nw.NRandomAccessFile
    * @see com.novell.service.file.nw.DataAccessableParameters
    */

   public DataAccessor openStream(int type, Object custom)
      throws IOException;

   /**
    * @internal
    *
    * SPI DEVELOPERS ONLY - Returns a subordinate DataAccessor given a type.
    *
    * <p>This method should not be used by the user application layer.
    *
    * <p>Returns a subordinate data accessor that allows platform independent 
    * I/O access to a DataAccessable resource.
    *
    * <p>The NInputStream, NOutputStream and RandomAccess classes 
    * constructors call this method to open the data access.  Service 
    * providers that extend these classes to provide for the custom parameter
    * support, would also call this method in their appropriate constructors.
    *
    * <p>Implementers of this interface should select an appropriate
    * device accessor implementation class depending on the local client type, 
    * and initialize this device accessor.  Upon return from this method,
    * the device must be open and ready for I/O.
    * </p>
    *
    * @param type                The type of open to do. 
    *                            <ul>
    *                            <li>READ
    *                            <li>WRITE
    *                            <li>RANDOM
    *                            </ul>
    *
    * @exception IOException     Throws an IOException if the stream can not
    *                            be opened, and a DataAccessor returned.
    *
    * @exception InputStreamNotSupportedException     If this device does not 
    *                                                 support input streams.
    *
    * @exception OutputStreamNotSupportedException    If this device does not
    *                                                 support output streams.
    *
    * @exception RandomAccessNotSupportedException    If this device does not
    *                                                 support random access.
    *
    * @exception NoSubordinateAccessException         If this device does not
    *                                                 support subordinate
    *                                                 access.
    *
    * @see com.novell.java.io.spi.DataAccessor#READ
    * @see com.novell.java.io.spi.DataAccessor#WRITE
    * @see com.novell.java.io.spi.DataAccessor#RANDOM
    */

   public DataAccessor openStream(String name, int type)
      throws IOException;

   /**
    * @internal
    *
    * SPI DEVELOPERS ONLY - Returns a subordinate DataAccessor, given a 
    * type and a custom parameter.
    *
    * <p>This method should not be used by the user application layer.
    *
    * <p>Returns a data accessor that allows platform independent I/O
    * access to a subordinate DataAccessable resource.
    *
    * <p>The NInputStream, NOutputStream and RandomAccess classes 
    * constructors call this method to open the data access.  Service 
    * providers that extend these classes to provide for the custom parameter
    * support, would also call this method in their appropriate constructors.
    *
    * <p>Implementers of this interface should select an appropriate
    * device accessor implementation class depending on the local client type, 
    * and initialize this device accessor.  Upon return from this method,
    * the device must be open and ready for I/O.
    * </p>
    *
    * @param type                The type of open to do. 
    *                            <ul>
    *                            <li>READ
    *                            <li>WRITE
    *                            <li>RANDOM
    *                            </ul>
    *
    * @param      custom         A provider based object that supplies custom
    *                            parameters which give the user application
    *                            control of implementation specific open
    *                            parameters (ie. openFlags, selector, size,
    *                            name space etc.)  If these types of 
    *                            parameters are required/available when 
    *                            opening the device, the implementor of this
    *                            interface should also supply the user 
    *                            application layer with this custom class.
    *                            For an example of this usage see;
    *                            com.novell.services.file.nw.NFileInputStream,
    *                            com.novell.services.file.nw.DataAccessableParameters.
    *
    * @exception IOException     Throws an IOException if the stream can not
    *                            be opened, and a DataAccessor returned.
    *
    * @exception InputStreamNotSupportedException     If this device does not 
    *                                                 support input streams.
    *
    * @exception OutputStreamNotSupportedException    If this device does not
    *                                                 support output streams.
    *
    * @exception RandomAccessNotSupportedException    If this device does not
    *                                                 support random access.
    *
    * @exception NoSubordinateAccessException         If this device does not
    *                                                 support subordinate
    *                                                 access.
    *
    * @exception IllegalArgumentException    If the custom object is not of
    *                                        the expected type for the given
    *                                        provider.
    *
    * @see com.novell.java.io.spi.DataAccessor#READ
    * @see com.novell.java.io.spi.DataAccessor#WRITE
    * @see com.novell.java.io.spi.DataAccessor#RANDOM
    * @see com.novell.service.file.nw.NFileInputStream
    * @see com.novell.service.file.nw.NFileOutputStream
    * @see com.novell.service.file.nw.NRandomAccessFile
    * @see com.novell.service.file.nw.DataAccessableParameters
    */

   public DataAccessor openStream(String name, int type, Object custom)
      throws IOException;
}
