/* **************************************************************************

  $Archive: /njcl_v2/src/com/novell/service/file/nw/NRandomAccessFile.java $
  $Revision: 4 $
  $Modtime: 12/17/99 10:47a $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.file.nw;

import java.io.IOException;
import java.io.EOFException;

import com.novell.java.io.RandomAccess;
import com.novell.java.io.DataAccessable;
import com.novell.java.io.SubordinateAccessOnlyException;
import com.novell.java.io.NoSubordinateAccessException;
import com.novell.java.io.RandomAccessNotSupportedException;

import com.novell.java.io.spi.DataAccessor;

/**
 * Opens a random access on a NetWare file.
 *
 * <p>NRandomAccessFile extends RandomAccess by adding convenience 
 * constructors for the various custom parameters.
 *
 * <p>The DataAccessableParameters class can also be used by the user 
 * application layer to obtain constant flags for the various constructors 
 * of this class, or it can instantiate a DataAccessableParameters object
 * and pass it into the RandomAccessor constructor.<p/>
 *
 * @see RandomAccess
 * @see DataAccessableParameters
 */

public class NRandomAccessFile extends RandomAccess
{
/* **************************************************************************
   non-subordinate stream
****************************************************************************/

   /**
    * Constructs a NRandomAccessFile object using default parameters.
    *
    * <p>Open a random access through the specified DataAccessable object,
    * using default parameters.<p/>
    *
    * @param accessor The DataAccessable object that will be 
    *                 called to open the random access.
    *
    * @exception IOException When an I/O error occurs.
    *
    * @exception SubordinateAccessOnlyException If the accessor does not
    *            support a stream to itself, for example a directory.
    *
    * @exception RandomAccessNotSupportedException if the accessor does not
    *            support random access.
    */

   public NRandomAccessFile(DataAccessable accessor)
      throws IOException
   {
      super(accessor);
   }

   /**
    * Constructs a NRandomAccessFile object using the openFlags
    * parameter. 
    *
    * <p>Given an openFlags object, opens a random access through the 
    * specified DataAccessable object.<p/>
    *
    * @param accessor  The DataAccessable object that will be 
    *                  called to open the random access.
    *
    * @param openFlags The flags that are used in opening the
    *                  access in different modes.
    *
    * @exception IOException When an I/O error occurs.
    *
    * @exception SubordinateAccessOnlyException If the accessor does not
    *            support a stream to itself, for example a directory.
    *
    * @exception RandomAccessNotSupportedException If the accessor does not
    *            support random access.
    */

   public NRandomAccessFile(DataAccessable accessor, int openFlags)
      throws IOException
   {
      this(
         accessor, 
         openFlags,
         DataAccessableParameters.DEFAULT_DATASELECTOR);
   }

   /**
    * Constructs a NRandomAccessFile object using the openFlags
    * and dataSelector parameters.
    *
    * <p>Given openFlags and dataSelector objects, opens a random access 
    * through the specified DataAccessable object.<p/>
    *
    * @param accessor     The DataAccessable object that will be called
    *                     to open the access.
    *
    * @param openFlags    The flags that are used in opening the
    *                     access in different modes.
    *
    * @param dataSelector An integral value that allows the
    *                     individual forks on a file to be
    *                     selected.
    *
    * @exception IOException When an I/O error occurs.
    *
    * @exception SubordinateAccessOnlyException If the accessor does not
    *            support a stream to itself, for example a directory.
    *
    * @exception RandomAccessNotSupportedException If the accessor does not
    *            support random access.
    */

   public NRandomAccessFile(
      DataAccessable accessor, 
      int openFlags, 
      int dataSelector)
      throws IOException
   {
      DataAccessableParameters custom = new 
         DataAccessableParameters(openFlags, dataSelector);
      
      setDataAccessor(accessor.openStream(DataAccessor.RANDOM, custom));
   }

/* **************************************************************************
   subordinate stream
****************************************************************************/

   /**
    * Constructs a subordinate NRandomAccessFile object using the
    * default parameters.
    *
    * <p>Open a subordinate random access through the specified 
    * DataAccessable object, using default parameters.<p/>
    *
    * @param name     The atomic subordinate name to open.
    *
    * @param accessor The DataAccessable object that will be 
    *                 called to open the random access.
    *
    * @exception  IOException When an I/O error occurs.
    *
    * @exception NoSubordinateAccessException If the accessor does not
    *            support opening a subordinate access, for example, a file.
    *
    * @exception RandomAccessNotSupportedException If the accessor does not
    *            support random access.
    */

   public NRandomAccessFile(String name, DataAccessable accessor)
      throws IOException
   {
      super(name, accessor);
   }

   /**
    * Constructs a subordinate NRandomAccessFile object using the
    * openFlags parameter. 
    *
    * <p>Given an openFlags object, opens a subordinate random access
    * through the specified DataAccessable object.<p/>
    *
    * @param name      The atomic subordinate name to open.
    *
    * @param accessor  The DataAccessable object that will be 
    *                  called to open the random access.
    *
    * @param openFlags The flags that are used in opening the
    *                  access in different modes.
    *
    * @exception IOException When an I/O error occurs.
    *
    * @exception NoSubordinateAccessException If the accessor does not
    *            support opening a subordinate access, for example, a file.
    *
    * @exception RandomAccessNotSupportedException If the accessor does not
    *            support random access.
    */

   public NRandomAccessFile(
      String name,
      DataAccessable accessor, 
      int openFlags)
      throws IOException
   {
      this(
         name,
         accessor, 
         openFlags,
         DataAccessableParameters.DEFAULT_DATASELECTOR);
   }

   /**
    * Constructs a subordinate NRandomAccessFile object using the
    * openFlags and dataSelector parameters.
    *
    * <p>Given openFlags and dataSelector objects, opens a subordinate
    * random access through the specified DataAccessable object.<p/>
    *
    * @param name         The atomic subordinate name to open.
    *
    * @param accessor     The DataAccessable object that will be 
    *                     called to open the access.
    *
    * @param openFlags    The flags that are used in opening the
    *                     access in different modes.
    *
    * @param dataSelector An integral value that allows the
    *                     individual forks on a file to be
    *                     selected.
    *
    * @exception IOException When an I/O error occurs.
    *
    * @exception NoSubordinateAccessException  If the accessor does not
    *            support opening a subordinate access, for example, a file.
    *
    * @exception RandomAccessNotSupportedException If the accessor does not
    *            support random access.
    */

   public NRandomAccessFile(
      String name,
      DataAccessable accessor, 
      int openFlags, 
      int dataSelector)
      throws IOException
   {
      DataAccessableParameters custom = new 
         DataAccessableParameters(openFlags, dataSelector);
      
      setDataAccessor(
         accessor.openStream(name, DataAccessor.RANDOM, custom));
   }
}
