/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/jncpv2r/net/NetBuffer.java $
  $Revision: 12 $
  $Modtime: 1/05/01 1:11p $

  Copyright (c) 1997 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/
package com.novell.service.jncpv2r.net;

import java.rmi.RemoteException;
import com.novell.service.jncp.*;
import java.io.Serializable;
import com.novell.java.lang.IntegerBuffer;
import com.novell.service.session.SessionException;
import com.novell.service.toolkit.jcl.NWInteger;

/**
 * Represents a request or reply buffer for NDS operations.
 */
public class NetBuffer implements Serializable
{
   /* values for NetBuffer size */
   public final static int  MAX_MESSAGE_LEN          = 63 * 1024;
   public final static int  DEFAULT_MESSAGE_LEN      = 16 * 1024;
   public final static int  MIN_MESSAGE_LEN          =  2 * 1024;

   /* values for NetBuffer operation */
   public final static int  DSV_READ                      =  3;
   public final static int  DSV_COMPARE                   =  4;
   public final static int  DSV_LIST                      =  5;
   public final static int  DSV_SEARCH                    =  6;
   public final static int  DSV_ADD_ENTRY                 =  7;
   public final static int  DSV_MODIFY_ENTRY              =  9;
   public final static int  DSV_READ_ATTR_DEF             = 12;
   public final static int  DSV_DEFINE_CLASS              = 14;
   public final static int  DSV_READ_CLASS_DEF            = 15;
   public final static int  DSV_MODIFY_CLASS_DEF          = 16;
   public final static int  DSV_LIST_CONTAINABLE_CLASSES  = 18;
   public final static int  DSV_ADD_PARTITION             = 20;


   private NetService service;
   private int handle;


   /**
    * Constructs an uninitialized NetBuffer
    */
   public NetBuffer ()
   {
      this.handle = 0;

   } /*  */

   /**
    * Constructs an NetBuffer using an NDSContext.
    *
    * <p>This constructor does not initialize the buffer.
    * </p>
    *
    * @param <i>contextHandle</i>      (in) NDSContext associated with the buffer.
    *
    * @return                    An uninitialized NetBuffer.
    *
    * @exception NSIException    Generic, source-unknown exception.
    */
   public NetBuffer (
         NetService service)
      throws NSIException
   {
      this (service, DEFAULT_MESSAGE_LEN);

   } /* NetBuffer () */


   /**
    * Constructs an NetBuffer using an NetContext and a size.
    *
    * <p>This constructor does not initialize the buffer.
    * </p>
    *
    * @param <i>contextHandle</i>      (in) NDSContext associated with the buffer.
    * @param <i>size</i>         (in) Number of bytes to allocate for the buffer.
    *
    * @return                    An uninitialized NetBuffer.
    *
    * @exception NSIException    Generic, source-unknown exception.
    */
   public NetBuffer (
         NetService service,
         int size)
      throws NSIException
   {
      try
      {
         this.service = service;
         handle = service.allocBuf (size);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* NetBuffer () */

   /**
    * Constructs an NetBuffer using an NDSContext.
    *
    * <p>This constructor initializes the buffer.
    * </p>
    *
    * @param <i>contextHandle</i>      (in) NDSContext associated with the buffer.
    * @param <i>operation</i>          (in) Specifies the Directory operation.
    * @param <i>size</i>               (in) Number of bytes to allocate for the buffer.
    *
    * @return                    An initialized NetBuffer.
    *
    * @exception NSIException    Generic, source-unknown exception.
    */
   public NetBuffer (
         NetService service,
         int operation,
         int size)
      throws NSIException
   {
      this (service, size);

      initBuf (operation);

   } /* NetBuffer () */


   /**
    * Initializes an request buffer for a request using an operation.
    *
    * <p>NOTE: Only request buffers need initialization
    * </p>
    *
    * @exception NSIException    Generic, source-unknown exception.
    */
   public final void initBuf (
         int operation)
      throws NSIException
   {
      try
      {
         service.initBuf (operation, handle);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* initBuf () */

   /**
    * Returns the number of attributes in the reply buffer.
    * </p>
    *
    * @return                    The number of attributes in the buffer.
    *
    * @exception NSIException    Generic, source-unknown exception.
    */
   public final int getAttrCount ()
      throws NSIException
   {
      try
      {
         return (service.getAttrCount (handle));
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getAttrCount () */

   /**
    * Returns the name of the attribute in the reply buffer.
    * </p>
    *
    * @exception NSIException    Generic, source-unknown exception.
    */
   public final void getAttrName (
         StringBuffer attrName,
         NWInteger attrValCount,
         NWInteger syntaxID)
      throws NSIException
   {
      try
      {
         service.getAttrName (handle, attrName, attrValCount, syntaxID);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getAttrName () */

   /**
    * @deprecated    use NetMemory getAttributeValue (syntaxId)
    */
   public final void computeAttrValSize (
         int syntaxID,
         NWInteger attrValSize)
      throws NSIException
   {
      try
      {
         service.computeAttrValSize (handle, syntaxID, attrValSize);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* computeAttrValSize () */

   /**
    *
    */
   public final int computeAttributeValueSize (
         int syntaxID)
      throws NSIException
   {
      try
      {
         NWInteger attrValSize = new NWInteger ();

         service.computeAttrValSize (handle, syntaxID, attrValSize);
         return (attrValSize.getValue ());
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* computeAttrValSize () */

   /**
    * @deprecated    use NetMemory getAttributeValue (syntaxId)
    */
   public final void getAttrVal (
         int syntaxID,
         NetMemory attrVal)
      throws NSIException
   {
      try
      {
         service.getAttrVal (handle, syntaxID, attrVal);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getAttrVal () */

   /**
    *
    */
   public final byte[] getAttributeValue (
         int syntaxID)
      throws NSIException
   {
      try
      {
         return (service.getAttributeValue (handle, syntaxID));
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getAttributeValue () */

   /**
    * Gets the class definition.
    * </p>
    *
    * @exception NSIException
    */
   public final void getClassDef (
         StringBuffer className,
         NetClassInfo classInfo)
      throws NSIException
   {
      try
      {
         service.getClassDef (handle, className, classInfo);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getClassDef () */

   /**
    * Returns the number of object-class definitions in the reply buffer.
    * </p>
    *
    * @return                    The number of class definitions in the buffer.
    *
    * @exception NSIException    Generic, source-unknown exception.
    */
   public final int getClassDefCount ()
      throws NSIException
   {
      try
      {
         IntegerBuffer count = new IntegerBuffer ();

         service.getClassDefCount (handle, count);
         return (count.intValue ());
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getClassDefCount () */

   /**
    * Returns an object-class item from the reply buffer.
    *
    * @return                    Object-class items in the
    *                            buffer.
    *
    * @exception NSIException    Generic, source-unknown exception.
    */
   public final String getClassItem ()
      throws NSIException
   {
      try
      {
         StringBuffer classItem = new StringBuffer();

         service.getClassItem (handle, classItem);
         return (classItem.toString ());
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getClassItem () */

   /**
    * Returns the number of object-class definition items in the reply buffer.
    * </p>
    *
    * @return                    Number of object-class definitions in the
    *                            buffer.
    *
    * @exception NSIException    Generic, source-unknown exception.
    */
   public final int getClassItemCount ()
      throws NSIException
   {
      try
      {
         IntegerBuffer classItemCount = new IntegerBuffer ();

         service.getClassItemCount (handle, classItemCount);
         return (classItemCount.intValue ());
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getClassItemCount () */

   /**
    * Returns the buffer handle used for native calls.
    *
    * @return                    The buffer handle used for native calls.
    */
   public final int getHandle ()
   {
      return (handle);
   }

   /**
    *
    */
   public final int getObjectCount ()
      throws NSIException
   {
      try
      {
         NWInteger objectCount = new NWInteger ();

         service.getObjectCount (handle, objectCount);
         return (objectCount.getValue());
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getObjectCount */

   /**
    * Returns the object name and info from the reply buffer.
    * @param      objectName     (out) The name of the current object
    * @param      attrCount      (out) The number of attributes for the object
    * @param      objectInfo     (out) Additional information for the object
    */
   public final void getObjectName (
         StringBuffer objectName,
         NWInteger attrCount,
         NetObjectInfo objectInfo)
      throws NSIException
   {
      try
      {
         service.getObjectName (handle, objectName, attrCount, objectInfo);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getObjectName () */

   /**
    *
    */
   public final void getAttrDef (
         StringBuffer attrName,
         NetAttributeInfo attrInfo)
      throws NSIException
   {
      try
      {
         service.getAttrDef (handle, attrName, attrInfo);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getAttrDef () */

   /**
    *
    */
   public final void getSyntaxDef (
         StringBuffer syntaxName,
         NetSyntaxInfo syntaxDef)
      throws NSIException
   {
      try
      {
         service.getSyntaxDef (handle, syntaxName, syntaxDef);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }
   } /* getSyntaxDef () */


   /**                      nds
    *
    */
   public final int getSyntaxCount ()
      throws NSIException
   {
      try
      {
         NWInteger syntaxCount = new NWInteger ();

         service.getSyntaxCount (handle, syntaxCount);
         return (syntaxCount.getValue ());
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getSyntaxCount () */


   public final void putChange (
         int            changeType,
         String         attrName)
      throws NSIException
   {
      try
      {
         service.putChange (handle, changeType, attrName);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* putChange () */

   public final void putClassItem (
         String   itemName)
      throws NSIException
   {
      try
      {
         service.putClassItem (handle, itemName);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* putClassItem () */

   /**@internal
    *
    */
   public final void putClassName (
         String attributeName)
      throws NSIException
   {
      try
      {
         service.putClassName (handle, attributeName);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* putAttributeName () */

   /**@internal
    *
    */
   public final void putAttributeName (
         String attributeName)
      throws NSIException
   {
      try
      {
         service.putAttrName (handle, attributeName);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* putAttributeName () */

   /**@internal
    *
    */
   public final void putSyntaxName (
         String syntaxName)
      throws NSIException
   {
      try
      {
         service.putSyntaxName (handle, syntaxName);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* putSyntaxName () */

   /**
    *
    */
   public final void putAttributeValue (
         int                              syntaxId,
         byte[]                           attributeValue)
      throws NSIException
   {
      try
      {
         service.putAttributeValue (handle, syntaxId, attributeValue);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* putAttributeValue () */

   /**
    *
    */
   public final void beginClassItem ()
      throws NSIException
   {
      try
      {
         service.beginClassItem (handle);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* beginClassItem () */


   // ******************** NWDSListPartitions () ********************

   /**
    *
    */
   public final String getServerName (
         IntegerBuffer partitionCount)
      throws NSIException
   {
      try
      {
         StringBuffer serverName = new StringBuffer ();

         service.getServerName (handle, serverName, partitionCount);
         return (new String (serverName));
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getServerName () */

   /**
    *
    */
   public final void getPartitionInfo (
         StringBuffer   partitionName,
         IntegerBuffer  replicaType)
      throws NSIException
   {
      try
      {
         service.getPartitionInfo (handle, partitionName, replicaType);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }

   } /* getPartitionInfo () */

   /**
    *
    */
   protected void finalize ()
   {
      try
      {
         service.freeBuf (handle);
      }
      catch (SessionException e)
      {
         NSIException ne = new NSIException ();

         ne.setRootCause (e);
         throw (ne);
      }
      catch(RemoteException e)
      {
         throw NSIExceptionBuilder.build(e);
      }
   }

} /* NetBuffer */
