
/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/nds/NdsInteger.java $
  $Revision: 12 $
  $Modtime: 11/15/00 3:43p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import java.io.Serializable;

import com.novell.service.schema.SchemaNumber;


/**
 * Provides access to values of the Integer attribute.
 * It is used for attributes whose values are signed integers.
 * The attributes for two integers match for equality if they
 * are the same. The comparison for ordering uses signed integer
 * rules. Attributes using this syntax can set size limits.
 * 
 * <p>Matching Rules: Equality and Ordering</p>
 *
 * @see NdsSyntaxId
 */
public class NdsInteger
   implements NdsAttributeValue, SchemaNumber, Cloneable, Serializable
{
   private static final int matchingRules = equality|ordering;

  /**
   * @internal
   */
   protected int value;

  /**
   * @internal
   */
   protected String name;

  /**
   * Constructs an NdsInteger object based on the specified
   * value parameter.
   *
   * @param value The long value of the signed integer to be stored.
   */
   public NdsInteger (
         long value)
   {
      this.value = (int) (value & INT_MASK);
   }

  /**
   * @internal
   */
   protected NdsInteger ()
   {
      this.value = -1;
   }

  /**
   * @internal
   */
   protected NdsInteger (
         String name,
         long value)
   {
      this.name = name;
      this.value = (int) (value & INT_MASK);
   }

  /**
   * @internal
   */
   protected NdsInteger (
         NdsInteger anInteger)
   {
      this.value = anInteger.value;
      this.name = anInteger.name;
   }


   // ******************** SchemaValue Interface ********************

  /**
   * Compares two objects for ordering, or compares two strings
   * lexicographically. For the Integer syntax, the compareTo
   * method compares this NDS Object with a reference NDS
   * Object contained in the anObject parameter. The
   * comparison is done according to the Ordering matching
   * rule for this syntax.
   * 
   * @param anObject The reference NDS Object with which
   *                 to compare.
   *
   * @return An int set to 0 if the reference Object is equal
   *         to this Object; an int <0 if the reference Object
   *         is less than this Object; an int >0 if the
   *         reference Object is greater than this Object.
   *
   * @exception IllegalArgumentException The passed-in object is of
   *            a different type than this NDS Object.
   */
    public int compareTo (
         Object anObject)
      throws Exception
   {
      if (anObject instanceof NdsInteger)
      {
         int value;

         value = ((NdsInteger) anObject).intValue ();
         return (this.value > value ? 1 : this.value < value ? -1 : 0);
      }
      throw (new IllegalArgumentException ());
   }

  /**
   * Compares two Objects for equality. The equals method compares
   * this object value with the value of the reference object in
   * the anObject parameter.
   *
   * <p>The equals method implements the most discriminating possible
   * equivalence relation on objects; that is, for any reference values
   * X and Y, this method returns TRUE if and only if X and Y refer to
   * the same object (X==Y has the value TRUE).</p>
   * 
   * @param anObject The reference object with which to compare.
   *
   * @return A boolean set to TRUE if and only if the argument is not
   *         NULL and is an NDS integer object that contains the
   *         same value as this object, otherwise set to FALSE.
   */
   public boolean equals (
         Object anObject)
   {
      if ((anObject != null) && (anObject instanceof NdsInteger))
      {
         return (value == ((NdsInteger) anObject).intValue ());
      }
      return (false);
   }

  /**
   * @internal
   */
   public String getName ()
   {
      return (name);
   }

  /**
   * @internal
   */
   public String getSyntaxId ()
   {
      return (NdsSyntaxId.INTEGER);
   }


   // ******************** SchemaNumber Interface ********************

   /**
    * Returns the value of this NDS object as an int.
    * 
    * @return The int value of this NDS object.
    */
   public int intValue ()
   {
      return (value);
   }

   /**
    * Returns the value of this NDS object as a long.
    * 
    * @return The long value of this NDS object.
    */
   public long longValue ()
   {
      return ((long)(value & INT_MASK));
   }

   /**
    * Returns the value of this NDS object as a float.
    * 
    * @return The float value of this NDS object.
    */
   public float floatValue ()
   {
      return ((float) value);
   }

   /**
    * Returns the value of this NDS object as a double.
    * 
    * @return The double value of this NDS object.
    */
   public double doubleValue ()
   {
      return ((double) value);
   }

   /**
    * Returns the value of this NDS object as a byte.
    * 
    * @return The byte value of this NDS object.
    */
   public byte byteValue ()
   {
      return ((byte) value);
   }

   /**
    * Returns the value of this NDS object as a short.
    * 
    * @return The short value of this NDS object.
    */
   public short shortValue ()
   {
      return ((short) value);
   }


   // ******************** NdsAttributeValue Interface ********************

  /**
   * Compares two Objects using the approximate matching rule.
   * 
   * The approximate method does not apply to the NdsInteger syntax
   * because this syntax does not support the approximate equals
   * matching rule.
   */
   public boolean approximate (
         Object anObject)
      throws Exception
   {
      throw (new Exception ());
   }

  /**
   * Returns the int that represents this NDS syntax ID.
   *
   * @return The syntax ID as an int.
   */
   public int getNdsSyntaxId ()
   {
      return (NdsSyntaxId.INTEGER_ID);
   }

  /**
   * Checks to see if this object supports the specified
   * matching rules. The NdsInteger syntax supports the
   * Equality and Ordering matching rules.
   *
   * @param matchingRules The set of matching rules to check.
   *
   * @return A Boolean set to TRUE if the matching rules for this
   *         object are equal to the matchingRules parameter.
   */
   public boolean supportsMatchingRules (
         int matchingRules)
   {
      if ((matchingRules & this.matchingRules) == matchingRules)
      {
         return (true);
      }
      return (false);
   }


   // ******************** Object Class ********************

  /**
   * Creates a new object of the same class as this object. It
   * then initializes each of the new object's fields by assigning
   * them the same value as the corresponding fields in this object.
   * No constructor is called.
   *
   * @return A clone of this object instance containing the cloned
   *         syntax.
   */
   public Object clone ()
   {
      try
      {
         NdsInteger integer = (NdsInteger) super.clone ();

         integer.value = this.value;
         integer.name = this.name;

         return (integer);
      }
      catch (CloneNotSupportedException e)
      {
         // this shouldn't happen, since we are Cloneable
         throw (new InternalError ());
      }

   } /* clone () */

  /**
   * Generates a string representation of the object. It
   * returns a string that textually represents the object.
   * The result should be a concise but informative
   * representation that is easy to read.
   *
   * @return The String representation of the object.
   */
   public String toString ()
   {
      if (null == name)
      {
         return (getSyntaxId () + " " + String.valueOf (value));
      }
      return (getSyntaxId () + " " + name + " = " + String.valueOf (value));
   }


   // ******************** NdsInteger Class ********************

  /**
   * Returns the integer as a long value (Java) representation.
   *
   * @return The integer as a long value.
   */
   public long getInteger ()
   {
      return (value & INT_MASK);
   }

} /* NdsInteger */

