
/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/nds/NdsIterator.java $
  $Revision: 21 $
  $Modtime: 1/05/01 3:59p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;

import javax.naming.NamingEnumeration;

import com.novell.service.jncp.NSIException;

import com.novell.service.jncpv2r.net.NetIterator;

/**
 * Creates a large virtual list of objects on the particular server in
 * question containing all the entries that pass the search filter.
 *
 * <p>An iterator in NDS is a concept for handling large virtual lists. 
 * It provides list and search functionality, and it can deal with very
 * large result sets, on the order of hundreds of thousands. An iterator
 * for contexts allows the programmer to traverse the NdsIterator results
 * in either direction. The variables defined here are used to set the
 * iterator position in the list to top, bottom or end of list.
 *
 * @see NdsIteratorControls
 * @see NdsIteratorFactory
 * @see NdsIteratorInfo
 * @see NdsIteratorResult
 */
public interface NdsIterator
{
 /**
  * The position at the top of the list.
  *
  * <p>(DS_ITR_FIRST = 0)</p>
  */
   public final static int DS_ITR_FIRST                     =    0;

 /**
  * The position at the bottom of the list.
  *
  * <p>(DS_ITR_LAST = 1000)</p>
  */
   public final static int DS_ITR_LAST                      = 1000;

 /**
  * The position at the end of the list, just after the last entry,
  * or no entries are present.
  *
  * <p>(DS_ITR_EOF = 1001)</p>
  */
   public final static int DS_ITR_EOF                       = 1001;


   /**
    * Returns the current iterator position in the list as an
    * int in the range 0-1000, or 1001 (DS_ITR_EOF).
    *
    * @param timeout The time in seconds allowed before returning.
    *
    * @return The current position in the list.
    *
    * @exception NSIException Standard NJCL exception for working on top
    *            of XPlat libraries.
    *
    * @see #setPosition(int, int)
    */
   int getPosition (
         int timeout)
      throws NSIException;

   /**
    * Sets the current iterator position in the list as an integer in
    * the range of 0 to 1001. The symbols DS_ITR_FIRST(0) and
    * DS_ITR_LAST(1000) set the position to the first or last entry,
    * respectively. The DS_ITR_EOF symbol sets the position to EOF
    * (End-Of-List), just after the last entry.
    * 
    * @param position The position to set as an integer from 0 to 1001.
    * @param timeout  The time in seconds allowed before returning.
    *
    * @exception NSIException Standard NJCL exception for working on top
    *            of XPlat libraries.
    */
   void setPosition (
         int position,
         int timeout)
      throws NSIException;

   /**
    * Sets the current iterator position to that of another iterator.
    * It is not necessary that the two iterators be identical. The system
    * tries to find the closest match in the source iterator and positions
    * the destination iterator accordingly.
    *
    * @param iterator The source iterator.
    * @param timeout  The time in seconds allowed before returning.
    *
    * @exception NSIException Standard NJCL exception for working on top
    *            of XPlat libraries.
    */
   void setPosition (
         NdsIterator iterator,
         int timeout)
      throws NSIException;

   /**
    * Sets the iterator position according to the passed-in attribute and 
    * value parameters. It implements the 'typedown' functionality.
    *
    * <p>For example, if you have a list sorted by surname, you can specify
    * D and it will position to the first name starting with D. If you
    * specify DA, it positions to the first name starting with DA, and
    * so forth. If there are no entries matching the value string, it
    * positions to the first one that is greater than the specified value,
    * or it positions to DS_ITR_EOF.
    *
    * @param attribute The attribute to use for typedown.
    * @param value     The string value to use for typedown.
    *
    * @exception NSIException Standard NJCL exception for working on top
    *            of XPlat libraries.
    */
   void setPosition (
         String attribute,
         String value,
         int timeout)
      throws NSIException;

   /**
    * Skips the number of entries in the list specified by count,
    * either forward (positive number) or backward (negative number).
    *
    * <p>The number of entries to skip is calculated from the current 
    * position. If the skip count extends beyond the boundary of the
    * list, the actual entries skipped is smaller than count. When
    * skip() returns because of timeout, the return value will not equal
    * the count requested.
    *
    * @param count   The number of entries to skip.
    * @param timeout The time in seconds allowed before returning.
    *
    * @return The number of entries actually skipped.
    *
    * @exception NSIException Standard NJCL exception for working on top
    *            of XPlat libraries.
    */
   int skip (
         int count,
         int timeout)
      throws NSIException;

    /**
     * Returns a boolean indicating whether this NdsIterator has more
     * elements when traversing the list in the forward direction. In
     * other words, if next() would return entries rather than returning
     * an empty enumeration, TRUE is returned; otherwise, FALSE is returned.
     *
     * @return A boolean set to TRUE if the NdsIterator has more elements
     *         when traversing forward in the list; otherwise, set to
     *         FALSE.
     *
     * @exception NSIException Standard NJCL exception for working on top
     *            of XPlat libraries.
     */
    boolean hasNext()
      throws NSIException;

    /**
     * Returns the next set (enumeration) of NdsIteratorResult objects
     * (entries) in the list. This method may be called repeatedly
     * to iterate through the list, or intermixed with calls to
     * previous to go back and forth. Note that alternating calls to
     * next and previous will return the same elements repeatedly.
     *
     * @param entries The number of next entries to return.
     * @param timeout The time in seconds allowed before returning.
     *
     * @return The NamingEnumeration of the next elements in the list.
     *
     * @exception NSIException Standard NJCL exception for working on top
     *            of XPlat libraries.
     * @exception NoSuchElementException If the iteration has no next elements.
     */
    NamingEnumeration next (
         int entries,
         int timeout)
      throws NSIException;

    /**
     * Returns a boolean indicating whether this NdsIterator has more
     * elements when traversing the list in the reverse direction. In
     * other words, if previous() would return an element rather than
     * returning an empty enumeration, TRUE is returned; otherwise, FALSE
     * is returned.
     *
     * @return A boolean set to TRUE if the NdsIterator has more elements
     *         when traversing backward in the list; otherwise, set to
     *         FALSE.
     *
     * @exception NSIException Standard NJCL exception for working on top
     *            of XPlat libraries.
     * @exception NoSuchElementException  The iteration has no next elements.
     */
    boolean hasPrevious ()
      throws NSIException;

    /**
     * Returns the previous set (enumeration) of NdsIteratorResult objects
     * (entries) in the list. The previous() method may be called repeatedly
     * to iterate through the list backwards, or it may be intermixed with
     * calls to next() to go back and forth. Note that alternating calls to
     * next() and previous() will return the same element(s) repeatedly.
     *
     * @param entries The number of previous entries to return.
     * @param timeout The time in seconds allowed before returning.
     *
     * @return A NamingEnumeration of the previous elements in the list.
     *
     * @exception NSIException Standard NJCL exception for working on top
     *            of XPlat libraries.
     * @exception NoSuchElementException If the iteration has no previous
     *            elements.
     */
    NamingEnumeration previous (
         int entries,
         int timeout)
      throws NSIException;

   /**
    * Returns the entry (NdsIteratorResult object) at the current position.
    *
    * @return The entry (NdsIteratorResult object) at the current position.
    *
    * @exception NSIException Standard NJCL exception for working on top
    *            of XPlat libraries.
    */
   Object current ()
      throws NSIException;

   /**
    * Counts the number of remaining entries (objects) in this iterator.
    * Entries are scanned starting from the current position. The list is
    * scanned until the timeout expires or the maxCount entries have been
    * counted or the end of the list is reached.
    *
    * <p>If both timeout and maxCount are zero, the entire list is scanned
    * and an exact count is returned. This may be very slow on large lists.
    * If scanning stops due to timeout or maxCount, the getPosition() 
    * method may be called to establish the real count. If more
    * accuracy is needed, count() may be called again with a longer timeout
    * or maxCount.
    *
    * @param timeout  The time in seconds allowed before returning.
    *                 Zero signifies no time limit.
    * @param maxCount The number of entries to scan before returning.  
    *                 Zero signifies no maximum limit.
    * @param updatePosition A boolean set to TRUE if the current position
    *                 will be left pointing to the last entry scanned.
    *                 If set to FALSE the current position is left
    *                 unchanged.
    *
    * @return The actual number of objects counted.
    *
    * @exception NSIException Standard NJCL exception for working on top
    *            of XPlat libraries.
    */
   int count (
         int timeout,
         int maxCount,
         boolean updatePosition)
      throws NSIException;

   /**
    * Returns encapsulated information about the iterator.
    *
    * @return The information about the iterator.
    */
   NdsIteratorInfo getInfo ()
      throws NSIException;

   /**
    * Creates a new object of the same class as this object. It then 
    * initializes each of the new object's fields by assigning the 
    * same value as the corresponding field in this object. No constructor 
    * is called.
    *
    * @return A clone of this instance.
    */
   public Object clone ();

   /**
    * Closes the iterator and frees all associated memory.
    *
    * @exception NSIException Standard NJCL exception for working on top
    *            of XPlat libraries.
    */
   public void close ()
      throws NSIException;

} /* NdsIterator */


