
/* **************************************************************************

  $Archive: /njcl_v2rmi/src/com/novell/service/nds/NdsIteratorFactory.java $
  $Revision: 18 $
  $Modtime: 11/15/00 3:43p $
 
  Copyright (c) 1998 Novell, Inc.  All Rights Reserved.

  THIS WORK IS  SUBJECT  TO  U.S.  AND  INTERNATIONAL  COPYRIGHT  LAWS  AND
  TREATIES.   NO  PART  OF  THIS  WORK MAY BE  USED,  PRACTICED,  PERFORMED
  COPIED, DISTRIBUTED, REVISED, MODIFIED, TRANSLATED,  ABRIDGED, CONDENSED,
  EXPANDED,  COLLECTED,  COMPILED,  LINKED,  RECAST, TRANSFORMED OR ADAPTED
  WITHOUT THE PRIOR WRITTEN CONSENT OF NOVELL, INC. ANY USE OR EXPLOITATION
  OF THIS WORK WITHOUT AUTHORIZATION COULD SUBJECT THE PERPETRATOR TO
  CRIMINAL AND CIVIL LIABILITY.

****************************************************************************/

package com.novell.service.nds;


import javax.naming.directory.Attributes;

import com.novell.service.jncp.NSIException;


/**
 * Generates NdsIterator objects used to create a large virtual list
 * based on a set of parameters. Five createIterator() methods are
 * defined, each passing in a different set of parameters used in
 * generating the NdsIterator objects.
 *
 * @see NdsIterator
 * @see NdsIteratorControls
 * @see NdsIteratorInfo
 * @see NdsIteratorResult
 */
public interface NdsIteratorFactory
{
   /** 
    * Creates an NdsIterator object based on three parameters. 
    * The parameters passed in are the name of the context to search
    * (name), the attributes for which to search (matchingAttributes),
    * and the attributes to return (attributesToReturn).
    *
    * <p>This NdsIterator() method searches in a single context for objects
    * that contain a specified set of attributes, and then returns the
    * entries or objects with the attributes specified by the 
    * attributesToReturn parameter in an NdsIterator object. The search
    * is performed using the default NdsIteratorControls settings.
    * 
    * <p>For an object to be selected, each attribute in the 
    * matchingAttributes parameter must match some attribute of the
    * object. If matchingAttributes is empty or NULL, all objects in
    * the target context are returned. The precise definition of
    * equality used in comparing attribute values is defined by the
    * underlying directory service. It might use the Java Object.equals
    * method, or it might use a schema to specify a different equality
    * operation.
    * 
    * @param name               The non-NULL name of the context to search.
    * @param matchingAttributes The attributes for which to search. If empty
    *                           or NULL, all objects in the target context
    *                           are returned.
    * @param attributesToReturn The attributes to return. NULL indicates
    *                           that all attributes should be returned. An
    *                           empty array indicates that no attributes are
    *                           to be returned.
    *
    * @return The objects with all their attributes in an NdsIterator.
    *
    * @exception NSIException If a naming exception occurs. NSIException
    *            is a standard NJCL exception for working on top
    *            of XPlat libraries.
    *
    * @see NdsIteratorControls
    * @see NdsIteratorResult
    */
   public NdsIterator createIterator (
         String name, 
         Attributes matchingAttributes,
         String[] attributesToReturn)
      throws NSIException;

   /** 
    * Creates an NdsIterator object based on two parameters. 
    * The parameters passed in are the name of the context to search
    * (name), and the attributes for which to search (matchingAttributes).
    *
    * <p>This NdsIterator() method searches in a single context for objects
    * that contain a specified set of attributes, and then returns the
    * objects with all their attributes in an NdsIterator. It is equivalent
    * to supplying NULL for the attributesToReturn parameter.
    *
    * @param name               The non-NULL name of the context to search.
    * @param matchingAttributes The attributes for which to search
    *                           (possibly NULL).
    *
    * @return The objects with all their attributes in an NdsIterator.
    *
    * @exception NSIException If a naming exception occurs. NSIException
    *            is a standard NJCL exception for working on top
    *            of XPlat libraries.
    *
    * @see NdsIteratorControls
    * @see NdsIteratorResult
    */
   public NdsIterator createIterator (
         String name,
         Attributes matchingAttributes)
      throws NSIException;

   /**
    * Creates an NdsIterator object based on three parameters. 
    * The parameters passed in are the name of the context to search
    * (name), the filter expression to use for the search (filter),
    * and the iterator controls that control the search (cons).
    *
    * <p>This NdsIterator() method searches in the named context or
    * named object for entries that satisfy the given search filter
    * and then returns the objects with all their attributes in an
    * NdsIterator. It performs the search as specified by the iterator
    * controls cons parameter, as defined in the NdsIteratorControls
    * constructor.
    * 
    * <p>After a call to NdsIterator.next or NdsIterator.previous, the
    * result returned is an NdsIterator. Each NdsIteratorResult contains
    * the name of the object and other information about the object (see
    * NdsIteratorResult). The name is either relative to the target context
    * of the search, which is named by the name parameter, or it is a URL
    * string. If the target context is included in the enumeration (as is
    * possible when the cons parameter specifies a search scope of
    * NdsIteratorControls.OBJECT_SCOPE), its name is the empty string.
    *
    * @param name   The non-NULL name of the context or object at which 
    *               to perform the search.
    * @param filter The non-NULL filter expression to use for the search.
    * @param cons   The iterator controls that control the search. 
    *               If NULL, it is equivalent to using the NULL
    *               NdsIteratorControls constructor.
    * 
    * @return The objects with all their attributes in an NdsIterator.
    *
    * @exception NSIException If a naming exception occurs. NSIException
    *            is a standard NJCL exception for working on top
    *            of XPlat libraries.
    * @exception InvalidSearchFilterException The search filter specified
    *            is not supported or understood by the underlying directory.
    * @exception InvalidSearchControlsException The iterator controls
    *            contains invalid settings.
    *
    * @see NdsIteratorControls
    * @see NdsIteratorResult
    */
   public NdsIterator createIterator (
         String name, 
         String filter,
         NdsIteratorControls cons)
      throws NSIException;

   /**
    * Creates an NdsIterator object based on four parameters. 
    * The parameters passed in are the name of the context to search
    * (name), the filter expression to use for the search (filterExpr),
    * an array of filter arguments (filterArgs), and the iterator
    * controls that control the search (cons).
    *
    * <p>This NdsIterator() method searches in the named context or
    * named object for entries that satisfy the given search filter
    * and then returns the objects with all their attributes in an
    * NdsIterator. It performs the search as specified by the iterator
    * controls cons parameter, which is defined in the NdsIteratorControls
    * constructor.
    *
    * <p>After a call to NdsIterator.next or NdsIterator.previous, this
    * method returns an NdsIterator, which contains the name of the object
    * and other information about the object (see NdsIteratorResult). The
    * name is either relative to the target context of the search, which
    * is named by the name parameter, or it is a URL string. If the target
    * context is included in the enumeration (as is possible when the cons
    * parameter specifies a search scope of NdsIteratorControls.OBJECT_SCOPE),
    * its name is an empty string.
    *
    * @param name       The non-NULL name of the context or object at which
    *                   to perform the search.
    * @param filterExpr The non-NULL filter expression to use for the search.
    *                   The expression may contain variables of the
    *                   form {i} where i is an integer.
    * @param filterArgs The array of arguments (possibly NULL) to substitute 
    *                   for the variables in filterExpr. The value of
    *                   filterArgs[i]  will replace each occurrence of
    *                   {i}.
    * @param cons       The iterator controls that control the search. 
    *                   If NULL, it is equivalent to using the no arguments
    *                   NdsIteratorControls constructor.
    * 
    * @return The objects with all their attributes in an NdsIterator.
    *
    * @exception NSIException If a naming exception occurs. NSIException
    *            is a standard NJCL exception for working on top
    *            of XPlat libraries.
    * @exception ArrayIndexOutOfBoundsException If filterExpr contains
    *            {i} expressions where i is outside the bounds of the
    *            array filterArgs.
    * @exception InvalidSearchControlsException If the cons parameter
    *            contains invalid settings.
    * @exception InvalidSearchFilterException If filterExpr and filterArgs 
    *            expresses an invalid search filter.
    *
    * @see NdsIteratorControls
    */
   public NdsIterator createIterator (
         String name,
         String filterExpr,
         Object[] filterArgs,
         NdsIteratorControls cons)
      throws NSIException;

   /**
    * Creates an NdsIterator object based on four parameters. 
    * The parameters passed in are the name of the context to search
    * (name), the iterator controls that control the search (cons),
    * the name of the object class (className) and the name of the
    * subordinate in which to search.
    *
    * <p>This NdsIterator() method lists subordinates in the named context
    * or named object for entries that satisfy the given filter of class
    * and subordinate names, then returns the objects without attribute
    * names or values in an NdsIterator.
    * 
    * <p>Of the items stored in the NdsIteratorControls, the items that
    * apply to the list iterator with their default values are:
    *
    * <ol><li>scalability = NdsIteratorControls.DS_ITR_PREFER_SCALABLE
    * <li>timeout = 0
    * <li>retObj = false</ol>
    *
    * <p>The name given for the className filter is the name of an
    * object class, such as User, Computer, or Server. The value given
    * for subordinateName filter can be one of the following:
    *
    * <ul><li>The left-most name of an object, such as Adam or Graphics
    * Printer. 
    * <li>A string with asterisks (*), such as A* or Gr*. 
    * <li>NULL, which means any name is valid.</ul>
    *
    * <p>The location of the subordinate object(s) in the NDS tree is
    * immediately subordinate to the object specified by objectName.
    * The relationship between className and subordinateName is an AND
    * relationship. When className and subordinateName are provided, a
    * list of immediate subordinate objects restricted by both filters
    * is returned. When className is NULL and subordinateName is NULL,
    * a list of all immediate subordinates is returned. When className
    * is provided and subordinateName is NULL, a list of immediate
    * subordinates restricted only by the className filter is returned.
    * When className is NULL and subordinateName is provided, a list of
    * immediate subordinates restricted only by the subordinateName
    * filter is returned.
    *
    * <p>The following examples show how to use wildcards for untyped names:
    *
    * <ul><li>c* - Any object whose left-most name begins with a "c"
    * character. 
    * <li>M*y - Any object beginning with "M" and ending with "y" such as
    * Mary.</ul>
    *
    * <p>If the wildcard name specified for subordinateName includes a type,
    * such as CN, the name must include the equals (=) sign. The following
    * examples show how to use wildcards for typed names:
    *
    * <ul><li>cn=* - Any object whose left-most name is a common name. 
    * <li>cn=c* - Any object whose left-most name is a common name and
    * begin with "c." 
    * <li>o*=* - Any object whose left-most name has a naming attribute
    * beginning with an "o," such as O or OU. 
    * <li>o*=c* - Any object whose left-most name has a naming attribute
    * beginning with an "o," and whose name begins with "c."</ul></p>
    *
    * @param name            The non-NULL name of the context or object at
    *                        which to perform the search.
    * @param cons            The iterator controls that control the search. 
    *                        If NULL, it is equivalent to using the no
    *                        arguments NdsIteratorControls constructor.
    * @param className       The name of the object class filter.
    * @param subordinateName The name of the subordinate filter.
    *
    * @return The subordinate objects in an NdsIterator.
    *
    * @exception NSIException If a naming exception occurs. NSIException
    *            is a standard NJCL exception for working on top
    *            of XPlat libraries.
    * @exception InvalidSearchControlsException If the cons parameter
    *            contains invalid settings.
    *
    * see NdsIteratorControls
    */
   public NdsIterator createIterator (
         String name,
         NdsIteratorControls cons,
         String className,
         String subordinateName)
      throws NSIException;

} /* NdsIteratorFactory */


